package heuser.updater;

import java.io.*;
import java.net.URL;
import java.net.URLConnection;

/**
 * Enables applications to Update themselves through the internet.<BR>
 * Only three methods have to be called by the application: {@link Update#newVersionAvailable(java.lang.String, java.lang.String, java.lang.String)},
 * {@link Update#installNew()} and {@link Update#getErrors()} if {@link Update#error} is true.<BR><BR>
 * <b>Format of .cvr files:</b><BR>
 * version, eg. "1.0.2"<BR>
 * location, eg. "download/apps/"
 * filename, eg. "killerApp_1.0.2.jar"
 * @author Jens Heuser, heuserjens@users.sourceforge.net
 */
public class Update {

    private static String appName;
    private static String old_ver;
    private static String loc;

    private static boolean upToDate;

    /**
     * Set to <code>true</code> if errors occured.
     */
    public static boolean error = false;

    private static String errorMsg = "";

    private String onlineVersion;
    private static String onlineLoc;
    private static String onlineName;


    /**
     * Checks if a new version is available.
     * @param appName The name of the application to be updated.
     * @param old_ver The version of the application currently running.
     * @param loc The location of the .cvr file.
     * @return <code>true</code> if a new version is available.
     */
    public static boolean newVersionAvailable(String appName, String old_ver, String loc){
        new Update(appName, old_ver, loc);
        return upToDate;
    }


    /**
     * Triggers the installation of a new version.
     */
    public static void installNew(){
        new Update();
    }


    /**
     * Constructor for updating.<BR>{@link Update#newVersionAvailable(java.lang.String, java.lang.String, java.lang.String)} has to be called first.
     */
    public Update(){
        /*if (!checkOS().equals("Windows")) {
            logError("Currently only Windows is supported.\nInstall manually from " + loc + onlineLoc);
            return;
        }*/
        if (appName.length() == 0) return;
        downloadNewVersion();
        install(prepareFiles(checkOS()));
    }


    /**
     * Constructor for checking for a new version.
     * @param appName The name of the application to be updated.
     * @param old_ver The version of the application currently running.
     * @param loc The location of the .cvr file.
     */
    public Update (String appName, String old_ver, String loc){
        this.appName = appName;
        this.old_ver = old_ver;
        this.loc = loc;

        deleteUpdateScript();
        upToDate = newerAvailable();
    }


    /**
     * Returns the errors that (might) have occured.
     * @return Errormessage(s).
     */
    public static String getErrors(){
        return errorMsg;
    }


    private boolean newerAvailable(){
        retrieveAppInfo();
        if (errorMsg.length() != 0) return false;

        if (firstNewer(onlineVersion, old_ver)) return true;
        return false;
    }


    private void retrieveAppInfo(){
        BufferedReader info;
        URL url;

        try{
            url = new URL(loc + appName + ".cvr");
            info = new BufferedReader(new InputStreamReader(url.openStream()));
            onlineVersion = info.readLine();
            onlineLoc = info.readLine();
            onlineName = info.readLine();
            info.close();
        } catch (Exception e){
            logError("Error connecting server and / or reading information file.\n");
        }
    }


    private boolean firstNewer(String ver1, String ver2){
        int major1, major2;
        String tmp1, tmp2;

        
        major1 = Integer.parseInt(ver1.substring(0, ver1.indexOf(".")));
        major2 = Integer.parseInt(ver2.substring(0, ver2.indexOf(".")));

        if (major1 > major2) return true;
        if (major1 < major2) return false;


        tmp1 = ver1.substring(ver1.indexOf(".") + 1);
        tmp2 = ver2.substring(ver2.indexOf(".") + 1);
        if (!tmp1.endsWith(".")) tmp1 += ".";
        if (!tmp2.endsWith(".")) tmp2 += ".";

        if ((tmp2.length() == 1) && (tmp1.length() > 1)) return true;
        if ((tmp1.length() == 1) && (tmp2.length() > 1)) return false;

        if ((tmp2.length() == 1) && (tmp1.length() == 1)) return false; //besser machen...

        return firstNewer(tmp1, tmp2);
    }


    private void downloadNewVersion(){
        try{
            URL url = new URL(loc + onlineLoc + onlineName);
            URLConnection connection = url.openConnection();
            InputStream stream = connection.getInputStream();
            BufferedInputStream in = new BufferedInputStream(stream);

            FileOutputStream file = new FileOutputStream(onlineName);
            BufferedOutputStream out = new BufferedOutputStream(file);
            int i;
            while ((i = in.read()) != -1) {
                out.write(i);
            }
            out.close();
        }catch(Exception e){
            e.printStackTrace();
            logError("Error downloading new version.");
        }
    }


    private String prepareFiles(String os){
        String commands[] =  {  "@del " + appName + "_" + old_ver + ".jar",
                                "@start javaw -jar " + onlineName};

        if (os.equals("Linux")) {
            commands[0] = "#bin/bash" + System.getProperty("line.separator") +  "rm " + appName + "_" + old_ver + ".jar";
            commands[1] = "java -jar " + onlineName;
        }

        File batch;
        if ((os.equals("Windows") || (os.equals("DOS")))) batch = new File("update_" + appName + ".bat");
        else batch = new File("update_" + appName + ".sh");
        
        if (batch.exists()) batch.delete();

        try{
            BufferedWriter out = new BufferedWriter(new FileWriter(batch));
            for (int i = 0; i < commands.length; i++){
                out.write(commands[i]);
                out.newLine();
            }
            out.close();
        }catch(Exception e){
            logError("Error creating batch file. Check space and / or permissions.\n");
        }

        return batch.getAbsolutePath();
    }


    private void deleteUpdateScript(){
        File updateScript;

        updateScript = new File("update_" + appName + ".bat");

        if (updateScript.exists())
            updateScript.delete();

        updateScript = new File("update_" + appName + ".sh");

        if (updateScript.exists())
            updateScript.delete();
    }


    private void install(String path){
        if (error) return;
        try{
            Runtime.getRuntime().exec(path);
            Runtime.getRuntime().exit(0);
        }catch(Exception e){e.printStackTrace();}
    }


    private String checkOS(){
        String os = System.getProperty("os.name");

        if (os.indexOf("Windows") != -1) return "Windows";
        if (os.indexOf("DOS") != -1) return "DOS";
        if (os.indexOf("Linux") != -1) return "Linux";
        return "Other";
    }


    private void logError(String anError){
        errorMsg += anError;
        error = true;
    }
}
