/* ExportOptionsDialog - jfig.gui.ExportOptionsDialog.java
 *
 * specify the options for jfig fig2dev/GIF export and handle export
 *
 * 21.01.02 - epic, eepic support by Jose Tomas Palma Mendez, jpalma@dif.um.es
 * 28.03.01 - use SetupManager to init magnification, PS center, orientation
 * 19.11.00 - some more redesign code
 * 17.11.00 - complete redesign started
 * 12.10.00 - use double values in get/setMagnification
 * 11.12.97 - first version
 *
 * (C) 1997-2000 F.N.Hendrich, hendrich@informatik.uni-hamburg.de
 */

package jfig.gui;

import jfig.canvas.FigCanvas;
import jfig.objects.FigAttribs;
import jfig.objects.FigBbox;
import jfig.utils.*;


import java.awt.*;
import java.awt.event.*;
import java.io.*;
import java.lang.reflect.*;
import java.util.Enumeration;


/**
 * dialog window to specify fig2dev export options and handle export.
 * <p>
 */
public class      ExportOptionsDialog
       extends    Frame  {


   public final int FULL_COLORS    = 2;
   public final int IO_COLORS      = 3;
   public final int BLACK_WHITE    = 4;

   public final String FULL_COLORS_STRING = "full color";
   public final String IO_COLORS_STRING   = "black and white, I/O colored";
   public final String BLACK_WHITE_STRING = "black and white only";

   public final int A4_WIDTH       = 26173;  // at 2400 dpi, with 1 cm border
   public final int A4_HEIGHT      = 17952;


   TextField   magnificationTF, xoffsetTF, yoffsetTF;
   Choice      languageChoice;
   Choice      orientationChoice;
   Choice      centerChoice;
   Choice      colorChoice;
   Checkbox    autosaveCB;

   Button      exportNowButton, cancelButton;
   Button      filenameButton, fitToPageButton;
   TextField   filenameTF, directoryTF, fig2devPathTF, commandLineTF;
   Label       sizeLabel;
   FileDialog  fileDialog;



   boolean     debug = true;
   boolean     autosave = true;
   boolean     waitExec = true; // wait for subprocess?

   String[][] data = {
// language,                  extension,    methodname,     internal
{  "Postscript",              "ps",         "doExportPS",   "no" },
{  "Encapsulated Postscript", "eps",        "doExportEPS",  "no" },
{  "IBM GL",                  "gl",         "doExportIBMGL","no" },
{  "Latex",                   "latex",      "doExportLATEX","no" },
{  "Pictex",                  "pictex",     "doExportPICTEX","no" },
{  "PSTEX",                   "pstex",      "doExportPSTEX","no" },
{  "PPM",                     "ppm",        "doExportPPM",  "no" },
{  "JPEG",                    "jpg",        "doExportJPG",  "no" },
{  "GIF",                     "gif",        "doExportGIF",  "no" },
{  "PNG",                     "png",        "doExportPNG",  "no" },
{  "EPIC",                    "epic",       "doExportEPIC", "no" },
{  "EEPIC",                   "eepic",      "doExportEEPIC","no" },
};



   String  printCommand,
           fig2devPath,
           fig2devPS,
           fig2devEPS,
           fig2devIBMGL,
           fig2devLATEX,
           fig2devPSTEX,
           fig2devPICTEX,
           fig2devJPG,
           fig2devGIF,
           fig2devTIFF,
           fig2devPNG,
           fig2devPPM,
           fig2devEPIC,
           fig2devEEPIC;
   boolean fig2devPSCentered = true;
   boolean fig2devPSOrientation = true; // means Portrait
   double  fig2devMagnification = 1.0;



   ModularEditor    editor = null;
   FigBbox  bbox = null;   // bounding box of all FIG objects, for fit



   /**
    * construct a new ExportOptionsDialog,
    * ask SetupManager for the fig2dev parameters
    */
   public ExportOptionsDialog() {
      super( "Select Export Options" );
      getAndSetDefaultFont();
      loadExportSetup();
      buildGUI();
      buildCallbacks();
      updateCommandLine();
   }


   public void setModularEditor( ModularEditor editor ) {
     this.editor = editor;
     updateCommandLine();
   }



   public void buildGUI() {
     this.setBackground(
        SetupManager.getColor(
          "jfig.gui.EditDialog.BackgroundColor", Color.lightGray ));

     // the "bottom" panel with the "OK" and "Export now" buttons
     //
     //jfig.gui.InsetsPanel okPanel = new jfig.gui.InsetsPanel(10,10);
     //okPanel.setLayout( new FlowLayout( FlowLayout.CENTER ) );
     //okPanel.setLowered();
     cancelButton    = new Button( "Cancel" );
     exportNowButton = new Button( "Export now" );

     //  okPanel.add( cancelButton );
     //  okPanel.add( new Label( "   " ) );
     //  okPanel.add( exportNowButton );

     // the "label" panel on the left of the window
     //
     Panel labelPanel = new Panel( new GridLayout( 0, 1 ));
     labelPanel.add( new Label( "output format:", Label.RIGHT));
     //labelPanel.add( new Label( "color option:", Label.RIGHT));
     //labelPanel.add( new Label( "" ));
     labelPanel.add( new Label( "magnification:", Label.RIGHT ));
     labelPanel.add( new Label( "orientation:", Label.RIGHT ));
     labelPanel.add( new Label( "offsets x,y:", Label.RIGHT ));
     labelPanel.add( new Label( "" ));
     labelPanel.add( new Label( "output filename:", Label.RIGHT ));
     labelPanel.add( new Label( "directory:", Label.RIGHT ));
     labelPanel.add( new Label( "" ));
     //labelPanel.add( new Label( "fig2dev path:", Label.RIGHT ));
     labelPanel.add( new Label( "command line:", Label.RIGHT ));

       Panel cancelPanel = new Panel( new FlowLayout( FlowLayout.RIGHT ));
       cancelPanel.add( cancelButton );
     labelPanel.add( cancelPanel );

     // the "options" panel on the right of the window
     //
     Panel optionsPanel = new Panel( new GridLayout( 0, 1 ));

       Panel languagePanel = new Panel();
       languagePanel.setLayout( new FlowLayout( FlowLayout.LEFT ));
       languageChoice = new Choice();
       for( int i=0; i < data.length; i++ ) {
         languageChoice.addItem( data[i][0] );
       }
       languageChoice.select( 0 );
       languagePanel.add( languageChoice );
     optionsPanel.add( languagePanel );

       Panel colorPanel = new Panel( new FlowLayout( FlowLayout.LEFT ));
       colorChoice = new Choice();
       colorChoice.addItem( FULL_COLORS_STRING );
       //colorChoice.addItem( IO_COLORS_STRING ); // only useful in HADES
       colorChoice.addItem( BLACK_WHITE_STRING );
       colorChoice.select( FULL_COLORS_STRING );
       colorPanel.add( colorChoice );
     //optionsPanel.add( colorPanel );
     //optionsPanel.add( new Label( "" ));

       Panel magnificationPanel = new Panel();
       magnificationPanel.setLayout( new FlowLayout( FlowLayout.LEFT ) );
       magnificationTF =  new TextField( "100", 6 );
       setMagnification( fig2devMagnification );
       sizeLabel = new Label( "  [10.0 x 10.0 inches]  " );
       fitToPageButton = new Button( "Fit to A4" );
       magnificationPanel.add( magnificationTF );
       magnificationPanel.add( new Label( "%" ));
       magnificationPanel.add( sizeLabel );
       magnificationPanel.add( fitToPageButton );
     optionsPanel.add( magnificationPanel );

       Panel orientationPanel = new Panel();
       orientationPanel.setLayout( new FlowLayout( FlowLayout.LEFT ) );
       orientationChoice = new Choice();
       orientationChoice.addItem( "landscape" );
       orientationChoice.addItem( "portrait" );
       setFig2devPSOrientation( fig2devPSOrientation );
       centerChoice = new Choice();
       centerChoice.addItem( "center" );
       centerChoice.addItem( "flush left" );
       setFig2devPSCentered( fig2devPSCentered );
       orientationPanel.add( orientationChoice );
       orientationPanel.add( centerChoice );
     optionsPanel.add( orientationPanel );

       Panel offsetPanel = new Panel( new FlowLayout( FlowLayout.LEFT ));
       xoffsetTF = new TextField( "0", 4 );
       yoffsetTF = new TextField( "0", 4 );
       offsetPanel.add( xoffsetTF );
       offsetPanel.add( yoffsetTF );
       offsetPanel.add( new Label( "inches" ));
     optionsPanel.add( offsetPanel );
     optionsPanel.add( new Label( "" ));

       Panel filenamePanel = new Panel( new FlowLayout( FlowLayout.LEFT ));
       filenameTF     = new TextField( "", 40 );
       //filenameTF.setEditable( false );
       filenameButton = new Button( "Browse" );
       filenamePanel.add( filenameTF );
       filenamePanel.add( filenameButton );
     optionsPanel.add( filenamePanel );

       Panel directoryPanel = new Panel( new FlowLayout( FlowLayout.LEFT ));
       directoryTF     = new TextField( "", 40 );
       //directoryTF.setEditable( false );
       directoryPanel.add( directoryTF );
     optionsPanel.add( directoryPanel );

       Panel autosavePanel = new Panel( new FlowLayout( FlowLayout.LEFT ));
       autosaveCB = new Checkbox( "save FIG file before export" );
       //autosavePanel.add( autosaveCB );
     optionsPanel.add( autosavePanel );

     //  Panel fig2devPanel = new Panel( new FlowLayout( FlowLayout.LEFT ));
     //  fig2devPathTF = new TextField( fig2devPath, 40 );
     //  fig2devPanel.add( fig2devPathTF );
     //optionsPanel.add( fig2devPanel );

       Panel commandLinePanel = new Panel( new FlowLayout( FlowLayout.LEFT ));
       commandLineTF = new TextField( "", 60 );
       commandLinePanel.add( commandLineTF );
     optionsPanel.add( commandLinePanel );

       Panel exportNowPanel = new Panel( new FlowLayout( FlowLayout.LEFT ));
       exportNowPanel.add( exportNowButton );
       exportNowPanel.add( new Label( "using above command" ));
     optionsPanel.add( exportNowPanel );


     this.setLayout( new BorderLayout() );
     this.add( "West",  labelPanel );
     this.add( "East", optionsPanel );
     //this.add( "South", okPanel );
     this.pack();
   } // buildGUI



  public void buildCallbacks() {
     OptionChangeHandler optionChangeHandler = new OptionChangeHandler();
     TextFieldMultiplexer        multiplexer = new TextFieldMultiplexer();

     filenameTF     .addKeyListener( multiplexer );
     directoryTF    .addKeyListener( multiplexer );
     xoffsetTF      .addKeyListener( multiplexer );
     yoffsetTF      .addKeyListener( multiplexer );
     magnificationTF.addKeyListener( multiplexer );

     //filenameTF.addActionListener( optionChangeHandler );
     //directoryTF.addActionListener( optionChangeHandler );
     //magnificationTF.addActionListener( optionChangeHandler );
     //xoffsetTF.addActionListener( optionChangeHandler );
     //yoffsetTF.addActionListener( optionChangeHandler );

     languageChoice.addItemListener( optionChangeHandler );
     orientationChoice.addItemListener( optionChangeHandler );
     centerChoice.addItemListener( optionChangeHandler );
     colorChoice.addItemListener( optionChangeHandler );

     fitToPageButton.addActionListener( new ActionListener() {
       public void actionPerformed( ActionEvent e ) {
         dbg( "-#- fitToPageButton: " + e );
         fitToLandscapeA4();
         updateCommandLine();
       }
     });

     cancelButton.addActionListener( new ActionListener() {
       public void actionPerformed( ActionEvent e ) {
         dbg( "-#- cancelButton: " + e );
         ExportOptionsDialog.this.setVisible( false );
       }
     });

     exportNowButton.addActionListener( new ActionListener() {
       public void actionPerformed( ActionEvent e ) {
         dbg( "-#- exportNowButton: " + e );
         doExportNow();
       }
     });

     filenameButton.addActionListener( new ActionListener() {
       public void actionPerformed( ActionEvent e ) {
         dbg( "-#- filenameButton: " + e );
         doBrowseFilename();
       }
     });

     this.addWindowListener( new WindowAdapter() {
       public void windowClosing( WindowEvent e ) {
         dbg( "-#- ExportOptionsDialog: windowClosing..." );
         ExportOptionsDialog.this.setVisible( false );
       }
     });
  } // buildCallbacks




  /**
   * inner class to react to item state changes:
   * we just update the filename extension and command line
   */
  class OptionChangeHandler implements ItemListener, ActionListener {
    public void itemStateChanged( ItemEvent e ) {
      dbg( "-#- optionChangeHandler: " + e );
      updateOutFilenameExtension();
      updateCommandLine();
      showSize();
    }

    public void actionPerformed( ActionEvent e ) {
      dbg( "-#- optionChangeHandler: " + e );
      updateCommandLine();
      showSize();
    }
  } // OptionChangeHandler




  /**
   * inner class to propagate TextField inputs to the commandline TextField
   */
  class TextFieldMultiplexer implements KeyListener {
    public void keyPressed ( KeyEvent e ) { dbg( "-*- " + e ); mux(); }
    public void keyReleased( KeyEvent e ) { dbg( "-*- " + e ); mux(); }
    public void keyTyped   ( KeyEvent e ) { dbg( "-*- " + e ); mux(); }

    public void mux() {
      updateCommandLine();
      showSize();
    }
  } // TextFieldMultiplexer



  /**
   * load the following export and fig2dev related setup parameters:
     <pre>
       jfig.gui.ExportOptionsDialog.Debug
       jfig.gui.ExportOptionsDialog.Autosave
       jfig.gui.ExportOptionsDialog.Print
       jfig.gui.ExportOptionsDialog.fig2devPath
       jfig.gui.ExportOptionsDialog.fig2devPS
       jfig.gui.ExportOptionsDialog.fig2devEPS
       jfig.gui.ExportOptionsDialog.fig2devIBMGL
       jfig.gui.ExportOptionsDialog.fig2devLATEX
       jfig.gui.ExportOptionsDialog.fig2devPSTEX
       jfig.gui.ExportOptionsDialog.fig2devPICTEX
       jfig.gui.ExportOptionsDialog.fig2devPPM
       jfig.gui.ExportOptionsDialog.fig2devJPG
       jfig.gui.ExportOptionsDialog.fig2devGIF
       jfig.gui.ExportOptionsDialog.fig2devPNG
       jfig.gui.ExportOptionsDialog.fig2devWaitFor
     </pre>
   */
  public void loadExportSetup() {
    String base = "jfig.gui.ExportOptionsDialog.";
    debug         = SetupManager.getBoolean(  base+"Debug", true );
    autosave      = SetupManager.getBoolean(  base+"Autosave", true );
    waitExec      = SetupManager.getBoolean(  base+"Autosave", true );
    printCommand  = SetupManager.getProperty( base+"Print",  "lpr" );

    fig2devPath   = SetupManager.getProperty( base+"fig2devPath",  "c:\\utils\\jfig\\fig2dev" );
    fig2devPS     = SetupManager.getProperty( base+"fig2devPS",    "-L ps -P" );
    fig2devEPS    = SetupManager.getProperty( base+"fig2devEPS",   "-L ps" );
    fig2devIBMGL  = SetupManager.getProperty( base+"fig2devIBMGL", "-L ibmgl" );
    fig2devLATEX  = SetupManager.getProperty( base+"fig2devLATEX", "-L latex" );
    fig2devPSTEX  = SetupManager.getProperty( base+"fig2devPSTEX", "-L latex" );
    fig2devPICTEX = SetupManager.getProperty( base+"fig2devPICTEX","-L pictex");

    fig2devGIF    = SetupManager.getProperty( base+"fig2devGIF",   "-L gif" );
    fig2devJPG    = SetupManager.getProperty( base+"fig2devJPG",   "-L jpeg" );
    fig2devPPM    = SetupManager.getProperty( base+"fig2devPPM",   "-L ppm" );
    fig2devPNG    = SetupManager.getProperty( base+"fig2devPNG",   "-L png" );
    fig2devEEPIC  = SetupManager.getProperty( base+"fig2devEEPIC", "-L eepic" );
    fig2devEPIC  = SetupManager.getProperty(  base+"fig2devEPIC",  "-L epic" );

    fig2devMagnification=SetupManager.getDouble(
                                              base+"fig2devMagnification",1.0);
    fig2devPSCentered=SetupManager.getBoolean(base+"fig2devPSCentered", true);
    fig2devPSOrientation=SetupManager.getBoolean(
                                             base+"fig2devPSOrientation", true);


    if (debug) {
      System.err.println( "-#- ExportOptionsDialog.loadExportSetup: \n"
        + "debug " + debug + " Autosave " + autosave + "\n"
        + "fig2dev path " + fig2devPath + "\n"
        + "fig2devPS " + fig2devPS + "\n"
        + "fig2devEPS " + fig2devEPS + "\n"
        + "fig2devIBMGL " + fig2devIBMGL + "\n"
        + "fig2devLATEX " + fig2devLATEX + "\n"
        + "fig2devPSTEX " + fig2devPSTEX + "\n"
        + "fig2devPICTEX " + fig2devPICTEX + "\n"
      );
    }

  }


  public void getAndSetDefaultFont() {
    String name; int style, size;
    String base = "jfig.gui.ExportOptionsDialog";
    name=SetupManager.getProperty( base+"FontName","SansSerif");
    size=SetupManager.getInteger(  base+"FontSize",12 );
    style=Font.PLAIN;
    Font font = new Font( name, style, size );
    if (font != null) this.setFont( font );
  }






   /**
    * get the currently selected color option.
    * Possible values are FULL_COLORS, IO_COLORS, and BLACK_WHITE.
    */
   public int getColorOption() {
     String s = colorChoice.getSelectedItem();

     if      (s.equals( FULL_COLORS_STRING)) return FULL_COLORS;
     else if (s.equals( IO_COLORS_STRING))   return IO_COLORS;
     else if (s.equals( BLACK_WHITE_STRING)) return BLACK_WHITE;
     else {
       msg( "-E- unkwown color selection: " + s );
       return IO_COLORS;
     }
   }





   /**
    * display the size of the current drawing at selected magnification,
    * in inches, using the sizeLabel GUI component.
    * Note: This method assumes that the bounding box of the FIG objects
    * has already been calculated via updateFigObjectsBoundingBox().
    */
   public void showSize() {
     Dimension d   = new Dimension( bbox.getXr()-bbox.getXl(),
                                    bbox.getYb()-bbox.getYt() );
     double mag    = getMagnification();
     double size_x = (mag*d.width)  / 2400.0;
     double size_y = (mag*d.height) / 2400.0;

     sizeLabel.setText( "  ["
       + new jfig.utils.Format("%3.1f").form( size_x ) + " x "
       + new jfig.utils.Format("%3.1f").form( size_y ) + " "
       + "inches]  " );
   }


   /**
    * set a new value for the magnification factor,
    * e.g. setMagnification( 1.0 );
    */
   public void setMagnification( double d ) {
     fig2devMagnification = d;
     int percent = (int) (100 * d + 0.5);
     magnificationTF.setText( "" + percent );
   }


   /**
    * return the magnification factor to use for exporting or printing.
    */
   public double getMagnification() {
     double mag;
     try {
       mag = 0.01 * Double.valueOf(
                              magnificationTF.getText().trim() ).doubleValue();
     }
     catch( Exception e ) {
       msg( "-W- Illegal NumberFormat in getMagnification: " +
             magnificationTF.getText().trim() );
       msg( "-W- using mag=100% instead" );
       mag = 1.0;
     }
     fig2devMagnification = mag;
     return mag;
   }


   public int getOrientation() {
     return
       ("landscape".equals( orientationChoice.getSelectedItem()))
       ? FigAttribs.Landscape : FigAttribs.Portrait;
   }


   public int getJustification() {
     return
       ("center".equals( centerChoice.getSelectedItem()))
       ? FigAttribs.CenterJustification
       : FigAttribs.FlushLeftJustification;
   }


  /**
   * set and show the filename selected by the user
   */
  public void setFilename( String filename ) {
    filenameTF.setText( filename );
  }



  /**
   * get filename extensions corresponding to current export
   * language selection. <p>
   */
  public String getExtension() {
    return data[languageChoice.getSelectedIndex() ][1];
  }


  /**
   * update the current filename's extension based on the currently
   * selected export language, e.g. update "foo.gl" to "foo.ppm"
   * if PPM is selected.
   * Note: This method assumes that filename is non-null.
   */
  public void updateOutFilenameExtension() {
    dbg( "-#- updateOutFilenameExtension..." );
    String filename = filenameTF.getText();
    try {
      int ix = filename.lastIndexOf( '.' );
      if (ix >= 0) filename = filename.substring( 0, ix )+ '.' +getExtension();
      else         filename = filename + '.' + getExtension();
    }
    catch( Exception e ) { dbg( "-#- " + e ); }

    filenameTF.setText( filename );
  }


  public void updateCommandLine() {
    dbg( "-#- updateCommandLine..." );

    int     lang        = languageChoice.getSelectedIndex();
    String  figFilename = getFigFilename();
    String  outFilename = getOutFilename();
    String  commandLine = null;

    switch( lang ) {
      case  0: // Postscript
               commandLine = fig2devPath + " " + fig2devPS
                             + getFig2devMagnification()
                             + getFig2devPSCentered()
                             + getFig2devPSOrientation()
                             + getFig2devPSOffset()
                             + " " + figFilename
                             + " " + outFilename;
               break;
      case  1: // EPS
               commandLine = fig2devPath + " " + fig2devEPS
                             + getFig2devMagnification()
                             + getFig2devPSCentered()
                             + getFig2devPSOrientation()
                             + getFig2devPSOffset()
                             + " " + figFilename
                             + " " + outFilename;
               break;
      case  2: // IBM GL
               commandLine = fig2devPath + " " + fig2devIBMGL
                             + getFig2devMagnification() + " "
                             + figFilename + " "
                             + outFilename;
               break;
      case  3: // LATEX
               commandLine = fig2devPath + " " + fig2devLATEX
                             + getFig2devMagnification()
                             + " " + figFilename
                             + " " + outFilename;
               break;
      case  4: // PICTEX
               commandLine = fig2devPath + " " + fig2devPICTEX
                             + getFig2devMagnification()
                             + " " + figFilename
                             + " " + outFilename;
               break;
      case  5: // PSTEX
               commandLine = "xxx (not yet supported)";
               break;
      case  6: // PPM
               commandLine = fig2devPath + " " + fig2devPPM
                             + getFig2devMagnification()
                             + " " + figFilename
                             + " " + outFilename;
               break;
      case  7: // JPG
               commandLine = fig2devPath + " " + fig2devJPG
                             + getFig2devMagnification()
                             + " -q 70"        // quality
                             + " " + figFilename
                             + " " + outFilename;
               break;
      case  8: // GIF
               commandLine = fig2devPath + " " + fig2devGIF
                             + getFig2devMagnification()
                             + " " + figFilename
                             + " " + outFilename;
               break;
      case  9: // PNG
               commandLine = fig2devPath + " " + fig2devPNG
                             + getFig2devMagnification()
                             + " " + figFilename
                             + " " + outFilename;
               break;
      case  10: // EPIC
               commandLine = fig2devPath + " " + fig2devEPIC
                             + getFig2devMagnification()
                             + " " + figFilename
                             + " " + outFilename;
               break;
      case  11: // EEPIC
               commandLine = fig2devPath + " " + fig2devEEPIC
                             + getFig2devMagnification()
                             + " " + figFilename
                             + " " + outFilename;
               break;


      default:
               msg( "-E- internal: unknown export language: " + lang );
               return;
    }

    commandLineTF.setText( commandLine );
  }


  public String getFigFilename() {
    if (editor == null) return "unnamed.fig";

    String s = editor.getFilename();
    if (s == null) return "unnamed.fig";
    else           return "\""+ s + "\"";
  }


  public String getOutFilename() {
    String dir  = directoryTF.getText();
    String name = filenameTF.getText();

    if (dir != null && dir.length() > 0) {
      return dir + File.separator + name;
    }
    else if (name == null || name.length() == 0) {
      return "unnamed.ps";
    }
    else {
      return name;
    }
  }


  /**
   * this method should be called whenever a new FIG file is loaded
   * into the editor, to guarantee that the output filename is
   * changed to match the new input file.
   * It should not be called for the same file, in order not to overwrite
   * any name changes by the user (via the browse button and file dialog).
   */
  public void updateOutFilenameFromFigFilename() {
    dbg( "-#- updateOutFilenameFromFigFilename..." );
    String source = null, path = null, name = null;
    try {
      source = getFigFilename();
      int  i = source.lastIndexOf( File.separatorChar );

      if (i < 0) {  // no separator char: use working directory
         path = "";
         name = source.substring( i+1 );
      }
      else {
         path = source.substring( 0, i );
         name = source.substring( i+1 );
      }
      filenameTF.setText( name );
      directoryTF.setText( path );

      updateOutFilenameExtension();
    }
    catch( Exception e ) {
      msg( "-E- internal: " + e );
      msg( "-E- filenames: " + source + " " + path + " " + name );
      e.printStackTrace();
    }
  }



  public String getFig2devMagnification() {
    return " -m " + getRoundedValue( getMagnification());
  }

  public String getRoundedValue( double d ) {
    return new jfig.utils.Format( "%3.2f" ).form( d );
  }


  public String getFig2devPSOrientation() {
    if ("landscape".equals( orientationChoice.getSelectedItem()))
         return " -l dummy";
    else return "";
  }

  public void setFig2devPSOrientation( boolean portrait ) {
    if (portrait) orientationChoice.select( "portrait" );
    else          orientationChoice.select( "landscape" );
  }


  public String getFig2devPSCentered() {
    if ("center".equals( centerChoice.getSelectedItem())) return " -c";
    else                                                  return " -e"; // edge
  }

  public void setFig2devPSCentered( boolean centered ) {
    if (centered) centerChoice.select( "center" );
    else          centerChoice.select( "flush left" );
  }

  public String getFig2devPSOffset() {
    double xoffset = 0.0;
    double yoffset = 0.0;
    String       s = "";

    try {
      xoffset = Double.valueOf( xoffsetTF.getText().trim() ).doubleValue();
    }
    catch( Exception e ) { }

    try {
      yoffset = Double.valueOf( yoffsetTF.getText().trim() ).doubleValue();
    }
    catch( Exception e ) { }

    if (xoffset != 0) s = s + " -x " + getRoundedValue(xoffset/72.0);
    if (yoffset != 0) s = s + " -y " + getRoundedValue(yoffset/72.0);

    return s;
  }



  private void call( String methodname ) {
    try {
      Class[]  formals = new Class[1];
      formals[0]       = Object.class;
      Method   method  = getClass().getMethod( methodname, formals );

      Object[] args = new Object[0];
      method.invoke( this, args );
    }
    catch( InvocationTargetException ite  ) {
      Throwable t = ite.getTargetException();
      msg( "-E- call: " + methodname + " " + t );
      t.printStackTrace();
    }
    catch( NoSuchMethodException e ) {
      msg( "-E- call: " + e + ":" + methodname );
    }
    catch( Exception e ) {
      msg( "-E- call: " + e + " for command: " + methodname );
      e.printStackTrace();
    }
  }


  /**
   * set the magnification factor to fit all objects into a page of
   * format A4, landscape orientation.
   * Show the corresponding magnification factor and size in the
   * exportOptionsDialog.
   * Note: This method assumes that the bounding box of all Fig objects
   * has already been calculated via updateFigObjectsBoundingBox().
   */
  public void fitToLandscapeA4() {
    Dimension  d = new Dimension( bbox.getXr()-bbox.getXl(),
                                  bbox.getYb()-bbox.getYt());
    double factor, fx, fy, dx, dy;

    fx = 1.0 * A4_WIDTH / d.width;
    fy = 1.0 * A4_HEIGHT / d.height;

    // the smaller of both, but at least 0.01
    factor = Math.min( fx, fy );
    factor = Math.max( 0.01, factor );
    dx     = factor * d.width;
    dy     = factor * d.height;

    setMagnification( factor );
    showSize();
  }


  /**
   * (re-) calculate the bounding box of all FIG objects on objectCanvas
   */
  public void updateFigObjectsBoundingBox() {
     try {
       bbox = BoundingBoxCalculator.getBoundingBox(
               editor.getDrawableObjects() );
     }
     catch( Throwable t ) {
       bbox = new FigBbox( 0, 0, A4_WIDTH, A4_HEIGHT );
     }
     if (debug) msg( "-#- updateFigObjectBoundingBox: " + bbox );

     showSize();
  }


  /**
   * this method just calls Runtime.exec() with the current value
   * of the 'command line'.
   * The motivation for this is to allow the user to edit the command line
   * as desired.
   */
  public void doExportNow() {
    dbg( "-#- doExportNow: " + commandLineTF.getText() );
    doAutoSave();
    try {
      Process PR = Runtime.getRuntime().exec( commandLineTF.getText() );

      if (waitExec) {
        PR.waitFor();
        printSubprocessOutput( PR );
      }
    }
    catch( Exception e ) {
      msg( "-E- exception during export: " + e );
      e.printStackTrace();
    }
  };


  public void doExportPS() {
    languageChoice.select( 0 ); // Postscript
    updateOutFilenameExtension();
    updateCommandLine();
    doExportNow();
  }



  public void doExportEPS() {
    languageChoice.select( 1 ); // EPS
    updateOutFilenameExtension();
    updateCommandLine();
    doExportNow();
  }



  public void doExportNativeGIF() {
    languageChoice.select( 8 ); // GIF
    updateOutFilenameExtension();
    updateCommandLine();
    String outFilename = getOutFilename();
    msg( "-I- exporting GIF to file '" + outFilename + "'..." );

    //setWaitCursor();
    FigCanvas objectCanvas = editor.getObjectCanvas();
    try {
      int xsize = objectCanvas.getSize().width;
      int ysize = objectCanvas.getSize().height;
      Image im = this.createImage( xsize, ysize );
      if (im == null) {
         msg( "-E- Failed to create buffer image, cannot write GIF..." );
         return;
      }

      msg( "...drawing objects..." );
      Graphics g = im.getGraphics();
      g.setColor( Color.white );
      g.fillRect( 0, 0, xsize, ysize );
      objectCanvas.drawAllObjects( g );
      g.dispose();

      msg( "...encoding GIF..." );
      GIFEncoder.GIFEncoder encoder = new GIFEncoder.GIFEncoder( im );

      msg( "...writing GIF to file..." );
      FileOutputStream gifFile = new FileOutputStream( outFilename );
      OutputStream of = new BufferedOutputStream( gifFile );
      encoder.Write( of );
      gifFile.close();
    }
    catch( Exception e ) {
      msg( "-E- Exception during GIF export: " + e.toString() );
      ExceptionTracer.trace( e );
    }
    //setDefaultCursor();
    msg( "...ready" );
  }



  public void doExportNativePPM() {
    languageChoice.select( 6 ); // PPM
    updateOutFilenameExtension();
    updateCommandLine();
    String outFilename = getOutFilename();
    msg( "-I- exporting PPM to file '" + outFilename + "'..." );

    //setWaitCursor();
    FigCanvas objectCanvas = editor.getObjectCanvas();

    try {
      int xsize = objectCanvas.getSize().width;
      int ysize = objectCanvas.getSize().height;
      Image im = this.createImage( xsize, ysize );
      if (im == null) {
         msg( "-E- Failed to create buffer image, cannot write PPM..." );
         return;
      }

      msg( "...drawing objects..." );
      Graphics g = im.getGraphics();
      g.setColor( Color.white );
      g.fillRect( 0, 0, xsize, ysize );
      objectCanvas.drawAllObjects( g );
      g.dispose();

      msg( "...encoding and writing PPM..." );
      DataOutputStream ppmstream = new DataOutputStream(
                                   new BufferedOutputStream(
                                     new FileOutputStream( outFilename )));
      new PPMWriter().writePPM( im, ppmstream );
      ppmstream.close();
      msg( "...ok. " );
    }
    catch( Exception e ) {
      msg( "-E- Exception during PPM export: " + e.toString() );
      ExceptionTracer.trace( e );
    }
    //setDefaultCursor();
    msg( "...ready" );
  }


  public void doExportEPIC() {
    languageChoice.select( 10 ); // EPIC
    updateOutFilenameExtension();
    updateCommandLine();
    doExportNow();
  }

 public void doExportEEPIC() {
    languageChoice.select( 11 ); // EEPIC
    updateOutFilenameExtension();
    updateCommandLine();
    doExportNow();
  }


  public void doExportPrintPS() {
    languageChoice.select( 0 ); // Postscript
    updateOutFilenameExtension();
    updateCommandLine();

    String  figFilename = getFigFilename();
    String  outFilename = getOutFilename();

    String cmd1 = fig2devPath + " " + fig2devPS
                  + getFig2devMagnification()
                  + getFig2devPSCentered()
                  + getFig2devPSOrientation()
                  + getFig2devPSOffset()
                  + " " + figFilename
                  + " " + outFilename;

    String cmd2 = printCommand + " " + outFilename;

    try {
       Process PR;
       PR = Runtime.getRuntime().exec( cmd1 );
       msg( "-I- " + PR.toString() + ": " + cmd1 );
       PR.waitFor();
       printSubprocessOutput( PR );

       PR = Runtime.getRuntime().exec( cmd2 );
       msg( "-I- " + PR.toString() + ": " + cmd2 );
       PR.waitFor();
       printSubprocessOutput( PR );
    }
    catch( Exception e ) {
       msg( "-E- Exception during export: " + e.toString() );
       ExceptionTracer.trace( e );
    }
  }




  public void doAutoSave() {
    ; // empty
  }


  public void doAutoSaveDummy() {
    File tmpfile = null;
    try {
      tmpfile = File.createTempFile( "jfig", ".fig" );
      tmpfile.deleteOnExit();
    }
    catch( java.lang.NoSuchMethodError e ) { // we're running on JDK 1.1
      int i = 10000;
      boolean exists = true;
      while( exists ) {
        i = i + 1;
        tmpfile = new File( "jfig" + i + ".fig" );
        exists = tmpfile.exists();

        if (i > 11000) throw new Error( "could not create tmp file, tried 10000 times." );
      }
    }
    catch( Throwable t ) {
      t.printStackTrace();
    }

    try {
      FileOutputStream fos = new FileOutputStream( tmpfile );
      fos.write( 'x' );
      fos.close();
    }
    catch( Throwable t ) {
      t.printStackTrace();
    }
  }


  public void printSubprocessOutput( Process PR ) throws IOException {
    BufferedReader out, err;
    out = new BufferedReader( new InputStreamReader( PR.getInputStream() ));
    err = new BufferedReader( new InputStreamReader( PR.getErrorStream() ));

    String  s = null;
    if (err == null) return;
    while( (s = err.readLine()) != null) {
      msg( s );
    }
    if (out == null) return;
    while( (s = out.readLine()) != null) {
      msg( s );
    }
  }



  public void doBrowseFilename() {
    if (fileDialog == null) {
      fileDialog = new FileDialog( this, "Select export file:" );
    }
    fileDialog.setMode( FileDialog.SAVE );
    fileDialog.show();

    if (fileDialog.getFile() != null) {
      filenameTF.setText(  fileDialog.getFile() );
      directoryTF.setText( fileDialog.getDirectory() );
    }

    updateCommandLine();
  }



  public void msg( String s ) {
    System.out.println( s );
  }


  public void dbg( String s ) {
    if (debug) System.out.println( s );
  }


  /**
   * selftest for debugging
   */
  public static void main( String argv[] ) {
    SetupManager.loadLocalProperties( "jfig.cnf" );
    ExportOptionsDialog f = new ExportOptionsDialog();
    f.updateFigObjectsBoundingBox();
    f.show();

    f.setModularEditor( null );
    f.updateOutFilenameFromFigFilename();
  }
}


/* end ExportOptionsDialog.java */
