package dks.src.borderEditor;

import java.awt.BasicStroke;
import java.awt.Color;
import java.io.ObjectStreamException;
import java.io.Serializable;

import javax.swing.event.ChangeListener;

import org.jdom.Attribute;
import org.jdom.Element;
import org.jdom.JDOMException;

import dks.src.utils.XML.XMLWritable;
import dks.src.utils.XML.XMLWriter;
import dks.src.utils.listener.CChangeListenerDelegate;
import dks.src.utils.listener.Changeable;

/**
 * This class implement a border which can be saved in a XML format <br> date : 3 sept. 07
 * @author   DarK Sidious
 */
public class CBorder implements XMLWritable, Serializable, Changeable {

	private static final long serialVersionUID = 8741370441205029492L;
	protected static final String ERROR_BORDER_COLOR_NOT_FOUND_DESCRIPTION = "Le fichier xml est invalide : la couleur de la bordure n'a pas t trouve";
	protected static final String ERROR_WIDTH_NOT_FOUND_DESCRIPTION = "Le fichier xml est invalide : la largeur de la bordure n'a pas t trouve";
	protected static final String XML_COLOR_PROPERTY = "color";
	protected static final String XML_WIDTH_PROPERTY = "width";
	protected static final String XML_VISIBLE_PROPERTY = "visible";

	protected int _width;
	protected Color _color;
	protected boolean _visible;
	protected transient boolean _changed;
	protected transient BasicStroke _border;
	protected transient CChangeListenerDelegate _changeListeners;

	/**
	 * The Constructor.
	 *
	 * @param color the color of the border
	 * @param width the width of the border
	 */
	public CBorder(Color color, int width) {
		_color = color;
		_width = width;
		_visible = true;
		_changed = true;
		_changeListeners = new CChangeListenerDelegate();
	}

	/**
	 * Gets the color.
	 *
	 * @return the color of the border
	 */
	public Color getColor() {
		return _color;
	}

	/**
	 * Sets the color.
	 *
	 * @param color the color of the border
	 */
	public void setColor(Color color) {
		if (!_color.equals(color)) {
			_color = color;
			_changed = true;
			_changeListeners.notifyChanged();
		}
	}

	/**
	 * Gets the width.
	 *
	 * @return the width of the border
	 */
	public int getWidth() {
		return _width;
	}

	/**
	 * Sets the width.
	 *
	 * @param width the width of the border
	 */
	public void setWidth(int width) {
		if (_width != width) {
			_width = width;
			_changed = true;
			_changeListeners.notifyChanged();
		}
	}

	/**
	 * the visibility of the border.
	 *
	 * @return the visibility of the border
	 */
	public boolean isVisible() {
		return _visible;
	}

	/**
	 * the visibility of the border.
	 *
	 * @param visible the visibility of the border
	 */
	public void setVisible(boolean visible) {
		if (_visible != visible) {
			_visible = visible;
			_changed = true;
			_changeListeners.notifyChanged();
		}
	}

	/**
	 * Gets the stroke.
	 *
	 * @return the stroke to use for apply the border to a shape
	 */
	public BasicStroke getStroke() {
		if (_changed) {
			_border = new BasicStroke(_width, BasicStroke.JOIN_ROUND, BasicStroke.CAP_ROUND);
			_changed = false;
		}
		return _border;
	}

	/**
	 * XMLload.
	 *
	 * @param root the XML DOM element used to load the properties of the border
	 *
	 * @throws JDOMException the JDOM exception
	 *
	 * @see dks.src.utils.XML.XMLWritable#XMLload(org.jdom.Element)
	 */
	public void XMLload(final Element root) throws JDOMException {
		final Color color = XMLWriter.getColorElement(XML_COLOR_PROPERTY, root);
		if (color == null) {
			throw new JDOMException(ERROR_BORDER_COLOR_NOT_FOUND_DESCRIPTION);
		}
		_color = color;

		final String width = root.getAttributeValue(XML_WIDTH_PROPERTY);
		if (width == null) {
			throw new JDOMException(ERROR_WIDTH_NOT_FOUND_DESCRIPTION);
		}
		_width = Integer.parseInt(width);

		final Attribute attribute = root.getAttribute(XML_VISIBLE_PROPERTY);
		if (attribute != null) { // Proprit rajoute dans la version 1.0.1 => compatibilit ascendante avec les fichiers de la version 1.0.0
			_visible = Boolean.parseBoolean(attribute.getValue());
		} else {
			_visible = true;
		}
		_changed = true;
	}

	/**
	 * XMLsave.
	 *
	 * @param root the XML DOM element used to save the properties of the border
	 *
	 * @see dks.src.utils.XML.XMLWritable#XMLsave(org.jdom.Element)
	 */
	public void XMLsave(Element root) {
		 root.addContent(XMLWriter.createColorElement(XML_COLOR_PROPERTY, _color));
		 root.setAttribute(XML_WIDTH_PROPERTY, Integer.valueOf(_width).toString());
		 root.setAttribute(XML_VISIBLE_PROPERTY, Boolean.valueOf(_visible).toString());
	}

	/* (non-Javadoc)
	 * @see dks.src.utils.listener.Changeable#addChangeListener(javax.swing.event.ChangeListener)
	 */
	public void addChangeListener(ChangeListener listener) {
		_changeListeners.addListener(listener);
	}

	/* (non-Javadoc)
	 * @see dks.src.utils.listener.Changeable#removeChangeListener(javax.swing.event.ChangeListener)
	 */
	public void removeChangeListener(ChangeListener listener) {
		_changeListeners.removeListener(listener);
	}

	/**
	 * Hash code.
	 *
	 * @return the hashcode of the object
	 *
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public int hashCode() {
		final int PRIME = 31;
		int result = 1;
		result = PRIME * result + ((_color == null) ? 0 : _color.hashCode());
		result = PRIME * result + (_visible ? 1231 : 1237);
		result = PRIME * result + _width;
		return result;
	}

	/**
	 * Equals.
	 *
	 * @param obj the object to compare
	 *
	 * @return the equality of the object
	 *
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object obj) {
		if (obj == null) {
			return false;
		}
		if (!(obj instanceof CBorder)) {
			return false;
		}
		final CBorder other = (CBorder) obj;
		if (_color == null) {
			if (other._color != null) {
				return false;
			}
		} else if (!_color.equals(other._color)) {
			return false;
		}
		if (_visible != other._visible) {
			return false;
		}
		if (_width != other._width) {
			return false;
		}
		return true;
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	public String toString() {
		return "dks.src.borderEditor.CBorder[width=" + _width + ",color=" + _color + ",visible=" + _visible + "]";
	}

	protected Object readResolve() throws ObjectStreamException {
		_changed = true;
		_changeListeners = new CChangeListenerDelegate();
		return this;
	}
}
