package dks.src.wordart;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Paint;
import java.awt.RenderingHints;
import java.awt.Shape;
import java.awt.font.TextLayout;
import java.awt.geom.AffineTransform;
import java.awt.geom.Rectangle2D;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;
import java.io.ObjectStreamException;
import java.io.Serializable;

import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import org.jdom.Attribute;
import org.jdom.Element;
import org.jdom.JDOMException;

import dks.src.borderEditor.CBorder;
import dks.src.shadowEditor.CShadow;
import dks.src.textureEditor.CTexture;
import dks.src.utils.XML.XMLWriter;
import dks.src.utils.pictures.CPicture;
import dks.src.warpEditor.CWarp;

/**
 * This class implements the dksWordArt which can be saved to a XML format <br> date : 5 sept. 07
 * @author   DarK Sidious
 */
public class CWordArt implements Serializable {

	private static final long serialVersionUID = -7963946868275063637L;

	protected static final String XML_DKSWORDART_PROPERTY = "dksWordArt";
	protected static final String XML_BORDER_PROPERTY = "border";
	protected static final String XML_TEXTURE_PROPERTY = "texture";
	protected static final String XML_SHADOW_PROPERTY = "shadow";
	protected static final String XML_WARP_PROPERTY = "warp";
	protected static final String XML_FONT_PROPERTY = "font";
	protected static final String XML_TEXT_PROPERTY = "text";
	protected static final String XML_BACKGROUND_COLOR_PROPERTY = "backgroundColor";
	protected static final String XML_DIMENSION_PROPERTY = "dimension";
	protected static final String XML_STRETCH_PROPERTY = "stretch";
	protected static final String XML_AUTOSIZEFONT_PROPERTY = "autosizefont";
	protected static final String XML_TOTALSIZE_PROPERTY = "totalSize";

	protected static final Color DEFAULT_BACKGROUND_COLOR = new Color(255, 255, 255, 0);
	protected static final Dimension DEFAULT_DIMENSION = new Dimension(200, 50);
	protected static final Dimension DEFAULT_TOTALSIZE = new Dimension(400, 100);
	protected static final boolean DEFAULT_STRETCH = true;
	protected static final boolean DEFAULT_AUTO_SIZE_FONT = true;

	protected CBorder _border;
	protected CShadow _shadow;
	protected CTexture _texture;
	protected CWarp _warp;
	protected String _text;
	protected Font _font;
	protected Color _backgroundColor;
	protected Dimension _dimension;
	protected boolean _stretch;
	protected boolean _autoSizeFont;
	protected Dimension _totalSize;

	protected transient BufferedImage _image;
	protected transient BufferedImage _stretchedImage;
	protected transient BufferedImage _warpImageTemp;
	protected transient BufferedImage _warpImage;

	protected transient BorderChange _borderChangeListener;
	protected transient TextureChange _textureChangeListener;
	protected transient WarpChange _warpChangeListener;
	protected transient ShadowChange _shadowChangeListener;

	protected transient boolean _borderChanged;
	protected transient boolean _textureChanged;
	protected transient boolean _warpChanged;
	protected transient boolean _shadowChanged;
	protected transient boolean _wordArtChanged;

	protected transient Shape _contours;
	protected transient Paint _texturePaint;

	/**
	 * @param text the text of the wordart
	 * @param font the font of the text
	 * @param texture the texture to use
	 * @param shadow the shadow to use
	 * @param warp the warp to use
	 */
	public CWordArt(String text, Font font, CTexture texture, CShadow shadow, CWarp warp) {
		_border = new CBorder(Color.BLACK, 2);
		_text = text;
		_texture = texture;
		_shadow = shadow;
		_warp = warp;
		_font = font;
		_backgroundColor = DEFAULT_BACKGROUND_COLOR;
		_dimension = DEFAULT_DIMENSION;
		_totalSize = DEFAULT_TOTALSIZE;
		_stretch = DEFAULT_STRETCH;
		_autoSizeFont = DEFAULT_AUTO_SIZE_FONT;
		_borderChangeListener = new BorderChange();
		if (_border != null) {
			_border.addChangeListener(_borderChangeListener);
		}
		_textureChangeListener = new TextureChange();
		if (_texture != null) {
			_texture.addChangeListener(_textureChangeListener);
		}
		_warpChangeListener = new WarpChange();
		if (_warp != null) {
			_warp.addChangeListener(_warpChangeListener);
		}
		_shadowChangeListener = new ShadowChange();
		if (_shadow != null) {
			_shadow.addChangeListener(_shadowChangeListener);
		}
		_borderChanged = true;
		_textureChanged = true;
		_warpChanged = true;
		_shadowChanged = true;
		_wordArtChanged = true;
	}

	/**
	 * @param text the text of the wordart
	 * @param font the font of the text
	 * @param shadow the shadow to use
	 * @param warp the warp to use
	 */
	public CWordArt(String text, Font font, CShadow shadow, CWarp warp) {
		this(text, font, null, shadow, warp);
	}

	/**
	 * @param text the text of the wordart
	 * @param font the font of the text
	 * @param texture the texture to use
	 * @param warp the warp to use
	 */
	public CWordArt(String text, Font font, CTexture texture, CWarp warp) {
		this(text, font, texture, null, warp);
	}

	/**
	 * @param text the text of the wordart
	 * @param font the font of the text
	 * @param texture the texture to use
	 * @param shadow the shadow to use
	 */
	public CWordArt(String text, Font font, CTexture texture, CShadow shadow) {
		this(text, font, texture, shadow, null);
	}

	/**
	 * @param text the text of the wordart
	 * @param font the font of the text
	 * @param warp the warp to use
	 */
	public CWordArt(String text, Font font, CWarp warp) {
		this(text, font, null, null, warp);
	}

	/**
	 * @param text the text of the wordart
	 * @param font the font of the text
	 * @param texture the texture to use
	 */
	public CWordArt(String text, Font font, CTexture texture) {
		this(text, font, texture, null, null);
	}

	/**
	 * @param text the text of the wordart
	 * @param font the font of the text
	 * @param shadow the shadow to use
	 */
	public CWordArt(String text, Font font, CShadow shadow) {
		this(text, font, shadow, null);
	}

	/**
	 * @param text the text of the wordart
	 * @param font the font of the text
	 */
	public CWordArt(String text, Font font) {
		this(text, font, null, null, null);
	}

	public CShadow getShadow() {
		return _shadow;
	}

	/**
	 * @param shadow the shadow to use
	 */
	public void setShadow(CShadow shadow) {
		if (!_shadow.equals(shadow)) {
			_shadow = shadow;
			_shadow.addChangeListener(_shadowChangeListener);
			_shadowChanged = true;
		}
	}

	/**
	 * @return the text to display
	 */
	public String getText() {
		return _text;
	}

	/**
	 * @param text the text to display
	 */
	public void setText(String text) {
		if (!_text.equals(text)) {
			_text = text;
			_wordArtChanged = true;
		}
	}

	/**
	 * @return the texture to use
	 */
	public CTexture getTexture() {
		return _texture;
	}

	/**
	 * @param texture the texture to use
	 */
	public void setTexture(CTexture texture) {
		if (!_texture.equals(texture)) {
			_texture = texture;
			_texture.addChangeListener(_textureChangeListener);
			_textureChanged = true;
		}
	}

	/**
	 * @return the warp to use
	 */
	public CWarp getWarp() {
		return _warp;
	}

	/**
	 * @param warp the warp to use
	 */
	public void setWarp(CWarp warp) {
		if (!_warp.equals(warp)) {
			_warp = warp;
			_warp.addChangeListener(_warpChangeListener);
			_warpChanged = true;
		}
	}

	/**
	 * @return the font of the text
	 */
	public Font getFont() {
		return _font;
	}

	/**
	 * @param font the font of the text
	 */
	public void setFont(Font font) {
		if (!_font.equals(font)) {
			_font = font;
			_wordArtChanged = true;
		}
	}

	/**
	 * @return the border of the text
	 */
	public CBorder getBorder() {
		return _border;
	}

	/**
	 * @param border the border of the text
	 */
	public void setBorder(CBorder border) {
		if (!_border.equals(border)) {
			_border = border;
			_border.addChangeListener(_borderChangeListener);
			_borderChanged = true;
		}
	}

	/**
	 * the background color of the wordart
	 * @return the background color of the wordart
	 */
	public Color getBackgroundColor() {
		return _backgroundColor;
	}

	/**
	 * the background color of the wordart
	 * @param backgroundColor the background color of the wordart
	 */
	public void setBackgroundColor(Color backgroundColor) {
		_backgroundColor = backgroundColor;
	}

	/**
	 * the dimension of the wordart
	 * @return the dimension of the wordart
	 */
	public Dimension getDimension() {
		return _dimension;
	}

	/**
	 * the dimension of the wordart
	 * @param dimension the dimension of the wordart
	 */
	public void setDimension(Dimension dimension) {
		if (!_dimension.equals(dimension)) {
			_dimension = dimension;
			_wordArtChanged = true;
		}
	}

	/**
	 * the stretching of the wordart
	 * @return the stretching of the wordart
	 */
	public boolean isStretch() {
		return _stretch;
	}

	/**
	 * the stretching of the wordart
	 * @param stretch the stretching of the wordart
	 */
	public void setStretch(boolean stretch) {
		if (_stretch != stretch) {
			_stretch = stretch;
			_wordArtChanged = true;
		}
	}

	/**
	 * @return if the font can be modify when the dimension of the wordart change
	 */
	public boolean isAutoSizeFont() {
		return _autoSizeFont;
	}

	/**
	 * @param autoSizeFont if true, the font will be modified when the dimension of the wordart change, and the font have the optimal size
	 */
	public void setAutoSizeFont(boolean autoSizeFont) {
		if (_autoSizeFont != autoSizeFont) {
			_autoSizeFont = autoSizeFont;
			_wordArtChanged = true;
		}
	}

	/**
	 * @return the totalSize of the wordart (the size of the final picture)
	 */
	public Dimension getTotalSize() {
		return _totalSize;
	}

	/**
	 * @param totalSize the totalSize of the wordart (the size of the final picture)
	 */
	public void setTotalSize(Dimension totalSize) {
		if (!_totalSize.equals(totalSize)) {
			_wordArtChanged = true;
			_totalSize = totalSize;
		}
	}

	/**
	 * @param g the graphics used to display the wordart
	 * @param x the position in the X axis of the wordart
	 * @param y the position in the Y axis of the wordart
	 * @param width the width of the wordart
	 * @param height the height of the wordart
	 * @deprecated replaced by the draw(Graphics g, int x, int y) : use the totalSize property to define the width and the height of the wordart
	 */
	@Deprecated
	public void draw(Graphics g, int x, int y, int width, int height) {
		draw(g, x, y, _totalSize.width, _totalSize.height, false);
	}

	/**
	 * @param g the graphics used to display the wordart
	 * @param x the position in the X axis of the wordart
	 * @param y the position in the Y axis of the wordart
	 * The size of the wordart is defined by the totalSize property
	 */
	public void draw(Graphics g, int x, int y) {
		draw(g, x, y, _totalSize.width, _totalSize.height, false);
	}

	/**
	 * @param g the graphics used to display the wordart
	 * @param x the position in the X axis of the wordart
	 * @param y the position in the Y axis of the wordart
	 * @param width the width of the wordart
	 * @param height the height of the wordart
	 * @param antialiasing the activity of the antialiasing
	 * @deprecated replaced by the draw(Graphics g, int x, int y, boolean antialiasing) : use the totalSize property to define the width and the height of the wordart
	 */
	@Deprecated
	public void draw(Graphics g, int x, int y, int width, int height, boolean antialiasing) {
		// optimisation : it's not necessary to compute the image if nothing has changed
		if (_image != null && !_wordArtChanged && !_borderChanged && !_textureChanged && !_warpChanged && !_shadowChanged) {
			final Graphics2D g2d = (Graphics2D) g;
			g2d.setBackground(getBackgroundColor());
			g2d.clearRect(0, 0, width, height);
			g2d.drawImage(_image, 0, 0, null);
			g2d.dispose();
			return;
		}

		int xDraw = x + (width - _dimension.width) / 2;
		int yDraw = y + (height - _dimension.height) / 2;

		// optimisation : it's not necessary to create a new buffer if the picture has the same size
		_image = CPicture.resizeAndClearImage(_image, width, height);
		final Graphics2D copyGraphics = _image.createGraphics();
		final Graphics2D g2d = (Graphics2D) g;

		// clear the background
		g2d.setBackground(getBackgroundColor());
		g2d.clearRect(0, 0, width, height);
		if (antialiasing) {
			copyGraphics.setRenderingHint(RenderingHints.KEY_ALPHA_INTERPOLATION, RenderingHints.VALUE_ALPHA_INTERPOLATION_QUALITY);
			copyGraphics.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
			copyGraphics.setRenderingHint(RenderingHints.KEY_COLOR_RENDERING, RenderingHints.VALUE_COLOR_RENDER_QUALITY);
			copyGraphics.setRenderingHint(RenderingHints.KEY_DITHERING, RenderingHints.VALUE_DITHER_ENABLE);
			copyGraphics.setRenderingHint(RenderingHints.KEY_FRACTIONALMETRICS, RenderingHints.VALUE_FRACTIONALMETRICS_ON);
			copyGraphics.setRenderingHint(RenderingHints.KEY_INTERPOLATION, RenderingHints.VALUE_INTERPOLATION_BICUBIC);
			copyGraphics.setRenderingHint(RenderingHints.KEY_STROKE_CONTROL, RenderingHints.VALUE_STROKE_NORMALIZE);
			copyGraphics.setRenderingHint(RenderingHints.KEY_RENDERING, RenderingHints.VALUE_RENDER_QUALITY);
			copyGraphics.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_GASP);
		} else {
			copyGraphics.setRenderingHint(RenderingHints.KEY_ALPHA_INTERPOLATION, RenderingHints.VALUE_ALPHA_INTERPOLATION_SPEED);
			copyGraphics.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_OFF);
			copyGraphics.setRenderingHint(RenderingHints.KEY_COLOR_RENDERING, RenderingHints.VALUE_COLOR_RENDER_SPEED);
			copyGraphics.setRenderingHint(RenderingHints.KEY_DITHERING, RenderingHints.VALUE_DITHER_DISABLE);
			copyGraphics.setRenderingHint(RenderingHints.KEY_FRACTIONALMETRICS, RenderingHints.VALUE_FRACTIONALMETRICS_OFF);
			copyGraphics.setRenderingHint(RenderingHints.KEY_INTERPOLATION, RenderingHints.VALUE_INTERPOLATION_NEAREST_NEIGHBOR);
			copyGraphics.setRenderingHint(RenderingHints.KEY_STROKE_CONTROL, RenderingHints.VALUE_STROKE_PURE);
			copyGraphics.setRenderingHint(RenderingHints.KEY_RENDERING, RenderingHints.VALUE_RENDER_SPEED);
			copyGraphics.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_OFF);
		}
		prepare(g);

		if (_wordArtChanged) {
			final TextLayout layout = new TextLayout(getText(), getFont(), copyGraphics.getFontRenderContext());
			Rectangle2D bounds = getBounds(g);
			_contours = layout.getOutline(AffineTransform.getTranslateInstance(0, bounds.getHeight()));
		}

		if (_texture != null) {
			_texturePaint = _texture.getPaint();
		}
		copyGraphics.setPaint(_texturePaint);
		copyGraphics.fill(_contours);

		if (_border.isVisible()) {
			copyGraphics.setStroke(_border.getStroke());
			copyGraphics.setColor(_border.getColor());
			copyGraphics.draw(_contours);
		}

		if (_stretch) {
			// optimisation : it's not necessary to compure the stretching picture if the appearence has not changed
			if (_wordArtChanged || _borderChanged || _textureChanged) {
				_stretchedImage = CPicture.resizeAndClearImage(_stretchedImage, _totalSize.width, _totalSize.height);
				Graphics2D graphics = (Graphics2D) _stretchedImage.getGraphics();
				graphics.drawImage(_image, 0, 0, _dimension.width, _dimension.height, 0, 0, _contours.getBounds().width, _contours.getBounds().height, null);

				final Graphics2D graphicsTemp = (Graphics2D) _image.getGraphics();
				graphicsTemp.setBackground(new Color(0, 0, 0, 0));
				graphicsTemp.clearRect(0, 0, _image.getWidth(), _image.getHeight());

				graphicsTemp.drawImage(_stretchedImage, 0, 0, null);
				graphicsTemp.dispose();
			}
		}
		if (_image != null) {
			if (getWarp().isActive()) {
				int warpXOffset = 0;
				int warpYOffset = 0;
				// optimisation : it's not necessary to compure the warp picture if only the shadow has changed
				if (_wordArtChanged || _borderChanged || _textureChanged || _warpChanged) {
					if (_stretch) {
						getWarp().setImage(_stretchedImage);
					} else {
						getWarp().setImage(_image);
					}
					_warpImage = getWarp().getWarpImage(_totalSize.width, _totalSize.height);
					warpXOffset = -getWarp().getXOffset();
					warpYOffset = -getWarp().getYOffset();
				}
				int widthDraw = _dimension.width + xDraw;
				int heightDraw = _dimension.height + yDraw;
				if (xDraw < 0) {
					widthDraw = _dimension.width + (xDraw * 2);
				}
				if (yDraw < 0) {
					heightDraw = _dimension.height + (yDraw * 2);
				}
				if (getShadow().isVisible()) {
					getShadow().setImage(_warpImage);
					getShadow().draw(g, xDraw + warpXOffset, yDraw + warpYOffset, width, height, 0, 0, widthDraw, heightDraw);
				} else {
					g.drawImage(_warpImage, xDraw + warpXOffset, yDraw + warpYOffset, width, height, 0, 0, widthDraw, heightDraw, null);
				}
			} else {
				if (getShadow().isVisible()) {
					getShadow().setImage(_image);
					getShadow().draw(g, xDraw, yDraw);
				} else {
					g.drawImage(_image, xDraw, yDraw, null);
				}
			}
		}
		copyGraphics.dispose();
	}

	/**
	 * @param g the graphics used to display the wordart
	 * @param x the position in the X axis of the wordart
	 * @param y the position in the Y axis of the wordart
	 * @param antialiasing the activity of the antialiasing
	 */
	public void draw(Graphics g, int x, int y, boolean antialiasing) {
		draw(g, x, y, _totalSize.width, _totalSize.height, antialiasing);
	}

	public void prepare(Graphics g) {
		if (_wordArtChanged) {
			if (_autoSizeFont) {
				Graphics2D g2d = (Graphics2D) g;
				_font = new Font(_font.getFontName(), _font.getStyle(), 256);
				g2d.setFont(_font);
				Rectangle2D bounds = getBounds(g);
				while ((bounds.getWidth() > _dimension.getWidth() || bounds.getHeight() > _dimension.getHeight()) && _font.getSize() > 0) {
					_font = new Font(_font.getFontName(), _font.getStyle(), _font.getSize() - 1);
					g2d.setFont(_font);
					bounds = getBounds(g);
				}
			}
		}
	}

	/**
	 * @param g the graphics used to display the wordard
	 * @return the bounds of the wordart
	 */
	public Rectangle2D getBounds(Graphics g) {
		final Graphics2D g2d = (Graphics2D) g;
		g2d.setStroke(_border.getStroke());
		TextLayout layout = new TextLayout(_text, _font, g2d.getFontRenderContext());
		return new Rectangle2D.Double(0.0, 0.0, layout.getBounds().getWidth(), layout.getBounds().getHeight());
	}

	/**
	 * @param file the file used to load the WordArt
	 * @throws IOException
	 * @throws JDOMException
	 */
	public void XMLload(File file) throws IOException, JDOMException {
		final Element root = XMLWriter.openDocument(file);
		if (!XML_DKSWORDART_PROPERTY.equals(root.getName())) {
			throw new JDOMException("L'lment root du fichier est invalide.");
		}
		XMLload(root);
	}

	/**
	 * @param root the DOM root element used to load the WordArt
	 * @throws IOException
	 * @throws JDOMException
	 */
	public void XMLload(Element root) throws IOException, JDOMException {
		_border.XMLload(root.getChild(XML_BORDER_PROPERTY));
		_texture.XMLload(root.getChild(XML_TEXTURE_PROPERTY));
		_shadow.XMLload(root.getChild(XML_SHADOW_PROPERTY));
		_warp.XMLload(root.getChild(XML_WARP_PROPERTY));
		_text = root.getAttributeValue(XML_TEXT_PROPERTY);
		if (_text == null) {
			throw new JDOMException("Le fichier est invalide : le texte n'a pas t trouv");
		}
		_font = XMLWriter.getFontElement(XML_FONT_PROPERTY, root);
		if (_font == null) {
			throw new JDOMException("Le fichier est invalide : la police de texte n'a pas t trouve.");
		}
		Element element = root.getChild(XML_BACKGROUND_COLOR_PROPERTY);
		if (element == null) { // cette fonctionnalit a t ajoute depuis la version 1.0.1 => compatibilit ascendante avec les fichiers des versions prcdentes
			_backgroundColor = DEFAULT_BACKGROUND_COLOR;
		} else {
			_backgroundColor = XMLWriter.getColorElement(XML_BACKGROUND_COLOR_PROPERTY, root);
		}
		element = root.getChild(XML_DIMENSION_PROPERTY);
		if (element == null) { // cette fonctionnalit a t ajoute depuis la version 1.1.0 => compatibilit ascendante avec les fichiers des versions prcdentes
			_dimension = DEFAULT_DIMENSION;
		} else {
			_dimension = XMLWriter.getDimensionElement(XML_DIMENSION_PROPERTY, root);
		}
		element = root.getChild(XML_TOTALSIZE_PROPERTY);
		if (element == null) { // cette fonctionnalit a t ajoute depuis la version 1.1.0 => compatibilit ascendante avec les fichiers des versions prcdentes
			_totalSize = DEFAULT_TOTALSIZE;
		} else {
			_totalSize = XMLWriter.getDimensionElement(XML_TOTALSIZE_PROPERTY, root);
		}
		_dimension = XMLWriter.getDimensionElement(XML_DIMENSION_PROPERTY, root);
		Attribute attribute = root.getAttribute(XML_STRETCH_PROPERTY);
		if (attribute == null) { // cette fonctionnalit a t ajoute depuis la version 1.1.0 => compatibilit ascendante avec les fichiers des versions prcdentes
			_stretch = false;
		} else {
			_stretch = Boolean.parseBoolean(attribute.getValue());
		}
		attribute = root.getAttribute(XML_AUTOSIZEFONT_PROPERTY);
		if (attribute == null) { // cette fonctionnalit a t ajoute depuis la version 1.1.0 => compatibilit ascendante avec les fichiers des versions prcdentes
			_autoSizeFont = false;
		} else {
			_autoSizeFont = Boolean.parseBoolean(attribute.getValue());
		}
	}

	/**
	 * @param file the file used to save the WordArt
	 * @throws IOException
	 */
	public void XMLsave(File file) throws IOException {
		final Element root = new Element(XML_DKSWORDART_PROPERTY);
		XMLsave(root);
		XMLWriter.saveDocument(file, root);
	}

	/**
	 * @param root the DOM root Element used to save the WordArt
	 * @throws IOException
	 */
	public void XMLsave(Element root) {
		final Element border = new Element(XML_BORDER_PROPERTY);
		_border.XMLsave(border);
		root.addContent(border);
		final Element shadow = new Element(XML_SHADOW_PROPERTY);
		_shadow.XMLsave(shadow);
		root.addContent(shadow);
		final Element texture = new Element(XML_TEXTURE_PROPERTY);
		_texture.XMLsave(texture);
		root.addContent(texture);
		final Element warp = new Element(XML_WARP_PROPERTY);
		_warp.XMLsave(warp);
		root.addContent(warp);
		root.addContent(XMLWriter.createFontElement(XML_FONT_PROPERTY, _font));
		root.addContent(XMLWriter.createColorElement(XML_BACKGROUND_COLOR_PROPERTY, _backgroundColor));
		root.addContent(XMLWriter.createDimensionElement(XML_DIMENSION_PROPERTY, _dimension));
		root.addContent(XMLWriter.createDimensionElement(XML_TOTALSIZE_PROPERTY, _totalSize));
		root.setAttribute(XML_TEXT_PROPERTY, _text);
		root.setAttribute(XML_STRETCH_PROPERTY, Boolean.valueOf(_stretch).toString());
		root.setAttribute(XML_AUTOSIZEFONT_PROPERTY, Boolean.valueOf(_autoSizeFont).toString());
	}

	protected class BorderChange implements ChangeListener {
		public void stateChanged(ChangeEvent arg0) {
			_borderChanged = true;
		}
	}

	protected class TextureChange implements ChangeListener {
		public void stateChanged(ChangeEvent arg0) {
			_textureChanged = true;
		}
	}

	protected class WarpChange implements ChangeListener {
		public void stateChanged(ChangeEvent arg0) {
			_warpChanged = true;
		}
	}

	protected class ShadowChange implements ChangeListener {
		public void stateChanged(ChangeEvent arg0) {
			_shadowChanged = true;
		}
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + (_autoSizeFont ? 1231 : 1237);
		result = prime
				* result
				+ ((_backgroundColor == null) ? 0 : _backgroundColor.hashCode());
		result = prime * result + ((_border == null) ? 0 : _border.hashCode());
		result = prime * result
				+ ((_dimension == null) ? 0 : _dimension.hashCode());
		result = prime * result + ((_font == null) ? 0 : _font.hashCode());
		result = prime * result + ((_shadow == null) ? 0 : _shadow.hashCode());
		result = prime * result + (_stretch ? 1231 : 1237);
		result = prime * result + ((_text == null) ? 0 : _text.hashCode());
		result = prime * result
				+ ((_texture == null) ? 0 : _texture.hashCode());
		result = prime * result
				+ ((_totalSize == null) ? 0 : _totalSize.hashCode());
		result = prime * result + ((_warp == null) ? 0 : _warp.hashCode());
		return result;
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		final CWordArt other = (CWordArt) obj;
		if (_autoSizeFont != other._autoSizeFont)
			return false;
		if (_backgroundColor == null) {
			if (other._backgroundColor != null)
				return false;
		} else if (!_backgroundColor.equals(other._backgroundColor))
			return false;
		if (_border == null) {
			if (other._border != null)
				return false;
		} else if (!_border.equals(other._border))
			return false;
		if (_dimension == null) {
			if (other._dimension != null)
				return false;
		} else if (!_dimension.equals(other._dimension))
			return false;
		if (_font == null) {
			if (other._font != null)
				return false;
		} else if (!_font.equals(other._font))
			return false;
		if (_shadow == null) {
			if (other._shadow != null)
				return false;
		} else if (!_shadow.equals(other._shadow))
			return false;
		if (_stretch != other._stretch)
			return false;
		if (_text == null) {
			if (other._text != null)
				return false;
		} else if (!_text.equals(other._text))
			return false;
		if (_texture == null) {
			if (other._texture != null)
				return false;
		} else if (!_texture.equals(other._texture))
			return false;
		if (_totalSize == null) {
			if (other._totalSize != null)
				return false;
		} else if (!_totalSize.equals(other._totalSize))
			return false;
		if (_warp == null) {
			if (other._warp != null)
				return false;
		} else if (!_warp.equals(other._warp))
			return false;
		return true;
	}

	public String toString() {
		return "dks.src.wordArt.CWordArt[text=" + _text + ",font=" + _font + ",backgroundColor=" + _backgroundColor + ",dimension=" + _dimension + ",totalSize=" + _totalSize + ",stretch=" + _stretch + ",autoSizeFont=" + _autoSizeFont + ",border=" + _border + ",shadow=" + _shadow + ",texture=" + _texture + ",warp=" + _warp + "]";
	}

	protected Object readResolve() throws ObjectStreamException {
		_borderChangeListener = new BorderChange();
		if (_border != null) {
			_border.addChangeListener(_borderChangeListener);
		}
		_textureChangeListener = new TextureChange();
		if (_texture != null) {
			_texture.addChangeListener(_textureChangeListener);
		}
		_warpChangeListener = new WarpChange();
		if (_warp != null) {
			_warp.addChangeListener(_warpChangeListener);
		}
		_shadowChangeListener = new ShadowChange();
		if (_shadow != null) {
			_shadow.addChangeListener(_shadowChangeListener);
		}
		_borderChanged = true;
		_textureChanged = true;
		_warpChanged = true;
		_shadowChanged = true;
		_wordArtChanged = true;
		return this;
	}
}
