/**
 * CHandler for JamochaMUD will act as our "proxy" between
 * the core of JamochaMUD and our different sockets, if we
 * have simultaneous connections to other MU*s
 * $Id: CHandler.java,v 1.30 2004/07/28 03:13:04 jeffnik Exp $
 */

/* JamochaMUD, a Muck/Mud client program
 * Copyright (C) 1998-2005 Jeff Robinson
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

package anecho.JamochaMUD;

import java.awt.Color;
import java.awt.Component;
// import java.awt.Container;
import java.awt.Font;

import java.util.Vector;

import anecho.gui.JMText;
import anecho.gui.JMSwingText;
import anecho.JamochaMUD.legacy.MuckConnAWT;

/** To allow an easier time of dealing with multiple connections,
 * any input or output from JamochaMUD will pass through the CHandler
 * class.  This class will then determine which MU* the outbound
 * traffic will go out (eg. the &quot;active" MU* if from the DataIn
 * window) or the appropriate window to send incoming traffic
 * based on the socket it is received from.  CHandler now extends
 * the Container class so that it can act as it's own autonomous
 * "widget" coordinating views of MU*s and their actual connections.
 *
 * CHandler will &quot;hold" the actual MuSockets but for considering
 * which connection is visible/active, we will always ask our display
 * object, be it the JMTextPanel or the JTabbedPane.
 */

public class CHandler {
    
    private Vector connections;   // A list of our active sockets/MUs
    // private int activeMU;               // which is our active MU*
    private JMConfig settings;          // a pointer to our settings
    // private MuckMain mainWindow;
    private boolean useSwing;   // Whether we are using Swing or not
    // private Component textPanel;
    
    private static final boolean DEBUG = false;
    
    /**
     * Our constructor.  Since this should only be called once
     * per invocation of JamochaMUD, we will set up the &quot;frames"
     * for our MU* client.
     * @param mainWindow The variable representing JamochaMUD's main frame.
     */
//     public CHandler(JMConfig mainSettings, MuckMain mainWindow) {
    public CHandler(MuckMain mainWindow) {
        // this.settings = mainSettings;
        settings = JMConfig.getInstance();
        // this.mainWindow = mainWindow;
        
        connections = new Vector(0, 1);
        
        // Since this has just been called, chances are we're
        // not connected to anything.  So let's start it off!
        // listConnections();
        // connectToNewMU();
        
        // We'll create a display object based on whether we use Swing or not
        /*
        if (settings.getJMboolean(JMConfig.USESWING)) {
            System.out.println("Swing being used by CHandler.");
            textPanel = new javax.swing.JTabbedPane();
        } else {
            System.out.println("We're not using swing in CHandler.");
            // anecho.JamochaMUD.legacy.JMTabPanel textPanel = new anecho.JamochaMUD.legacy.JMTabPanel();
            textPanel = new anecho.JamochaMUD.legacy.JMTabPanel();
            }
         */
        // textPanel = settings.getJMObject(JMConfig.TEXTPANEL);
        if (settings.getJMboolean(JMConfig.USESWING)) {
            useSwing = true;
        } else {
            useSwing = false;
        }
    }
    
    /**
     * Return the layout object that contains our MU* views
     * @return Returns a component that houses JamochaMUD's output display.
     */
    public Component getTextObject() {
        Component retComp = (Component)null;
        
        if (useSwing) {
            // return (javax.swing.JTabbedPane)settings.getJMObject(JMConfig.TEXTPANEL);
            retComp = (javax.swing.JTabbedPane)settings.getJMObject(JMConfig.TEXTPANEL);
        } else {
            // return (anecho.JamochaMUD.legacy.JMTabPanel)settings.getJMObject(JMConfig.TEXTPANEL);
            retComp = (anecho.JamochaMUD.legacy.JMTabPanel)settings.getJMObject(JMConfig.TEXTPANEL);
        }
        //System.out.println("CHandler is handing back our component: " + textPanel);
        // return textPanel;
        return retComp;
    }
    
    /**
     * Sets the active MU* (the MU* visible to the user) based on the index
     * number provided.
     * @param target The index number repesenting which MU* should be set active.
     */
    public void setActiveMU(final int target) {
        if (DEBUG) {
            System.err.println("CHandler.setActiveMU: " + target);
        }
        
        // if (target < 0) {
        // return;
        // }
        if (target > 0) {
            
            if (useSwing) {
                final javax.swing.JTabbedPane tempPanel = (javax.swing.JTabbedPane)getTextObject();
                
                try {
                    tempPanel.setSelectedIndex(target);  // Fix me!!
                } catch (Exception except) {
                    System.err.println("CHandler.setActiveMU exception: " + except);
                    return;
                }
                
                tempPanel.invalidate();
                tempPanel.validate();
                tempPanel.doLayout();
            } else {
                final anecho.JamochaMUD.legacy.JMTabPanel tempPanel = (anecho.JamochaMUD.legacy.JMTabPanel)getTextObject();
                if (DEBUG) {
                    System.err.println("CHandler.setActiveMU() going to setSelectedIndex.");
                }
                tempPanel.setSelectedIndex(target);  // Fix me!
                if (DEBUG) {
                    System.err.println("CHandler.setActiveMU() invalidating.");
                }
                tempPanel.invalidate();
                if (DEBUG) {
                    System.err.println("CHandler.setActiveMU() validating.");
                }
                tempPanel.validate();
                if (DEBUG) {
                    System.err.println("CHandler.setActiveMU() doLayout.");
                }
                tempPanel.doLayout();
            }
            
            if (DEBUG) {
                System.err.println("CHandler.setActiveMU() complete.");
            }
            
        }
    }
    
    /* Show the visible MU* */
    /* What is this one supposed to do?!  Fix Me XXX */
    // public void setVisibleMU() {
    /**
     * This method displays the last MU* connected to that is still open.
     */
    public synchronized void showLatestMU() {
        
        // if (totalConnections() < 1) {
        if (connections.size() < 1) {
            // There are no windows left to show.
            System.out.println("CHandler: No (additional) connections left to show.");
            return;
        }
        
        if (DEBUG) {
            System.err.println("Chandler.showLatestMU()");
        }
        
        if (useSwing) {
            final javax.swing.JTabbedPane tempPanel = (javax.swing.JTabbedPane)getTextObject();
            tempPanel.setSelectedIndex(tempPanel.getTabCount() - 1);
            tempPanel.invalidate();
            tempPanel.validate();
            tempPanel.doLayout();
        } else {
            final anecho.JamochaMUD.legacy.JMTabPanel tempPanel = (anecho.JamochaMUD.legacy.JMTabPanel)getTextObject();
            tempPanel.last();
            tempPanel.invalidate();
            tempPanel.validate();
            tempPanel.doLayout();
        }
        
        
    }
    
    /**
     * Add another MU* to our display object
     * @param name The human-readable name of the MU*
     * @param address The numeric or human-readable address for the MU* server
     * @param port The MU* server port to connect to
     * @param msock The MuSocket which is responsible for communicating between
     * JamochaMUD the new MU*.
     */
    public synchronized void addNewMU(final String name, final String address, final int port, final MuSocket msock) {
        if (DEBUG) {
            System.err.println("CHandler.addNewMU Adding " + name + " tab.");
        }
        
        if (useSwing) {
//            if (DEBUG) {
//                System.err.println("CHandler.addNewMU calling MuSocket.getSwingTextWindow()");
//            }
            
            final JMSwingText tempText = msock.getSwingTextWindow();
            
//            if (DEBUG) {
//                System.err.println("CHandler.addNewMU getting JTabbedPane...");
//            }
            
            final javax.swing.JTabbedPane tempPane = (javax.swing.JTabbedPane)getTextObject();
            
//            if (DEBUG) {
//                System.err.println("Adding tab " + name + " with " + tempText + " to " + tempPane + "...");
//            }
            tempPane.addTab(name, tempText);
            
//            if (DEBUG) {
//                System.err.println("CHandler.addNewMU tab added succesfully.");
//            }
            
        } else {
//            if (DEBUG) {
//                System.err.println("Getting temp JMText");
//            }
            final JMText tempText = msock.getTextWindow();
//            if (DEBUG) {
//                System.err.println("Adding tab");
//            }
            
            ((anecho.JamochaMUD.legacy.JMTabPanel)getTextObject()).addTab(name, tempText);
            
        }
        
        if (DEBUG) {
            System.err.println("showLatestMU.");
        }
        
        showLatestMU();
    }
    
    /**
     * Create a new socket and add a &quot;window" for it
     * @param name The human-readable name for the new MU*
     * @param address The human-readable or numeric address for the MU* server
     * @param port The MU* server port to connect to
     * @return Returns a new MuSocket object that will be responsible for
     * this connection.
     */
    // public synchronized void OpenSocket(String name, String address, int port) {
    public synchronized MuSocket openSocket(final String name, final String address, final int port) {
        // create a new MuSocket, and then populate it with the proper info
        if (DEBUG) {
            System.err.println("CHandler.OpenSocket: " + name + " " + address + " " + port);
        }
//        final MuSocket msock = new MuSocket(settings);
        final MuSocket msock = new MuSocket();
        
        msock.setTitle(name);
        // msock.resetTitle();
        msock.setAddress(address);
        msock.setPort(port);
        // msock.setTitle(name);
        
        // We'll set a timestamp to this MU* to differentiate incase the
        // user hooks up to the same MU* multiple times.
        msock.setTimeStamp(System.currentTimeMillis());
        
        // Set our logging option
        msock.setLogging(settings.getJMboolean(JMConfig.AUTOLOGGING));
        
        // Now add the new Socket to the connections vector
        connections.addElement(msock);
        
        // System.out.println("CHandler, connection size: " + connections.size());
        
        // activeMU = connections.size() - 1;
        // setActiveMU(connections.size() -1);     // Experimental change XXX
        
        // Add this new TextWindow to the existing main frame.
        // We're assuming, since this is the newest connection,
        // that this is the one we'll want visible
        // addNewMU(name, address, port, msock);
        
        /*
        mainWindow.addNewMU(msock);
        System.out.println("CHandler, adding new MU " + msock);
        mainWindow.setVisibleMU();  // XXX removed for trouble-shooting
         */
        
        // set the layout if it hasn't been done already.  We do this from here
        // so that we don't try to set a layout before we have at least one JMText area
        /*
        if (connections.size() == 1) {
            mainWindow.setMainLayout();
            }
         */
        
        // Now we'll start our socket-listening thread
        msock.start();
        
        // Update the connection menu on the main program?
        // Fix me XXX!
        
        // System.out.println("CHandler.openSocket returns: " + msock);
        return msock;
        
    }
    
    /** make the next MU* in the list the active MU*
     * The process that makes the call to this method
     * is responsible for updating the display appropriately
     */
    public void nextMU() {
        /*
        // Check to see if we have more than one MU, otherwise this is pointless
        if (connections.size() < 1) return;
         
        setActiveMU(activeMU + 1);
         */
        if (useSwing) {
            final javax.swing.JTabbedPane tempPanel = (javax.swing.JTabbedPane)getTextObject();
            int muNum = tempPanel.getSelectedIndex();
            muNum++;
            if (muNum > (tempPanel.getTabCount() - 1)) {
                muNum = 0;
            }
            tempPanel.setSelectedIndex(muNum);
        } else {
            final anecho.JamochaMUD.legacy.JMTabPanel tempPanel = (anecho.JamochaMUD.legacy.JMTabPanel)getTextObject();
            tempPanel.next();
        }
    }
    
    /** Make the previous MU* in the list the active MU*.
     * As with nextMU, the calling process is responsible
     * for updating the display appropriately.
     */
    public void previousMU() {
        // Don't do anything if we only have one connection
        /*
        if (connections.size() < 1) return;
         
        setActiveMU(activeMU - 1);
         */
        
        if (useSwing) {
            final javax.swing.JTabbedPane tempPanel = (javax.swing.JTabbedPane)getTextObject();
            int muNum = tempPanel.getSelectedIndex();
            muNum--;
            if (muNum < 0) {
                muNum = tempPanel.getTabCount() - 1;
            }
            tempPanel.setSelectedIndex(muNum);
        } else {
            final anecho.JamochaMUD.legacy.JMTabPanel tempPanel = (anecho.JamochaMUD.legacy.JMTabPanel)getTextObject();
            tempPanel.previous();
        }
    }
    
    /** Close the connection to the active MU*
     * @param stamp Timestamp of MU* to be closed (as confirmation)
     */
    public void closeActiveMU(final String stamp) {
        // System.out.println("CHandler.closeActiveMU...");
        final MuSocket active = getActiveMUHandle();
        // String ts = new String(active.getTimeStamp() + "");
        closeSocket(active);
    }
    
    /**
     * Remove the active MU* from our list altogether
     * @param stamp This is the time stamp used to verify that we are closing
     * the proper MU*.
     */
    public void removeActiveMU(final String stamp) {
        // Disconnect the MU* if still connected
        if (isActiveMUDConnected()) {
            closeActiveMU(stamp);
        }
        
        // "destroy" any physical entities
        // uhm... somehow.  We could rely on garbage collection but
        // I'd feel better actually knowing it was gone.
        // Fix this XXX
        
        // remove the MU* from our hashtable
        final int aMU = getActiveMUIndex();
        connections.removeElementAt(aMU);
        
        // setActiveMU(activeMU - 1);
        
    }
    
    /**
     * Remove the MU* indicated by the MuSocket and timeStamp
     * @param mSock The MuSocket to close and remove
     */
    public void removeMU(final MuSocket mSock) {
        if (mSock.isConnectionActive()) {
            if (DEBUG) {
                System.err.println("CHandler.removeMU(): Disconnecting active socket");
            }
            closeSocket(mSock);
        }
        
        // System.out.println("CHandler.removeMU: remove : " + mSock);
        int place;
        place = connections.indexOf(mSock);
        
        // Remove this from out layout as well
        if (useSwing) {
            final JMSwingText tempText = mSock.getSwingTextWindow();
            // Watch out for this one... when testing Java2  Fix Me XXX!!
            /*
            if (DEBUG) {
                System.err.println("TempText to be removed: " + tempText);
                System.err.println("Our tab to be removed: " + ((javax.swing.JTabbedPane)getTextObject()));
            }
             */
            
            final int index = ((javax.swing.JTabbedPane)getTextObject()).indexOfComponent(tempText);
            if (DEBUG) {
                System.err.println("CHandler.removeMU(): The index of our component is: " + index);
            }
            // ((javax.swing.JTabbedPane)getTextObject()).remove(tempText);
            
            ((javax.swing.JTabbedPane)getTextObject()).removeTabAt(index);
            // index = ((javax.swing.JTabbedPane)getTextObject()).getSelectedIndex();
        } else {
            final JMText tempText = mSock.getTextWindow();
            ((anecho.JamochaMUD.legacy.JMTabPanel)getTextObject()).remove(tempText, place);
            // index = ((anecho.JamochaMUD.legacy.JMTabPanel)getTextObject()).getSelectedIndex();
            
        }
        
        if (DEBUG) {
            System.err.println("CHandler.removeMU(): Component removed");
        }
        
        connections.removeElement(mSock);
        
        if (DEBUG) {
            System.err.println("CHandler.removeMU(): Completed successfully");
        }
// connections.removeElementAt(place);
        
        // System.out.println("CHandler.removeMU: activeMU is " + activeMU);
        /*
        if (activeMU >= place) {
        setActiveMU(activeMU--);
        }
         */
        // System.out.println("CHandler.removeMU: We've set our ActiveMU to " + activeMU);
    }
    
    /**
     * Close and &quot;destroy" an existing socket.
     * This allows any of our clases to shut things down
     * @param socket The socket to be closed.
     */
    public void closeSocket(final MuSocket socket) {
        /* Why was this stack trace here in the first place?
        if (DEBUG) {
            StringWriter sWriter = new StringWriter();
            new Throwable().printStackTrace(
                                            new PrintWriter( sWriter )
                                           );
            String callStack = sWriter.toString();
         
            System.err.println("Stack: " + callStack);
        }
         */
        
        // Clean up the existing socket
        socket.closeSocket();
        
        // Removed for testing purposes
        // Set the menu state
        // MuckMain main = settings.getMainWindowVariable();
        // main.setConnectionMenu();
    }
    
    /**
     * Return the 'address' for the active JMText object
     * @deprecated use JMText getActiveMUDText(String stamp)
     * @return Returns a variable representing the active MU*
     */
    public JMText getActiveMUDText() {
        final int aMU = getActiveMUIndex();
        final MuSocket msock = (MuSocket)connections.elementAt(aMU);
        return msock.getTextWindow();
    }
    
    /**
     * This returns the active JMSwingText
     * @return JMSwingText of the active MU*
     */
    public JMSwingText getActiveMUDSwingText() {
        JMSwingText retText;
        
        if (connections.size() > 0) {
            final int aMU = getActiveMUIndex();
            final MuSocket msock = (MuSocket)connections.elementAt(aMU);
            // return msock.getSwingTextWindow();
            retText = msock.getSwingTextWindow();
        } else {
            retText = (JMSwingText)null;
        }
        
        return retText;
    }
    
    /**
     * Return the active JMText component, verified with the passed timestamp
     * @param stamp A timestamp to verify the component returned
     * @return Returns a JMText component of the active MU*
     */
    public JMText getActiveMUDText(final String stamp) {
        final int aMU = getActiveMUIndex();
        final MuSocket msock = (MuSocket)connections.elementAt(aMU);
        return msock.getTextWindow();
        
    }
    
    /**
     * Return the index of the active MU
     * @return Returns the index number of the active MU*
     */
    public int getActiveMUIndex() {
        int index = 0;
        
        if (useSwing) {
            index = ((javax.swing.JTabbedPane)getTextObject()).getSelectedIndex();
        } else {
            index = ((anecho.JamochaMUD.legacy.JMTabPanel)getTextObject()).getSelectedIndex();
        }
        
        return index;
    }
    
    /**
     * Return the title of the MU based on the given index
     * @param index Index representing the MU* we are to retrieve the title of.
     * @return The title of the MU* represented by the given {@link index}
     */
    public String getTitle(final int index) {
        String title;
        
        if (index < 0 || index > (connections.size() - 1)) {
            if (DEBUG) {
                System.err.println("CHandler.getTitle out of range.");
            }
            title = "";
        } else {
            title = ((MuSocket)connections.elementAt(index)).getTitle();
        }
        return title;
    }
    
    /**
     * Return the proper MuSocket for the requested mu
     * @param muNum This variable represents the index of the MU* to retrieve
     * @return MuSocket representing the MU* specified by our {@link muNum}
     */
    public synchronized MuSocket getMUHandle(final int muNum) {
        MuSocket retSock;
        if (muNum < connections.size() && muNum > -1) {
            retSock = (MuSocket)connections.elementAt(muNum);
        } else {
            retSock = (MuSocket)null;
        }
        
        return retSock;
    }
    
    /**
     * Return the MuSocket for the currently &quot;active" MU*
     * @return The MuSocket of the currently active MU*
     */
    public synchronized MuSocket getActiveMUHandle() {
        MuSocket retSock;
        
        // Check to see if we have any connections yet
        if (connections.size() > 0) {
            /*
            if (activeMU < 0) {
                // System.out.println("CHandler.getActiveMUHandle, activeMU is less than 0.");
                // setActiveMU(0);
                activeMU = 0;
            }
            retSock = (MuSocket)connections.elementAt(activeMU);
             */
            int aMU = 0;
            if (useSwing) {
                aMU = ((javax.swing.JTabbedPane)getTextObject()).getSelectedIndex();
            } else {
                aMU = ((anecho.JamochaMUD.legacy.JMTabPanel)getTextObject()).getSelectedIndex();
            }
            
            retSock = (MuSocket)connections.elementAt(aMU);
        } else {
            retSock = (MuSocket)null;
        }
        
        return retSock;
    }
    
    /**
     * Return the connection status of the active MU*
     * @return <CODE>true</CODE> - The active MU* is connected to a server
     * <CODE>false</CODE> - The active MU* is not connected to a server
     */
    // public synchronized boolean isActiveMUDConnected() {
    public boolean isActiveMUDConnected() {
        boolean active = false;
        
        /*
        if (activeMU > connections.size() + 1) {
            activeMU = connections.size() - 1;
        }
         
        if (connections.size() > 0) {
            MuSocket msock = (MuSocket)connections.elementAt(activeMU);
            active = msock.isConnectionActive();
        }
         */
        
        int aMU = 0;
        
        if (useSwing) {
            aMU = ((javax.swing.JTabbedPane)getTextObject()).getSelectedIndex();
        } else {
            aMU = ((anecho.JamochaMUD.legacy.JMTabPanel)getTextObject()).getSelectedIndex();
        }
        
        if (aMU > connections.size() - 1) {
            if (DEBUG) {
                System.err.println("CHandler.isActiveMUConnected()");
                System.err.println("aMU: " + aMU + " and our connection size: " + connections.size());
            }
            // return active;
        } else {
            // if (aMU >= 0) {
            if (aMU >= 0 && aMU < connections.size()) {
                final MuSocket msock = (MuSocket)connections.elementAt(aMU);
                active = msock.isConnectionActive();
            }
        }
        
        return active;
    }
    
    /**
     * Send a string to the active MU*, using the
     * proper encoding method (ASCII or Unicode)
     * @param send The String to be sent to the active MU*
     */
    public void sendText(final String send) {
        // MuSocket msock = (MuSocket)connections.elementAt(activeMU);
        int aMU = 0;
        
        if (useSwing) {
            aMU = ((javax.swing.JTabbedPane)getTextObject()).getSelectedIndex();
        } else {
            aMU = ((anecho.JamochaMUD.legacy.JMTabPanel)getTextObject()).getSelectedIndex();
        }
        
        if (aMU > -1 && aMU < connections.size()) {
            final MuSocket msock = (MuSocket)connections.elementAt(aMU);
            
            msock.sendText(send);
        } else {
            if (DEBUG) {
                System.err.println("CHandler cannot send text as aMU " + aMU + " is not within the size of our connections " + connections.size());
            }
        }
    }
    
    /**
     * Change the font face and size on all the connections
     * @param newPal This array represents the colour palette that is to be used
     * @param fontFace The Font to use for the display.
     * @param fgColour The foreground colour to use
     * @param bgColour The background colour to use
     */
    public void setAllAttribs(final Font fontFace, final Color fgColour, final Color bgColour, final Color[] newPal) {
        MuSocket msock;       // our temporary socket
        JMText text;          // our temporary text object
        JMSwingText sText;
        
        final int total = connections.size();
        
        if (total < 1) {
            // There are no active MU*s
            return;
        }
        
        for (int i = 0; i < total; i++) {
            // Loop through our connections and change the fonts
            msock = (MuSocket)connections.elementAt(i);
            if (useSwing) {
                sText = msock.getSwingTextWindow();
                if (fontFace != null) {
                    sText.setFont(fontFace);
                }
                if (fgColour != null) {
                    sText.setForeground(fgColour);
                    sText.setBackground(bgColour);
                }
                if (newPal != null) {
                    try {
                        sText.setPalette(newPal);
                    } catch (Exception exc) {
                        if (DEBUG) {
                            System.err.println("Exception settings new custom palette from CHandler.");
                        }
                    }
                }
            } else {
                text = msock.getTextWindow();
                if (fontFace != null) {
                    text.setFont(fontFace);
                }
                if (fgColour != null) {
                    text.setForeground(fgColour);
                    text.setBackground(bgColour);
                }
                /*
                 if (fontFace != null) {
                 msock.setFont(fontFace);
                 }
                 if (fgColour != null) {
                 msock.setForeground(fgColour);
                 msock.setBackground(bgColour);
                 }
                 */
            }
        }
    }
    
    /**
     * Activate or deactivate logging in all connections
     * @param state <CODE>true</CODE> - enable logging
     * <CODE>false</CODE> - disable logging
     */
    public void setLogging(final boolean state) {
        MuSocket msock;       // our temporary socket
        final int total = connections.size();
        
        if (total < 1) {
            // There are no active MU*s
            return;
        }
        
        for (int i = 0; i < total; i++) {
            // Loop through our connections and change the fonts
            msock = (MuSocket)connections.elementAt(i);
            msock.setLogging(state);
        }
    }
    
    /**
     * Set the foreground and background colours for all the MU*s.
     * @param fgColour The foreground colour to use
     * @param bgColour The background colour to use
     */
    public void setAllColours(final Color fgColour, final Color bgColour) {
        setAllAttribs(null, fgColour, bgColour, null);
    }
    
    /**
     * This method sets the custom colours to be used on all our MU* windows.
     * @param newPal An array containing our new Colors.
     */
    public void setCustomPalette(final Color[] newPal) {
        setAllAttribs(null, null, null, newPal);
    }
    
    /**
     * Set the Font for all active MU*s
     * @param newStyle The Font style to use
     */
    public void setAllFonts(final Font newStyle) {
        setAllAttribs(newStyle, null, null, null);
    }
    
    // public synchronized void connectToNewMU() {
    // connectToNewMU(null, null, 0);
    // }
    
    /** Set-up a connection to a new MU* */
    public synchronized void connectToNewMU() {
        // Show our list of connections (the MuckConnector)
        if (DEBUG) {
            System.err.println("CHandler.connectToNewMU starting.");
        }
        
        // Check first to make certain we don't already have a connection dialogue
        if (!settings.getJMboolean(JMConfig.MUCKCONNVISIBLE)) {
            if (DEBUG) {
                System.err.println("CHandler.connectToNewMU creating new MuckConn.");
            }
            
            String name, address;
            int port;
            
            if (settings.getJMboolean(JMConfig.USESWING)) {
                final MuckConn muChoice = new MuckConn(settings);
                
                muChoice.setVisible(true);
                
                // Gather information about our chosen connection
                name = muChoice.getName();
                address = muChoice.getAddress();
                port = muChoice.getPort();
                
//                if (muChoice.getName() == null) {
//                    // A MU* was not selected
//                    if (DEBUG) {
//                        System.err.println("CHandler.connectToNewMU, no MU* chosen.");
//                    }
//                    return;
//                }
                
            } else {
                final MuckConnAWT muChoice = new MuckConnAWT(settings);
                
                muChoice.setVisible(true);
                
//                if (muChoice.getName() == null) {
//                    // A MU* was not selected
//                    if (DEBUG) {
//                        System.err.println("CHandler.connectToNewMU, no MU* chosen.");
//                    }
//                    return;
//                }
                
                
                // Gather information about our chosen connection
                name = muChoice.getName();
                address = muChoice.getAddress();
                port = muChoice.getPort();
                
                // Open our new connection
            }
            
            if (DEBUG) {
                System.err.println("CHandler.connectToNewMU calling OpenSocket for: " + name + " " + address + " " + port);
            }
            
            // Open a new socket if a MU* has been selected
            if (name != null) {
                final MuSocket msock = openSocket(name, address, port);
                
                if (DEBUG) {
                    System.err.println("CHandler.connectToNewMU calling addNewMU");
                }
                
                addNewMU(name, address, port, msock);
            }
        }
        
    }
    
    /**
     * Get the total number of MU*s open regardless of their connection status
     * @return The number of existing MuSockets
     */
    public synchronized int totalConnections() {
        int retSize = 0;
        
        if (DEBUG) {
            System.err.println("CHandler.totalConnections() getting connection size.");
        }
        
        if (connections.size() > 0) {
            retSize = connections.size();
        }
        // public int totalConnections() {
        if (DEBUG) {
            System.err.println("CHandler.totalConnections() returns: " + retSize);
        }
        
        // return connections.size();
        return retSize;
    }
    
    /** Allow another program to set which of our MU*s are active */
    /*
    public synchronized void setActiveMU(int origNum) {
        int muNum = origNum;
        MuSocket aMH;
     
        // Check to see if we have to "wrap" the list around
        if (muNum < 0 || muNum > (connections.size() -1)) {
            if (connections.size() > 0) {
                if (muNum < 0) {
                    // muNum = 0;
                    muNum = (connections.size() - 1);
                } else {
                    muNum = 0;
                    // muNum = (connections.size() - 1);
                }
            }
        }
     
        // Tell our old MU* it is no longer active
        // System.out.println("CHandler.setActiveMU...");
        aMH = getActiveMUHandle();
        if (aMH != null) {
            aMH.setActiveMU(false);
        }
     
        // Assign the new active MU*
        activeMU = muNum;
     
        // Set the new MU* as being active
        // System.out.println("CHandler.setActiveMU second instance...");
        aMH = getActiveMUHandle();
        if (aMH != null) {
            aMH.setActiveMU(true);
            }
     
            }
     */
    
    /**
     * Determines if the given MU* is the currently active MU*.
     * @return <CODE>true</CODE> - the provided MU* is the active MU*
     * <CODE>false</CODE> - the provided MU* is not the active MU*
     * @param mSock The MuSocket to check to see if it is current the active MU*.
     */
    public boolean isActiveMU(final MuSocket mSock) {
        boolean active = false;
        
        // System.out.println("CHandler.isActiveMU...");
        // if (getActiveMUHandle() == mSock) {
        if (getActiveMUHandle().equals(mSock)) {
            active = true;
        }
        
        return active;
    }
    
    
    /**
     * Return the title of the active MU*
     * @return Return a String of the title of the active MU*.
     */
    public synchronized String getActiveTitle() {
        String title;
        
        if (connections.size() < 1) {
            // We have no connections, thus no title
            title = "";
        } else {
            // MuSocket msock = (MuSocket)connections.elementAt(activeMU);
            // title = msock.getTitle();
            final int aMU = getActiveMUIndex();
            title = getTitle(aMU);
        }
        if (DEBUG) {
            System.out.println("getActiveTitle returns: " + title);
        }
        return title;
    }
    
    /**
     * Change the drawing type for the JMText... either
     * single buffered (false) or double buffered (true)
     * @param state <CODE>true</CODE> - set the display to double buffer
     * <CODE>false</CODE> - set the display to work without buffering
     */
    public void setDoubleBuffer(final boolean state) {
        MuSocket msock;
        for (int i = 0; i < connections.size(); i++) {
            msock = (MuSocket)connections.elementAt(i);
            (msock.getTextWindow()).setDoubleBuffer(state);
            // msock.setDoubleBuffer(state);
        }
    }
    
    /**
     * Change between aliased and antialised text in our
     * output windows.
     * @param state <CODE>true</CODE> - enable antialiasing
     * <CODE>false</CODE> - disalbe antialiasing
     */
    public void setAntiAliasing(final boolean state) {
        MuSocket msock;
        for (int i = 0; i < connections.size(); i++) {
            if (DEBUG) {
                System.out.println("CHandler.setAntiAliasing setting connection " + i + " to " + state);
            }
            msock = (MuSocket)connections.elementAt(i);
            (msock.getSwingTextWindow()).setAntiAliasing(state);
            // msock.setDoubleBuffer(state);
        }
    }
    
    /**
     * Set the MU* displays to operating in either "low colour" (8 colours,
     * normal and bold) or "high colour" (16 colours).
     * @param state <CODE>true</CODE> - set the display to low colour mode
     * <CODE>false</CODE> - set display to high colour mode
     */
    public void setLowColour(final boolean state) {
        MuSocket msock;
        for (int i = 0; i < connections.size(); i++) {
            if (DEBUG) {
                System.out.println("CHandler.setLowColour setting connection " + i + " to " + state);
            }
            msock = (MuSocket)connections.elementAt(i);
            (msock.getSwingTextWindow()).setBoldNotBright(state);
            
        }
    }
    
    /**
     * Return the ECHO state for the active MU*
     * @return <CODE>true</CODE> - the active MU* echo mode is enabled
     * <CODE>false</CODE> - the active MU* is not in echo mode
     */
    public boolean getActiveMUEchoState() {
        // System.out.println("CHandler.getActiveMUEchoState...");
        final MuSocket active = getActiveMUHandle();
        return active.getEchoState();
    }
    
    /** Reconnect to the currently active (visible) MU* */
    public synchronized void reconnectToMU() {
        
        final MuSocket handle = getActiveMUHandle();
        
        if (handle == null) {
            // No MU*s have previously been connected to, so we can't reconnect
            return;
        }
        
        final String name = handle.getMUName();
        final String address = handle.getAddress();
        final int port = handle.getPort();
        
        if (DEBUG) {
            System.err.println("Chandler.reconnectToMU activeMU Handle is: " + handle);
            System.err.println("CHandler.reconnectToMU attempting connection to: " + name + " " + address + " " + port);
            System.err.println("CHandler.reconnectToMU calling OpenSocket");
        }
        
        final MuSocket msock = openSocket(name, address, port);
        
        if (DEBUG) {
            System.err.println("CHandler.reconnectToMU Adding the new MU*");
        }
        
        addNewMU(name, address, port, msock);
        
        if (DEBUG) {
            System.err.println("CHandler.reconnectToMU removing old MU*.");
        }
        
        removeMU(handle);
        
        if (DEBUG) {
            System.err.println("CHandler.reconnectToMU completed.");
        }
        
    }
}
