/**
 * DataIn, text input window
 * $Id: DataIn.java,v 1.30 2004/07/28 03:13:05 jeffnik Exp $
 */

/* JamochaMUD, a Muck/Mud client program
 * Copyright (C) 1998-2005  Jeff Robinson
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

package anecho.JamochaMUD;

import java.awt.*;
import java.awt.event.*;

import java.util.Vector;

import anecho.extranet.event.TelnetEvent;
import anecho.extranet.event.TelnetEventListener;

import anecho.gui.SyncFrame;
import anecho.gui.ResReader;

/**
 * This is the input window where the user may type to send info
 * to the MUCK/MUD.  In addition, if features scrolling of long messages
 * and a 10 command 'recall' feature implemented by a right mouse-click.
 * @version $Id: DataIn.java,v 1.30 2004/07/28 03:13:05 jeffnik Exp $
 * @author Jeff Robinson
 */
public class DataIn extends SyncFrame implements ActionListener, KeyListener, MouseListener, WindowListener, TelnetEventListener{
    
    private PopupMenu backTrack;
    // private String typedLine; //, bCPopup[];
    private String backComm[];
    // private TextArea dataText;
    private Component dataText, dataInPane;
    private JMConfig settings;
    private CHandler connHandler;
    private JMTFCommands tfCommand;
    private JMTFKeys tfKeys;
    private int limit = 10; // Number of history entries allowed
    private boolean useSwing;
    // private boolean antiAlias = false;
    private static final boolean DEBUG = false;
    private ActionListener pml; // Pop-up menu listener
    private Vector historyV;
    
    /**
     * The constructor for DataIn, producing a new data input area.
     */
    // public DataIn(JMConfig mainSettings){
    public DataIn(){
        
        super("JamochaMUD");
        
        // this.settings = mainSettings;
        settings = JMConfig.getInstance();
        this.connHandler = settings.getConnectionHandler();
        
        historyV = new Vector();
        
        // tfCommand = new JMTFCommands(mainSettings);
//        tfCommand = new JMTFCommands(settings);
        tfCommand = new JMTFCommands();
        tfKeys = JMTFKeys.getInstance();        // Trying out the JMTFKeys singleton
        
        addWindowListener(this);
        // pml = new ActionListener();  // An ActionListener to be used with the pop-up menu
        
        // Set Gridbag layout
        GridBagConstraints constraints = new GridBagConstraints();
        
        backTrack = new PopupMenu();
        backComm = new String[limit];
        
        // Add the text field to the box.
        constraints.gridwidth = 1;
        constraints.gridheight = 1;
        constraints.gridx = 0;
        constraints.gridy = 0;
        constraints.fill = GridBagConstraints.BOTH;
        
        useSwing = settings.getJMboolean(JMConfig.USESWINGENTRY);
        
        if (useSwing) {
            if (DEBUG) {
                System.out.println("Swing-based Text entry initialising...");
            }
            // Use a Swing component instead
            // dataText = new anecho.gui.JMSwingEntry("", 3, 70);
            dataText = new anecho.gui.JMSwingEntry("", 70, 3);
            ((anecho.gui.JMSwingEntry)dataText).setLineWrap(true);
            ((anecho.gui.JMSwingEntry)dataText).addKeyListener(this);
            // ((anecho.gui.JMSwingEntry)dataText).add(backTrack); 		// Adds PopupMenu to data entry area
            ((anecho.gui.JMSwingEntry)dataText).addMouseListener(this);	// Adds MouseListener to data entry area
            ((anecho.gui.JMSwingEntry)dataText).setRequestFocusEnabled(true);
            ((anecho.gui.JMSwingEntry)dataText).setAntiAliasing(settings.getJMboolean(JMConfig.ANTIALIAS));
            // ((anecho.gui.JMSwingEntry)dataText).setCaretColor(settings.getJMColor(settings.BACKGROUNDCOLOUR));
            
            dataInPane = new javax.swing.JScrollPane((anecho.gui.JMSwingEntry)dataText);
            ((javax.swing.JScrollPane)dataInPane).setVerticalScrollBarPolicy(javax.swing.JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED);
            ((javax.swing.JScrollPane)dataInPane).setHorizontalScrollBarPolicy(javax.swing.JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
            // ((anecho.gui.JMSwingEntry)dataText).setRows(3);
            add((javax.swing.JScrollPane)dataInPane);
            if (DEBUG) {
                System.err.println("Done.");
            }
        } else {
            // Standard AWT TextArea
            dataText = (Component)new TextArea("", 3, 70, TextArea.SCROLLBARS_VERTICAL_ONLY);
            ((TextArea)dataText).addKeyListener(this);
            ((TextArea)dataText).add(backTrack); 		// Adds PopupMenu to data entry area
            ((TextArea)dataText).addMouseListener(this);	// Adds MouseListener to data entry area
            add((TextArea)dataText);
        }
        
        backTrack.addActionListener(this);
        pack();
        
        // Sure, this is a bit of work, but we so love our icons, don't we?
        MediaTracker tracker = new MediaTracker(this);
        Image jMUDImage = Toolkit.getDefaultToolkit().getImage("kehza.gif");
        tracker.addImage(jMUDImage, 0);
        try {
            tracker.waitForAll();
        } catch (Exception exc) {
            System.out.println("Icon load error.  Exception " + exc);
        }
        this.setIconImage(jMUDImage);
        
        // Display macroBar
        dataText.setForeground(settings.getJMColor(settings.FOREGROUNDCOLOUR));
        dataText.setBackground(settings.getJMColor(settings.BACKGROUNDCOLOUR));
        dataText.setFont(settings.getJMFont(settings.FONTFACE));
    }
    
    /**
     *
     * @param event
     */
    public void keyPressed(final KeyEvent event){
        
        final int arg = event.getKeyCode();
        
        if (arg == KeyEvent.VK_ENTER) {
            if (useSwing) {
                jMSendText(((anecho.gui.JMSwingEntry)dataText).getText());
            } else {
                jMSendText(((TextArea)dataText).getText());
            }
            event.consume();
        }
        
        if (arg == KeyEvent.VK_ALT && settings.getJMboolean(JMConfig.ALTFOCUS) && settings.getJMboolean(JMConfig.SPLITVIEW)) {
            // Check to see if the user wants to have to ALT key focus on the menu
            // if (settings.getJMboolean(JMConfig.ALTFOCUS) && settings.getJMboolean(JMConfig.SPLITVIEW)) {
                (settings.getJMFrame(JMConfig.MUCKMAINFRAME)).requestFocus();
            // }
        }
        
        /*
         if (arg == KeyEvent.VK_UP) {
         // Check to see if the cursor is allowed to move up
         if (useSwing) {
         if (((TextArea)dataText).getCaretPosition() <= ((TextArea)dataText).getColumns()) {
         // Caret is on first line, can't move up
         event.consume();
         }
         } else {
         if (((anecho.gui.JMSwingEntry)dataText).getCaretPosition() <= ((anecho.gui.JMSwingEntry)dataText).getColumns()) {
         event.consume();
         }
         }
         }
         */
        
//        if ((arg == KeyEvent.VK_X) && (event.isControlDown())) {
//            // 'Supervisor' key was called.
//            // Minimize all the windows
//            hideFramesQuick();
//        }
//
//        if ((arg == KeyEvent.VK_Z) && (event.isControlDown())) {
//            // restore minimized windows
//            restoreMinimizedFrames();
//
//        }
        
        if (event.isShiftDown()) {
            if (arg == KeyEvent.VK_PAGE_UP) {
                // Scroll the main window up one "page"
                if (DEBUG) {
                    System.err.println("DataIn.java: Scroll main window up one page.");
                }
                if (useSwing) {
                    final anecho.gui.JMSwingText text = connHandler.getActiveMUDSwingText();
                    text.movePage(anecho.gui.JMSwingText.PAGEUP);
                } else {
                    if (DEBUG) {
                        System.err.println("A scrolling method has not yet been implemented for AWT");
                    }
                    // Fix Me XXX
                    // Add method for handling AWT event
                }
                event.consume();
            }
            
            if (arg == KeyEvent.VK_PAGE_DOWN) {
                // Scroll the main window down one "page"
                if (DEBUG) {
                    System.err.println("DataIn.java: Scroll main window down one page.");
                }
                if (useSwing) {
                    final anecho.gui.JMSwingText text = connHandler.getActiveMUDSwingText();
                    text.movePage(anecho.gui.JMSwingText.PAGEDOWN);
                } else {
                    if (DEBUG) {
                        System.err.println("A scrolling method has not been added for AWT");
                    }
                    // Fix Me XXX - need to add scrolling method for AWT
                }
                event.consume();
            }
        }
        
        if (event.isControlDown()) {
            
            // Check for any "editor" controls, such as CTRL+SHIFT+ARROW
//            if (arg == KeyEvent.VK_LEFT || arg == KeyEvent.VK_RIGHT) {
            // Nothing to see here, let the system handle it!
//                if (!useSwing) {
//                    event.consume();
//                } else {
//                    return;
//                }
//                return;
            
//            } else {
            
            // if ((arg == KeyEvent.VK_X) && (event.isControlDown())) {
            if (arg == KeyEvent.VK_X) {
                // 'Supervisor' key was called.
                // Minimize all the windows
                hideFramesQuick();
            }
            
            // if ((arg == KeyEvent.VK_Z) && (event.isControlDown())) {
            if (arg == KeyEvent.VK_Z) {
                // restore minimized windows
                restoreMinimizedFrames();
                
            }
            
            if (arg != KeyEvent.VK_LEFT && arg != KeyEvent.VK_RIGHT) {
                
                // First, check to see if this is for TinyFugue emulation
//                if (settings.getJMboolean(JMConfig.TFKEYEMU) && tfKeys.jmTFKeyStroke(settings, event.getKeyCode())) {
                if (settings.getJMboolean(JMConfig.TFKEYEMU) && tfKeys.jmTFKeyStroke(event.getKeyCode())) {
                    event.consume();
                }
                
                // We'll send this to MuckMain, to see if
                // it is supposed to be a menu shortcut
                // MuckMain tempMain = settings.getMainWindowVariable();
                if (settings.getJMboolean(JMConfig.ALTFOCUS) && settings.getJMboolean(JMConfig.SPLITVIEW)) {
                    final Frame tempMain = settings.getJMFrame(JMConfig.MUCKMAINFRAME);
                    tempMain.dispatchEvent(event);
                }
                
            }
        }
        
        // return;
    }
    
    /**
     *
     * @param event
     */
    public void keyReleased(final KeyEvent event){}
    
    /**
     *
     * @param event
     */
    public void keyTyped(final KeyEvent event){
        // We must make one exception, if the user is using TinyFugue emulation
        // so that we do not spool out text if they're releasing the 'pause text' key
        // if (settings.getTFKeyEmu() && event.isControlDown()) {
        if (settings.getJMboolean(settings.TFKEYEMU) && event.isControlDown()) {
            event.consume();
        } else {
            checkPause();
        }
    }
    
    /**
     *
     * @param event
     */
    public void actionPerformed(final ActionEvent event){
        if (DEBUG) {
            System.err.println("DataIn received ActionEvent " + event);
        }
        
        // String arg = event.getActionCommand();
        
        final String cmd = event.getActionCommand();
        
        int cmdInt = 0;
        
        try {
            cmdInt = Integer.parseInt(cmd);
            if (DEBUG) {
                System.err.println("Chosen action command is " + cmd);
            }
            jMSetFromScrollBack(cmdInt);
        } catch (Exception e) {
            if (DEBUG) {
                System.err.println("Problem getting action command value.");
            }
        }
        
        // return;
    }
    
    // Check for mouse actions
    
    /**
     *
     * @param evt
     */
    public void mousePressed(final MouseEvent evt) {
        checkPopup(evt) ;
    }
    
    /**
     *
     * @param evt
     */
    public void mouseReleased(final MouseEvent evt) {
        checkPopup(evt) ;
    }
    
    /**
     *
     * @param evt
     */
    public void mouseClicked(final MouseEvent evt) {
        // This spools out any paused text
        checkPause();
    }
    
    /**
     *
     * @param evt
     */
    public void mouseEntered(final MouseEvent evt) {
    }
    
    /**
     *
     * @param evt
     */
    public void mouseExited(final MouseEvent evt) {
    }
    
    /**
     * Check for PopupTrigger to display command history
     * @param mouse 
     */
    void checkPopup(final MouseEvent mouse) {
        if (mouse.isPopupTrigger()) {
            // showPopup(this, mouse.getX(), mouse.getY());
            // showPopup(dataText, mouse.getX(), mouse.getY());
            if (DEBUG) {
                System.err.println("Our original component is " + mouse.getComponent());
            }
            
            showPopup(mouse.getComponent(), mouse.getX(), mouse.getY());
            mouse.consume();
        }
    }
    
    /**
     * Display the pop-up menu
     * @param origin The Component that originally received the mouse click.
     * @param xPos The x-position of the mouse within the origin component
     * @param yPos The y-position of the mouse within the origin component
     */
    public void showPopup(final Component origin, final int xPos, final int yPos) {
        Component objOrg = origin;
        int xaxis = xPos;
        int yaxis = yPos;
        
        if (DEBUG) {
            System.err.println("DataIn.ShowPopup: origin object: " + objOrg);
        }
        
        // if xPos and yPos come in as -1, then we set the popup location from just
        // the frame as this method was probably called by an outside class
        if (origin == null && xaxis == -1 && yaxis == -1) {
            if (DEBUG) {
                System.err.println("DataIn.ShowPopup: Our axi are both -1.");
            }
            // objOrg = this;
            objOrg = dataText;
            xaxis = (int)(this.getSize().width / 2);
            yaxis = (int)(this.getSize().height / 2);
            
            if (DEBUG) {
                System.err.println("Our new objOrg is " + objOrg);
                System.err.println("xaxis: " + xaxis);
                System.err.println("yaxis: " + yaxis);
            }
        }
        
        if (useSwing) {
            javax.swing.JMenuItem tempItem;
            final javax.swing.JPopupMenu jpm = new javax.swing.JPopupMenu("Command history");
            String tempName;
//            for (int i = 0; i < backComm.length; i++) {
//                if (backComm[i] != null) {
//                    if (backComm[i].length() > 30) {
//                        // jpm.add(backComm[i].substring(0, 30) + "...");
//                        tempName = backComm[i].substring(0, 30) + "...";
//                    } else {
//                        // jpm.add(backComm[i]);
//                        tempName = backComm[i];
//                    }
//
//                    if (DEBUG) {
//                        System.err.println("Adding item: " + tempName);
//                    }
//
//                    tempItem = new javax.swing.JMenuItem(tempName);
//                    tempItem.addActionListener(pml);
//                    jpm.add(tempItem);
//                } else {
//                    break;
//                }
//            }
            for (int i = 0; i < historyV.size(); i++) {
                tempName = historyV.elementAt(i).toString();
                tempItem = new javax.swing.JMenuItem(tempName);
                // tempItem.addActionListener(pml);
                tempItem.addActionListener(this);
                tempItem.setActionCommand(i + "");
                jpm.add(tempItem);
            }
            
            // jpm.addPopupMenuListener(javax.swing.event.PopupMenuListener pml);
            // jpm.pack();
            jpm.show(objOrg, xaxis, yaxis);
            
        } else {
            backTrack.removeAll();
            
//            for (int i = 0; i < backComm.length; i++) {
//                if (backComm[i] != null) {
//                    if (backComm[i].length() > 30) {
//                        backTrack.add(backComm[i].substring(0, 30) + "...");
//                    } else {
//                        backTrack.add(backComm[i]);
//                    }
//                } else {
//                    break;
//                }
//            }
            String tempName;
            java.awt.MenuItem tempItem;
            
            for (int i = 0; i < historyV.size(); i++) {
                tempName = historyV.elementAt(i).toString();
                tempItem = new java.awt.MenuItem(tempName);
                tempItem.addActionListener(pml);
                tempItem.setActionCommand(i + "");
                backTrack.add(tempItem);
            }
            
            backTrack.show(objOrg, xaxis, yaxis);
        }
    }
    
    /**
     * Bring the frame into focus so the user may type
     * @param event 
     */
    public void windowActivated(final WindowEvent event) {
        // First we insert a 6 millisecond kludge to make
        // up for the requestFocus timing problem in Java 1.1.7+
        final long startTime = java.lang.System.currentTimeMillis();
        long currentTime = java.lang.System.currentTimeMillis();
        
        while ((currentTime - startTime) < 7) {
            // Just an empty loop
            currentTime = java.lang.System.currentTimeMillis();
        }
        
        // Now request focus
        if (useSwing) {
            System.out.println("We asked for a requestFocus");
            ((anecho.gui.JMSwingEntry)dataText).requestFocus();
        } else {
            ((TextArea)dataText).requestFocus();
        }
        
    }
    
    /**
     *
     * @param event
     */
    public void windowClosed(final WindowEvent event) {}
    
    /**
     *
     * @param event
     */
    public void windowClosing(final WindowEvent event) {}
    
    /**
     *
     * @param event
     */
    public void windowDeactivated(final WindowEvent event) {}
    
    /**
     *
     * @param event
     */
    public void windowDeiconified(final WindowEvent event) {}
    
    /**
     *
     * @param event
     */
    public void windowIconified(final WindowEvent event) {
        // record the frame into the hashtable
        // settings.setDataBar(getBounds());
        settings.setJMValue(JMConfig.DATABAR, getBounds());
    }
    
    /**
     *
     * @param event
     */
    public void windowOpened(final WindowEvent event) {}
    
    // Our customer TelnetEventListener
    /**
     * Currently not used.
     * @param event A telnet event.
     */
    public void telnetMessageReceived(final TelnetEvent event) {
//        // System.out.println("Confirming that DataIn has received: " + event);
//        // System.out.println("With our message being: " + event.getMessage());
//        // String message = event.getMessage();
//        // if (message.equals("IAC DO TELOPT_ECHO")) {
//        // dataText.setEchoChar('*');
//        // }
//
//        // if (message.equals("IAC DONT TELOPT_ECHO")) {
//        // dataText.setEchoChar('\u0000');
//        // }
    }
    
    /**
     * This is a generic bit to access the
     * ResReader.class, for localization
     * (Multi-language support)
     */
    private String resReader(final String itemTarget) {
        final ResReader reader = new ResReader();
        // return reader.langString("JamochaMUDBundle", itemTarget);
        return reader.langString(JMConfig.BUNDLEBASE, itemTarget);
    }
    
    /**
     * Method to hide all the frames.  The 'supervisor' function
     */
    public void hideFramesQuick() {
        // Hide all the frames,
        // the supervisor's here!
        setTitle(".");
        toBack();
        
        // We'll unsync the databar, otherwise bringing it up to the
        // front may result in all the other windows showing up again.  Eeek!
        setSync(false);
    }
    
    /**
     * Restore frames hidden by the 'Supervisor' function
     */
    public void restoreMinimizedFrames() {
        // Restore the frames that
        // were minimized by the 'supervisor' key
        // Fix this XXX
        // MuckMain.textWindow.setVisible(true);
        
        // Restore frame icons
        // dataBar.setIconImage(MuckConn.jMUDImage);
        // MuckMain.textWindow.setIconImage(MuckConn.jMUDImage);
        
        // Querry the MuckMain **** menu, to
        // see if we should set the Macros visible
        // Fix this XXX
        // if (MuckMain.tWMacro.getState()) {
        // tWMacro is set to 'true', so show the Macro Frame
        // MuckConn.jmMacros.setVisible(true);
        //}
        
        // Fix this XXX
        // if (MuckMain.tWSyncWindowsItem.getState()) {
        // setSync(true);
        // }
        // now reset the title
        // resetTitles();
    }
    
    /**
     * Add an entry to the 'command history' pop-up
     */
    /*
    private void addPopUpEntry(String str) {
        if (backTrack.getItemCount() > 9) {
            backTrack.remove(0);  // Removes first entry to make room
            for(int i = 0; i < backTrack.getItemCount(); i++) {
                backComm[i] = backComm[i + 1];
            }
        }
     
        backComm[backTrack.getItemCount()] = str;
        if (str.length() > 30) {
            backTrack.add(str.substring(0, 30) + "...");
        } else {
            backTrack.add(str);
        }
    }
     */
    
    private void updateEntryList(final String str) {
        // int curSize = backComm.length - 1;  // Current number of items in our list
        // int curSize = 0;
        
        historyV.addElement(str);
        
        if (historyV.size() > limit) {
            historyV.removeElementAt(0);
        }
        
//        if (backComm[0] == null) {
//            if (DEBUG) {
//                System.err.println("backComm history list is empty.  Adding: " + str);
//            }
//            backComm[0] = str;
//        } else {
//
//            int loopLimit = limit -1;
//            for (int i = 0; i < loopLimit; i++) {
//                curSize = i;
//                if (backComm[i + 1] != null) {
//                    backComm[i] = backComm[i + 1];
//                } else {
//                    break;
//                }
//            }
//            backComm[curSize + 1] = str;
//
//        }
//
//        //        if (curSize > 0) {
//        //            if (curSize == limit) {
//        //                curSize = limit - 1;
//        //            }
//        //        }
//
//        // drop the oldest item from our list
////        for (int i = 0; i < curSize; i++) {
////            backComm[i] = backComm[i + 1];
////        }
////
//        backComm[curSize] = str;
        if (DEBUG) {
            System.err.println("Successfully reached the end of updating history.");
        }
    }
    
    /**
     * This is an &quot;empty" method that allows other
     * classes to request that we send our existing text
     * to the MU*.  This doesn't happen often, but can
     * be a handy ability to have */
    public void jMSendText() {
        if (useSwing) {
            jMSendText(((anecho.gui.JMSwingEntry)dataText).getText());
        } else {
            jMSendText(((TextArea)dataText).getText());
        }
    }
    
    /**
     * Package up the data to be sent to the MU*, parsing off any
     * yucky characters such as 'new lines', etc., also determining
     * via user-preference whether they should be sent in Unicode
     * or standard ASCII format
     * @param outGoing The text to be sent to the currently active MU*.
     */
    public void jMSendText(final String outGoing) {
        String sendStr = outGoing;
        // Safety first!
        
        if (connHandler == null) {
            connHandler = settings.getConnectionHandler();
        }
        
        if (DEBUG) {
            System.err.println("Sending text: " + outGoing);
        }
        // Now add the command to the Popupmenu
        // addPopUpEntry(sendStr);
        updateEntryList(sendStr);
        
        // Check to see if this is a TinyFugue-like command
        // if (!(sendStr.startsWith("/") && settings.getJMboolean(settings.TFKEYEMU) && tfCommand.command(sendStr))) {
        if (!(sendStr.charAt(0) == '/' && settings.getJMboolean(settings.TFKEYEMU) && tfCommand.command(sendStr))) {
            
            // Send the text to the appropriate MU* via its connection handler
            if (DEBUG) {
                System.err.println("Sending text to connHandler: " + connHandler);
            }
            sendStr = translateMacros(sendStr);
            connHandler.sendText(sendStr);
        }
        
        // If local echo is active, append it to the user's screen as well
        // if (settings.isLocalEchoEnabled()) {
        if (settings.getJMboolean(JMConfig.LOCALECHO) && connHandler.getActiveMUEchoState()) {
            // if (connHandler.getActiveMUEchoState()) {
                if (settings.getJMboolean(JMConfig.USESWING)) {
                    anecho.gui.JMSwingText text;
                    text = connHandler.getActiveMUDSwingText();
                    text.append(resReader("localEchoPrepend") + " " + sendStr + '\n');
                } else {
                    anecho.gui.JMText text;
                    text = connHandler.getActiveMUDText();
                    text.append(resReader("localEchoPrepend") + " " + sendStr + '\n');
                }
            // }
        }
        
        // return the cursor to the starting position and empty the text area
        if (useSwing) {
            ((anecho.gui.JMSwingEntry)dataText).setCaretPosition(0);
            ((anecho.gui.JMSwingEntry)dataText).setText("");
        } else {
            ((TextArea)dataText).setCaretPosition(0);
            ((TextArea)dataText).setText("");
        }
        
        
    }
    
    /**
     * Set the current text from the scrollback menu,
     * the item indicated by int pos
     * @param currentPos The index of the command history to use as the text
     * for the DataIn text area.
     */
    public void jMSetFromScrollBack(final int currentPos) {
        int pos = currentPos;
        
        // If the integer comes in as '-1', that means to use the last string
        if (pos == -1) {
            // pos = backTrack.getItemCount() - 1;
            pos = historyV.size() - 1;
        }
        
        // First, check to see if the integer is valid
//        if (pos > backTrack.getItemCount() || pos < 0) {
//            return;
//        }
        
        if (pos > historyV.size() || pos < 0) {
            return;
        }
        
        final String text = historyV.elementAt(pos).toString();
        
        if (useSwing) {
            // ((anecho.gui.JMSwingEntry)dataText).setText(backComm[pos]);
            ((anecho.gui.JMSwingEntry)dataText).setText(text);
            ((anecho.gui.JMSwingEntry)dataText).setCaretPosition(backComm[pos].length());
        } else {
            // Now we'll set the text from the menu item
            // ((TextArea)dataText).setText(backComm[pos]);
            ((TextArea)dataText).setText(text);
            // Explicitly set the cursor location, as this is
            // not automatic on some operating systems.
            ((TextArea)dataText).setCaretPosition(backComm[pos].length());
            
        }
    }
    
    /**
     * Set the text visible in the DataIn window.  This method will
     * remove any text that may have previously been showing.
     * @param text The text to display in the DataIn window.
     */
    public void setText(final String text) {
        if (useSwing) {
            ((anecho.gui.JMSwingEntry)dataText).setText(text);
        } else {
            ((TextArea)dataText).setText(text);
        }
        
    }
    
    /**
     * Set the font to be used for the DataIn window.
     * @param newFont The Font to use.
     */
    public void setFont(final Font newFont) {
        
        dataText.setFont(newFont);
    }
    
    /**
     * Append the given string to whatever text exists in the DataIn
     * window.
     * @param text The text to be appended.
     */
    public void append(final String text) {
        if (useSwing) {
            ((anecho.gui.JMSwingEntry)dataText).append(text);
        } else {
            ((TextArea)dataText).append(text);
        }
    }
    
    /**
     * Get all the text currently in the DataIn window.
     * @return A String representing all the text currently in the
     * DataIn area.
     */
    public String getText() {
        String retText = "";
        if (useSwing) {
            retText = ((anecho.gui.JMSwingEntry)dataText).getText();
        } else {
            retText = ((TextArea)dataText).getText();
        }
        // return ((TextArea)dataText).getText();
        return retText;
    }
    
    /**
     * Set the caret's location based on the given index from the
     * first character in the text area.
     * @param pos Location of the caret
     */
    public void setCaretPosition(final int pos) {
        if (useSwing) {
            ((anecho.gui.JMSwingEntry)dataText).setCaretPosition(pos);
        } else {
            ((TextArea)dataText).setCaretPosition(pos);
        }
    }
    
    /**
     * Get the location of our caret based on the number of characters
     * from the beginning of the text area.
     * @return Index of the caret's location.
     */
    public int getCaretPosition() {
        int retInt = 0;
        if (useSwing) {
            retInt = ((anecho.gui.JMSwingEntry)dataText).getCaretPosition();
        } else {
            retInt = ((TextArea)dataText).getCaretPosition();
        }
        // return ((TextArea)dataText).getCaretPosition();
        return retInt;
    }
    
    /**
     * Get the number of columns visible in our DataIn display
     * @return Number of columns visible.
     */
    public int getColumns() {
        int retCol = 0;
        if (useSwing) {
            retCol = ((anecho.gui.JMSwingEntry)dataText).getColumns();
        } else {
            retCol = ((TextArea)dataText).getColumns();
        }
        return retCol;
        // return ((TextArea)dataText).getColumns();
    }
    
    /**
     * Sets the foreground colour for the DataIn area.
     * @param fgColour The foreground colour to use.
     */
    public void setForegroundColour(final Color fgColour) {
        dataText.setForeground(fgColour);
    }
    
    /**
     * Set the background colour for the DataIn window.
     * @param bgColour The background colour.
     */
    public void setBackgroundColour(final Color bgColour) {
        dataText.setBackground(bgColour);
    }
    
    /** Set the title of this frame on the basis of if the
     * active MU* (the one visible at the moment) is
     * connected or not.  This method differs from the main
     * window's setWindowTitle() method in that it takes
     * the text being paused into consideration as well
     */
    public synchronized void setWindowTitle() {
        // This resets the window's title, depending if the setWindowTitle();
        // client is connected to a MU* or not
        final CHandler connection = settings.getConnectionHandler();
        final boolean active = connection.isActiveMUDConnected();
        
        if (active) {
            final MuSocket mSock = connection.getActiveMUHandle();
            
            if (mSock.isPaused()) {
                this.setTitle(resReader("outputPaused"));
            } else {
                this.setTitle(connection.getActiveTitle());
            }
        } else {
            // Connection is inactive, just have program's title
            this.setTitle("JamochaMUD - " + resReader("notConnected"));
        }
    }
    
    /**
     * We export our TextArea for use in a &quot;combined"
     * JamochaMUD configuration, where both the input and
     * output are shown in the same frame
     * @return A reference to the text area of our DataIn object.
     */
    // public TextArea exportText() {
    // public TextArea exportText() {
    public Component exportText() {
        Component retComp;
        this.setVisible(false);
        this.setSync(false);
        
        
        if (useSwing) {
            retComp = dataInPane;
        } else {
            retComp = dataText;
        }
        // We should also remove this frame from the sync-group?
        // return (TextArea)dataText;
        // return dataText;
        return retComp;
    }
    
    /** We are having our TextArea &quot;returned" to us from
     * the main window.  We'll add it back to our frame and
     * then set everything visible again
     */
    public void restoreText() {
        if (useSwing) {
            this.add((anecho.gui.JMSwingEntry)dataText);
        } else {
            this.add((TextArea)dataText);
        }
        // this.add(dataText);
        this.pack();
        
        // Check for sync value to restore us to our previous glory
        // if (settings.getSyncWindows()) {
        if (settings.getJMboolean(JMConfig.SYNCWINDOWS)) {
            this.setSync(true);
        }
        
        this.setVisible(true);
    }
    
    /**
     * Return the variable representing our DataIn area.
     * @return Variable representing our DataIn area.
     */
    public Component getTextVariable() {
        Component retComp;
        
        if (useSwing) {
            retComp = dataInPane;
        } else {
            retComp = dataText;
        }
        // We should also remove this frame from the sync-group?
        // return (TextArea)dataText;
        // return dataText;
        return retComp;
    }
    
    /** Remove the paused status from JamochaMUD because of some
     * action that we have received
     */
    private void checkPause() {
        final CHandler target = settings.getConnectionHandler();
        final MuSocket mSock = target.getActiveMUHandle();
        
        if (mSock != null && mSock.isPaused()) {
//            if (mSock.isPaused()) {
                // This spools out any paused text
                mSock.spoolText();
//            }
        }
    }
    
    /**
     * Set the number of rows available in our DataIn area.  This is
     * only useful if the DataIn frame is being viewed as a separate
     * frame.
     * @param rows The number of rows to set for the DataIn window.
     */
    public void setRows(final int rows) {
        if (useSwing) {
            ((anecho.gui.JMSwingEntry)dataText).setRows(rows);
        } else {
            ((TextArea)dataText).setRows(rows);
        }
    }
    
    /** Do any required macro replacement in our string */
    private String translateMacros(final String macroIn) {
        String input = macroIn;
        boolean process = true;
        int start, end;
        String command, sString, eString, macro;
        
        /** Create a loop to continuously process any (recursive) macros we find */
        // We need to be able to locate endless loops!  Fix this XXX
        while (process) {
            start = input.indexOf("${");
            
            if (start > -1) {
                
                end = input.indexOf("}", start);
                if (end > -1 && end > start) {
                    command = input.substring(start + 2, end);
                    // System.out.println("Our macro is: " + command);
                    
                    if (start > 0) {
                        sString = input.substring(0, start);
                    } else {
                        sString = "";
                    }
                    if (end + 1 < input.length()) {
                        eString = input.substring(end + 1);
                    } else {
                        eString = "";
                    }
                    
                    macro = settings.getVariable(command);
                    // System.out.println("Our macro returns: " + macro);
                    //if (macro != null) {
                    // input = sString + macro + eString;
                    // } else {
                    // input = sString + command + eString;
                    // }
                    
                    if (macro == null) {
                        input = sString + command + eString;
                    } else {
                        input = sString + macro + eString;
                    }
                    
                }
                
            } else {
                // No more macros to process
                process = false;
            }
        }
        
        return input;
    }
    
    /** This lets our other classes bring us into focus */
    public void jmGainFocus() {
        if (useSwing) {
            if (DEBUG) {
                System.out.println("JMGainFocus()");
            }
            ((anecho.gui.JMSwingEntry)dataText).requestFocus();
            ((anecho.gui.JMSwingEntry)dataText).grabFocus();
        } else {
            ((TextArea)dataText).requestFocus();
        }
    }
    
    // This can probably be removed XXX
    public synchronized void validate() {
        dataText.invalidate();
    }
    
    /**
     * Return the last word typed
     * @return String representing the last "word" typed by the user.
     */
    public String getLastWord() {
        String retString = "";
        
        String inputText = "";
        int cursorPos = 0;
        
        if (useSwing) {
            inputText = ((anecho.gui.JMSwingEntry)dataText).getText();
            cursorPos = ((anecho.gui.JMSwingEntry)dataText).getCaretPosition();
        } else {
            inputText = ((TextArea)dataText).getText();
            cursorPos = ((TextArea)dataText).getCaretPosition();
        }
        
        // System.out.println("inputText: " + inputText);
        //System.out.println("cursorPos: " + cursorPos);
        
        if (cursorPos > 1) {
            // We use cursorPos - 2 as by the time we check, the space is already in place!
            int start = inputText.lastIndexOf(" ", cursorPos - 2);
            if (start < 0) {
                start = 0;
            }
            // System.out.println("Start point: " + start);
            retString = inputText.substring(start, cursorPos);
            // retString = inputText.substring(start, cursorPos - 1);
        }
        
        return retString.trim();
    }
    
    /**
     * Enable or disable anti-aliasing
     * @param state <CODE>true</CODE> - enable anti-aliased text
     * <CODE>false</CODE> - disable anti-aliased text
     */
    public void setAntiAliasing(final boolean state) {
        if (useSwing) {
            ((anecho.gui.JMSwingEntry)dataText).setAntiAliasing(state);
        }
        
    }
    
}
