/** This section is for TinyFugue emulated &quot;/" commands
 * like TinyFugue uses
 * $Id: JMTFCommands.java,v 1.7 2004/02/01 03:28:01 jeffnik Exp $
 */

package anecho.JamochaMUD;

import java.util.StringTokenizer;
import java.util.Vector;

/**
 * This class processes user input and scans it for TinyFugue style
 * "/ commands", or user defined definitions and variables.  If any
 * of these are found, the class will process them, otherwise it lets
 * the user input travel through unchanged.
 */
public class JMTFCommands {
    
    /**
     * The class that contains all the JamochaMUD user settings
     */
    private JMConfig settings;
    /**
     * A String representing the ANSI escape code.
     */
    private static final String ESC = "\u001b";
    /**
     * A static final variable used to show (<CODE>true</CODE>)or hide (<CODE>false</CODE>) debugging information.
     */
    private static final boolean DEBUG = false;
    
    /**
     * The constructor for this class.
     */
    // public JMTFCommands(JMConfig settings) {
    public JMTFCommands() {
        // this.settings = settings;
        settings = JMConfig.getInstance();
    }
    
    /**
     * The String is received from the input window and processed in this method.
     * This method looks for any item that begins with a forward slash, which
     * then signals that additional processing is required.
     * @param command Input received from the user to be processed
     * @return This boolean indicates whether our JMTFCommands class has
     * has processed the user input.
     * <CODE>true</CODE> - indicates that the input has been modified by this
     * class and will not need to be handled any more.
     * <CODE>false</CODE> - indicates that the input does not have any special
     * definitions, commands, or variables and so should be left
     * as-is and sent off to the MU*.
     */
    public boolean command(final String command) {
        boolean response = false;       // return-code becomes true if we do process a command
        String workString, cmd, tempFlag;
        StringTokenizer tokens;
        final Vector options = new Vector(0, 1);        // contain any flags that may be present on the command
        final Vector args = new Vector(0, 1);
        
        workString = command.toLowerCase();
        
        final int space = workString.indexOf(' ');
        
        if (space < 0) {
            // Single word command
            cmd = workString;
        } else {
            cmd = workString.substring(0, space);
        }
        
        tokens = new StringTokenizer(command);
        
        final int total = tokens.countTokens();
        
        // Parse apart our arguments and options
        for (int i = 0; i < total; i++) {
            tempFlag = tokens.nextToken().trim();
            // if (tempFlag.startsWith("-")) {
            if (tempFlag.charAt(0) == '-') {
                options.addElement(tempFlag);
            } else {
                args.addElement(tempFlag);
            }
        }
        
        if (cmd.equals("/connect")) {
            connectToWorld(args, options);
            response = true;
        }
        
        if (cmd.equals("/localecho")) {
            localEcho(args);
            response = true;
        }
        
        if (cmd.equals("/set")) {
            setVariable(command, args, options);
            response = true;
        }
        
        if (cmd.equals("/unset")) {
            unsetVariable(command);
            response = true;
        }
        
        if (cmd.equals("/def")) {
            makeDefinition(command, args, options);
            response = true;
        }
        
        if (cmd.equals("/undef")) {
            unsetDefinition(command);
            response = true;
        }
        
        if (!response) {
            response = userDefinedCommand(command,
                    options,
                    args);
        }
        
        if (DEBUG) {
            System.err.println("JMTFCommands.Command response is: " + response);
        }
        
        return response;
    }
    
    /**
     * This class will connect to a new MU* based on the keyboard input.
     * <I>Not yet implimented.</I>
     * @param userArgs not used.
     * @param options not used.
     */
    private void connectToWorld(final Vector userArgs, final Vector options) {
        if (userArgs.size() < 1 || options.size() < 1) {
            if (DEBUG) {
                System.err.println("JMTFCommands.connectToWorld() had " + userArgs.size() + " args and " + options.size() + " options.  Returning.");
            }
            return;
        }
        System.out.println(java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("Connect_to_") + userArgs.elementAt(1) + ":" + userArgs.elementAt(2));
    }
    
    /**
     * This method will allow the user to enable and disable local echo via
     * the text input.
     * <I>Not yet implemented.</I>
     * @param userArgs not used.
     */
    private void localEcho(final Vector userArgs) {
        
    }
    
    /**
     * This sets the value of a user defined variable, or displays all the set
     * variables if no additional arguments are given.  A set command that
     * gives only the variable name with no additional arguements returns the
     * value of that specific variable.
     * @param command This String contains the entire command from the user.  This
     * String will be parsed to determine if all variables are to
     * be listed, a single variable is to be listed, or the value
     * of a variable is to be set.
     * @param args unused
     * @param options unused
     */
    private void setVariable(final String command, final Vector args, final Vector options) {
        String name, value;
        final int eqSign = command.indexOf("=");
        
        if (command.length() <= eqSign) {
            sendUserMessage(java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("Set_usage"));
            return;
        }
        
        // if (command.toLowerCase().trim().equals("/set")) {
        if (args.size() == 1) {
            // Display all of our set items
            if (options.size() > 0) {
                displaySettingsWindow(settings.getAllVariables());
            } else {
                displaySettings(settings.getAllVariables(), java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("Variables"));
            }
            return;
        }
        
        if (eqSign < 0 && command.indexOf(" ") > 0) {
            // Show the user the value of the given variable
            name = command.substring(4).trim();
            value = settings.getVariable(name);
            
            if (value != null) {
                sendUserMessage(java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("Variable_") + ESC + "[1m" + name + ESC + "[0m " + java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("is_set_to_") + ESC + "[1m" + value + ESC + "[0m\n");
            } else {
                sendUserMessage(java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("Variable_") + ESC + "[1m" + name + "" + ESC + "[0m " + java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("_is_not_set") + "\n");
            }
            
            return;
        }
        
        name = command.substring(4, eqSign).trim();
        value = command.substring(eqSign + 1).trim();
        
        settings.addVariable(name, value);
        sendUserMessage(java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("Variable_") + ESC + "[1m" + name + "" + ESC + "[0m " + java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("is_now_set_to_") + ESC + "[1m" + value + "" + ESC + "[0m\n");
        
    }
    
    /**
     * This method removes a user-defined variable from JamochaMUD.
     * @param command This string contains the /unset command plus the name of the
     * variable to be deleted.
     */
    private void unsetVariable(final String command) {
        if (command.length() < 7) {
            sendUserMessage(java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("Unset_usage"));
            return;
        }
        
        final String name = command.substring(6).trim();
        
        settings.removeVariable(name);
        sendUserMessage(java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("Variable_") + ESC + "[1m" + name + "" + ESC + "[0m " + java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("has_been_unset_") + "\n");
        
    }
    
    /**
     * This method is used to set user-defined definitions (which can contain
     * variables).
     * @param command This String contains the command from the user, and will be
     * parsed to remove the /def directive and then set the definition
     * to the given value.  eg: <CODE>/def foo Something something</CODE> will
     * set the definition of <I>foo</I> to be <I>Something something</I>.
     * @param args A vector that contains all the arguments beyond the initial /def.
     * @param options This vector contains the definition itself from the user.
     */
    private void makeDefinition(final String command, final Vector args, final Vector options) {
        String name, value;
        int eqSign;
        
        // Initially we are going to ignore the args
        eqSign = command.indexOf("=");
        
        if (command.length() <= eqSign) {
            sendUserMessage(java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("Define_usage"));
            return;
        }
        
        if (args.size() == 1) {
            // Display all the definitions
            if (options.size() > 0) {
                displaySettingsWindow(settings.getAllDefinitions());
            } else {
                displaySettings(settings.getAllDefinitions(), java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("Definitions"));
            }
            return;
        }
        
        if (eqSign < 0 && command.indexOf(" ") > 0) {
            // return;
            // }
            
            // if (args.size() > 0) {
            // name = command.substring(5, eqSign).trim();
            name = command.substring(4).trim();
            // } else {
            value = settings.getDefinition(name);
            
            if (value != null) {
                sendUserMessage(java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("Definition_") + ESC + "[1m" + name + "" + ESC + "[0m " + java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("is_set_to_") + ESC + "[1m" + value + "" + ESC + "[0m\n");
            } else {
                sendUserMessage(java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("Definition_") + ESC + "[1m" + name + "" + ESC + "[0m " + java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("is_not_set") + "\n");
            }
            
            return;
        }
        
        name = command.substring(5, eqSign).trim();
        value = command.substring(eqSign + 1);
        
        settings.addDefinition(name, value);
        
        sendUserMessage(java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("Definition_") + ESC + "[1m" + name + "" + ESC + "[0m " + java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("set_to_") + ESC + "[1m" + value + "" + ESC + "[0m\n");
        
    }
    
    /**
     * This method parses user input to expand any variables or act upon any
     * definitions that are supplied
     * @param command The original input string received from the user.
     * @param options not used
     * @param args not used
     * @return This class returns
     * <CODE>true</CODE> - indicates that the input contained a definition and
     * was processed by this method.
     * <CODE>false</CODE> - this string was not processed, and should be handled
     * without modification.
     */
    private boolean userDefinedCommand(final String command, final Vector options, final Vector args) {
        
        String name, value, temp;
        boolean expand = true;
        int start, end;
        final DataIn sender = settings.getDataInVariable();
        
        temp = args.elementAt(0).toString();
        name = temp.substring(1);
        
        value = settings.getDefinition(name);
        
        if (DEBUG) {
            System.err.println("JMTFCommands.userDefinedCommand has a return value of " + value);
            System.err.println("We will send this command directly on to the MU*:" + command);
        }
        
        if (value == null) {
            
            return false;
        }
        
        // Now parse our new value for any arguments that need to be translated
        
        // We're going after anything that looks like %1 or %{1-something}
        while (expand) {
            start = value.indexOf("%");
            
            if (start + 1 >= value.length()) {
                start = -1;
                expand = false;
            }
            
            if (start > -1 && expand) {
                String fillToken = "";
                int replace;
                
                if (value.charAt(start + 1) == '{') {
                    // Complex replacement
                    end = start + 2;
                    while (end < value.length() && Character.isDigit(value.charAt(end))) {
                        end++;
                    }
                    
                    replace = Integer.parseInt(value.substring(start + 2, end));
                    
                    // Now figure out what our default replacement token will be
                    if (value.charAt(end) == '-') {
                        final int startDash = end + 1;
                        final int endBracket = value.indexOf("}", end);
                        
                        if (endBracket > -1 && endBracket <= value.length()) {
                            
                            fillToken = value.substring(startDash, endBracket);
                            end = endBracket + 1;
                        } else {
                            end++;
                        }
                    } else {
                        replace = args.size() + 1;
                        end = start + 1;
                    }
                    
                    
                    
                } else {
                    end = start + 1;
                    while (end < value.length() && Character.isDigit(value.charAt(end))) {
                        end++;
                    }
                    
                    
                    // Figure out replacement
                    replace = Integer.parseInt(value.substring(start + 1, end));
                    
                }
                
                String startS, endS, repValue;
                if (start > 0) {
                    startS = value.substring(0, start);
                } else {
                    startS = "";
                }
                if (end < value.length()) {
                    endS = value.substring(end);
                } else {
                    endS = "";
                }
                
                if (replace < args.size()) {
                    repValue = (String)args.elementAt(replace);
                } else {
                    repValue = fillToken;
                }
                
                value = startS + repValue + endS;
                
            } else {
                expand = false;
            }
            
        }
        
        // Send the new value via DataIn
        sender.jMSendText(value);
        
        return true;
        
    }
    
    /**
     * This method removes a user defined definition.
     * @param command The String contains the /undef command in addition to the
     * name of the definition to be removed from JamochaMUD.
     */
    private void unsetDefinition(final String command) {
        if (command.length() < 7) {
            sendUserMessage(java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("Undefine_usage"));
            return;
        }
        
        final String name = command.substring(6).trim();
        
        settings.removeDefinition(name);
        sendUserMessage(java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("Definition_") + ESC + "[1m" + name + "" + ESC + "[0m " + java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("has_been_undefined") + "\n");
    }
    
    /**
     * Writes a message back to the user (usually a confirmation
     * on the active MU* that is showing).
     * @param message This string contains the text to be sent back to the user.
     */
    private void sendUserMessage(final String message) {
        final CHandler connHandler = settings.getConnectionHandler();
        
        if (settings.getJMboolean(JMConfig.USESWING)) {
            anecho.gui.JMSwingText text;
            text = connHandler.getActiveMUDSwingText();
            text.append(message);
        } else {
            anecho.gui.JMText text;
            text = connHandler.getActiveMUDText();
            text.append(message);
        }
        
    }
    
    /**
     * Display a new dialogue that lists all of our definitions in the hashtable
     * @param userDefs This hashtable contains all the user definitions, the key
     * being the command and the object being the definition itself.
     */
    private void displaySettingsWindow(final java.util.Hashtable userDefs) {
        String name, value, tempLine;
        int longLine = 0;
        
        final anecho.gui.SyncFrame defFrame = new anecho.gui.SyncFrame("JamochaMUD");
        final java.awt.TextArea defs = new java.awt.TextArea("", 20, 20, java.awt.TextArea.SCROLLBARS_VERTICAL_ONLY);
        defs.setEditable(false);
        defFrame.add(defs);
        
        // Iternate through our hashtable and show all our settings:
        final java.util.Enumeration keys = userDefs.keys();
        
        while (keys.hasMoreElements()) {
            name = (String)keys.nextElement();
            value = userDefs.get(name).toString();
            tempLine = name + " = " + value + '\n';
            // defs.append(name + " = " + value + '\n');
            if (tempLine.length() > longLine) {
                longLine = tempLine.length();
            }
            defs.append(tempLine);
        }
        
        defs.setColumns(longLine);
        
        defFrame.setCloseState(true);
        defFrame.pack();
        defFrame.setVisible(true);
    }
    
    /**
     * Write the hashtable out to the current MU*
     * @param userDefs This hashtable contains all the user definitions, the key
     * being the command and the object being the definition itself.
     */
    private void displaySettings(final java.util.Hashtable userDefs, final String title) {
        String name, value;
        
        // Determine is the Hashtable has any keys.  If there are no keys we
        // should give the user a message indicating this.
        if (userDefs.isEmpty()) {
            sendUserMessage(java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("JamochaMUD_currently_has_no_") + title + java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("_assigned") + "\n");
        } else {
            
            final java.util.Enumeration keys = userDefs.keys();
            
            while (keys.hasMoreElements()) {
                name = (String)keys.nextElement();
                value = userDefs.get(name).toString();
                sendUserMessage("" + ESC + "[1m" + name + "" + ESC + "[0m = " + ESC + "[1m" + value + "" + ESC + "[0m\n");
            }
        }
    }
}



