/**
 * A MuSocket is used to hold all information important to
 * a single socket connection
 * $Id: MuSocket.java,v 1.35 2004/07/28 03:13:05 jeffnik Exp $
 */

/* JamochaMUD, a Muck/Mud client program
 * Copyright (C) 1998-2005 Jeff Robinson
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

package anecho.JamochaMUD;

import anecho.gui.JMText;
import anecho.gui.JMSwingText;
import anecho.gui.ResReader;

import anecho.extranet.Socks5socket;
import anecho.extranet.MUDBufferedReader;
import anecho.extranet.event.TelnetEvent;
import anecho.extranet.event.TelnetEventListener;


import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;

import java.io.DataOutputStream;

import java.net.ConnectException;
import java.net.InetAddress;
import java.net.NoRouteToHostException;
import java.net.Socket;
import java.net.UnknownHostException;

import java.util.Vector;

public class MuSocket extends Thread implements MouseListener, TelnetEventListener {
    
    private String muckName, address;
    private int port;
    private long timeStamp;
    private int activity = 0;   // Keep track of new activity if we're not the activeMU
    private boolean activeStatus = true;   // Is this connection the "active" one?
    private boolean connected = false;        // The state of the connection
    private boolean paused;             // Has our output been paused?
    private boolean echo = true;        // MU*'s should always start with ECHO on.
    private JMText mainText;            // The text-area belonging to this connection
    private JMSwingText mainSText;      // The swing version of our text area
    private JMConfig settings;          // A pointer to our settings
    private Socket connSock;
    private String fromMU;              // a string to hold output from the MU*
    private MUDBufferedReader inStream; // input to us from the MU*
    private DataOutputStream outStream; // for sending info to the MU*
    private Vector heldResponse;        // A vector to hold any text incase we are "paused"
    private int minLineCount = 0; // the number of minimised lines
    private CHandler connHandler;
    private boolean useSwing = false;
    private String logTitle;    // An auto-generated file name for logs specific to this session
    private boolean logging = false;    // indicate if auto-logging should be occurring
    
    private java.io.FileOutputStream logFile;	// The outputstream for our log file
    private java.io.PrintWriter logWriter;      // The Printwriter for our log file
    
    private static final boolean DEBUG = false;
    
    /** The constructor.  Behold!! */
//    public MuSocket(JMConfig settings) {
    public MuSocket() {
        // this.settings = settings;
        settings = JMConfig.getInstance();
        
        if (settings.getJMboolean(JMConfig.USESWING)) {
            useSwing = true;
        }
        
        // Set up a new text area
        if (useSwing) {
            java.awt.Color[] newPal = (java.awt.Color[])settings.getJMObject(JMConfig.CUSTOMPALETTE);
            mainSText = new JMSwingText();
            mainSText.addMouseListener(this);
            mainSText.setForeground(settings.getJMColor(settings.FOREGROUNDCOLOUR));
            mainSText.setBackground(settings.getJMColor(settings.BACKGROUNDCOLOUR));
            mainSText.setFont(settings.getJMFont(settings.FONTFACE));
            mainSText.setPaintBackground(settings.getJMboolean(JMConfig.BGPAINT));
            mainSText.setEditable(false);
            mainSText.addKeyListener(settings.getMainWindowVariable());
            mainSText.setAntiAliasing(settings.getJMboolean(JMConfig.ANTIALIAS));
            mainSText.setBoldNotBright(settings.getJMboolean(JMConfig.LOWCOLOUR));
            try {
                mainSText.setPalette(newPal);
            } catch(Exception exc) {
                if (DEBUG) {
                    System.err.println("Error setting custom palette from MuSocket.");
                }
            }
            
        } else {
            mainText = new JMText("", 50, 80, JMText.SCROLLBARS_VERTICAL_ONLY);
            // Set the attributes of our text area
            mainText.setEditable(false);
            
            mainText.setDoubleBuffer(settings.getJMboolean(JMConfig.DOUBLEBUFFER));
            
            mainText.addMouseListener(this);
            
            mainText.setForeground(settings.getJMColor(settings.FOREGROUNDCOLOUR));
            mainText.setBackground(settings.getJMColor(settings.BACKGROUNDCOLOUR));
            mainText.setFont(settings.getJMFont(settings.FONTFACE));
        }
        
        heldResponse = new Vector(0, 1);
        connHandler = settings.getConnectionHandler();
    }
    
    // Check for mouse actions
    
    /**
     *
     * @param mEvent
     */
    public void mousePressed(final MouseEvent mEvent) {
        setPaused(true);
    }
    
    /**
     *
     * @param mEvent
     */
    public void mouseReleased(final MouseEvent mEvent) {
    }
    
    /**
     *
     * @param mEvent
     */
    public void mouseClicked(final MouseEvent mEvent) {
        // Determine number of mouse clicks
        final int count = mEvent.getClickCount();
        
        if (count >= 2) {
            // Call outside routines (plug-ins) to handle the selected text
            mEvent.consume();
            spoolText();
            
            // grab the 'selected' URL
            String tempText;
            
            if (useSwing) {
                if (DEBUG) {
                    System.err.println("MuSocket.mouseClicked trying to get information from mainSText() " + mainSText);
                }
                tempText = (mainSText.getSelectedText()).trim();
            } else {
                tempText = (mainText.getSelectedText()).trim();
            }
            
            final StringBuffer tentativeURL = new StringBuffer(tempText);
            // StringBuffer tentativeURL = new StringBuffer((mainText.getSelectedText()).trim());
            if (DEBUG) {
                System.out.println("MuSocket.mouseClicked() using " + tentativeURL + " for external program.");
            }
            
            // Fix this XXX
            // The frame is sent to anchor any potential dialogues
            // send the 'URL' to launch the correct program
            // ExternalProgs ep = new ExternalProgs(settings.getMainWindowVariable(), settings);
//            final ExternalProgs extp = new ExternalProgs(settings.getJMFrame(JMConfig.MUCKMAINFRAME), settings);
            final ExternalProgs extp = new ExternalProgs(settings.getJMFrame(JMConfig.MUCKMAINFRAME));
            // ep.launchProgram(settings.getMainWindowVariable(), tentativeURL);
            extp.launchProgram(settings.getJMFrame(JMConfig.MUCKMAINFRAME), tentativeURL);
            
            // As a quick fix we'll copy this to the Clipboard instead.
            if (useSwing) {
                // java.awt.datatransfer.Clipboard clip = mainSText.getToolkit().getSystemClipboard();
                mainSText.copy();
            } else {
                final java.awt.datatransfer.Clipboard clip = mainText.getToolkit().getSystemClipboard();
                
                try {
                    final java.awt.datatransfer.StringSelection selection = new java.awt.datatransfer.StringSelection(tentativeURL.toString());
                    clip.setContents(selection, selection);
                } catch (Exception clipOopse) {
                    System.out.println("MuSocket exception copying to Clipboard." + clipOopse);
                }
                
                // Deselect the text
                mainText.select(mainText.getSelectionEnd(), mainText.getSelectionEnd());
                if (DEBUG) {
                    System.err.println("MuSocket: Deselect text?");
                }
                
            }
            
            
        } else {
            // A single click signals a pause
            // This will pause the output window,
            // the incoming lines held in queue
            setPaused(true);
        }
    }
    
    /**
     *
     * @param mEvent
     */
    public void mouseEntered(final MouseEvent mEvent) {
    }
    
    /**
     *
     * @param mEvent
     */
    public void mouseExited(final MouseEvent mEvent) {
    }
    
    /**
     * The name of our connection
     * @param newName 
     */
    public synchronized void setTitle(final String newName) {
        if (DEBUG) {
            System.out.println("MuSocket.setTitle setting muckName to: " + newName);
        }
        muckName = newName;
        setFrameTitles();
        if (DEBUG) {
            System.out.println("MuSocket.setTitle complete.");
        }
        
    }
    
    /**
     * getTitle does a small bit of book-keeping, returning a title based
     * on whether we're connected to the MU* or not.  If we're not connected
     * then we'll return &quot;JamochaMUD" instead of the name
     * @return 
     */
    public synchronized String getTitle() {
        String retStr;
        
        if (!connected) {
            retStr = "Not Connected: " + address;
        } else {
            
            if (muckName == null) {
                if (DEBUG) {
                    System.err.println("MuSocket.getTitle -> current muckName is null and will be set to " + address);
                }
                muckName = address;
            }
            retStr = "JamochaMUD (" + muckName + ")";
        }
        
        return retStr;
    }
    
    /**
     * This returns the true name of the MU*, regardless of our connection
     * @return 
     */
    public String getMUName() {
        return muckName;
    }
    
    /** Call this to reset the frame's title to the &quot;standard" title.
     * This is usually just the location */
    public synchronized void resetTitle() {
        setFrameTitles();
    }
    
    /**
     * The Address of our connection
     * @param address 
     */
    public void setAddress(final String address) {
        this.address = address;
    }
    
    /**
     *
     * @return
     */
    public String getAddress() {
        return address;
    }
    
    /**
     * The port our connection will connect to
     * @param port 
     */
    public void setPort(final int port) {
        this.port = port;
    }
    
    /**
     *
     * @return
     */
    public int getPort() {
        return port;
    }
    
    /**
     * The timestamp of our connection, for internal use mostly
     * @param time 
     */
    public void setTimeStamp(final long time) {
        this.timeStamp = time;
    }
    
    /**
     *
     * @return
     */
    public synchronized long getTimeStamp() {
        return timeStamp;
    }
    
    /** This is the main method for our thread.  Veryvery important!
     *  Basically, it'll be a big loop that only gets broken when our
     * connection disappears... either from the socket's end or the
     * user's end */
    public void run() {
        final String tempAddy = this.getAddress();
        
        // Give some visual notification that we're attempting a connection
        setVisuals(tempAddy);
        
        // Write the location to the main window's title, as well
        
        // Try to establish a connection
        final int port = this.getPort();
        InetAddress serverAddy;
        
        // We'll give sockPort and sockHost initial values to satisfy later conditions
        int sockPort = 0;
        String sockHost = "";
        final boolean socks = settings.getJMboolean(JMConfig.PROXY);
        
        if (socks) {
            sockHost = settings.getJMString(JMConfig.PROXYHOST);
            sockPort = settings.getJMint(JMConfig.PROXYPORT);
        }
        
        try {
            serverAddy = InetAddress.getByName(tempAddy);
            if (socks) {
                if (DEBUG) {
                    System.out.println("Attempt to use Socks5socket.");
                }
                connSock = (Socket)new Socks5socket(tempAddy, port, sockHost, sockPort);
                if (DEBUG) {
                    System.out.println("Socks5socket successfully connected");
                }
            } else {
                connSock = new Socket(serverAddy, port);
            }
            outStream = new DataOutputStream(connSock.getOutputStream());
        } catch (NoRouteToHostException oops) {
            // No route to host, or operation timed out
            disconnectMenu();
            
            final String tempString = oops + "";
            if (tempString.endsWith("unreachable")) {
                // Host unreachable
                // mainText.append(resReader("noRouteToHostException") + '\n');
                write(resReader("noRouteToHostException") + '\n');
            } else {
                // mainText.append(resReader("operationTimedOutException") + '\n');
                write(resReader("operationTimedOutException") + '\n');
            }
        } catch (UnknownHostException oops) {
            disconnectMenu();
            // mainText.append(resReader("unknownHostException") + '\n');
            write(resReader("unknownHostException") + '\n');
        } catch (ConnectException oops) {
            disconnectMenu();
            // mainText.append(resReader("connectException") + '\n');
            write(resReader("connectException") + '\n');
        } catch (Exception oops) {
            disconnectMenu();
            System.out.println("From Net, no socket " + oops);
            
            // mainText.append(resReader("exception") + '\n');
            write(resReader("exception") + '\n');
            // Fix this XXX
            // closeConnection();
        }
        
        // Create a MUDBufferedReader for our input stream from the MU*
        try {
            inStream = new MUDBufferedReader(connSock.getInputStream(), outStream);
            inStream.addTelnetEventListener(settings.getDataInVariable());
            inStream.addTelnetEventListener(this);
        } catch (Exception except) {
            if (DEBUG) {
                System.out.println("MuSocket.run -> No inStream");
            }
            
            // mainText.append("---> " + e + " <---" + '\n');
            write("---> " + except + " <---" + '\n');
            closeCleanUp();
            return;
        }
        
        // Okay, we're connected.  Do our little set-up stuff
        
        // Some specialised MU*'s require a form of authentication immediately
        // after connection.  This could be done here by adding something like:
        // sendText("foo");
        
        // Read from the socket until we get tired and fall down
        connected = true; // This notifies the program of true connection
        
        // Change the titles of our frames to show we're connected
        setFrameTitles();
        
        // Update the main window menus to reflect our connected state
        connectMenu();
        
        try {
            while (connected) {
                // Commented out 2005-09-24 to move exception handling outside of while loop
                fromMU = inStream.readLine();
//            try {
//                fromMU = inStream.readLine();
//            } catch (Exception except) {
//                // We've run into some sort've problem, so chances are
//                // our connection has been terminated in some way.
//                if (DEBUG) {
//                    System.out.println("Error at FromNet inStream.readLine " + except);
//                    except.printStackTrace();
//                }
//
//                fromMU = (String)null;
//                connected = false;
//            }
                
                // Check for output plugins
                // Fix this XXX
                // we should probably not be making a "static" call like this
                
//                // Check to see if we somehow got disconnected.  (It can happen!)
//                if (fromMU == null) {
//                    // sets out menu to the correct state as well as closes the socket
//                    // mainText.append("---> " + resReader("connectionClosed") + " <---" + '\n');
//                    write("---> " + resReader("connectionClosed") + " <---" + '\n');
//                    closeSocket();
//                    connected = false;
//                    disconnectMenu();
//                    return;
//                }
                
                fromMU = EnumPlugIns.callPlugin(fromMU, "output", this);
                
                // Send the current line to our Text-field
                if (!paused) {
                    // Check for any URLs
                    fromMU = checkURL(fromMU);
                    
                    write(fromMU);
                    
                    // We'll notify the main window if needed
                    if (!isActiveMU()) {
                        if (activity < 1) {
                            // Append an activity notice to the active window
                            connHandler.getActiveMUDText().append(resReader("activityOn") + " " + getMUName() + '\n');
                            activity++;
                        }
                    } else {
                        activity = 0;
                        // System.out.println("Reset activity.");
                    }
                    
                    // Check to see if the main frame is minimised.  If so, we add
                    // the new text to the title bar
                    // if (settings.isMainWindowIconified()) {
                    if (settings.getJMboolean(JMConfig.MAINWINDOWICONIFIED)) {
                        addTitleText(fromMU);
                    } else {
                        minLineCount = 0;
                    }
                    
                } else {
                    // System.out.println("paused: " + fromMU);
                    heldResponse.addElement(fromMU);
                }
            }
        } catch (Exception exc) {
            if (DEBUG) {
                System.err.println("MuSocket.run(): We've fallen out of our while loop." + exc);
            }
            // Check to see if we somehow got disconnected.  (It can happen!)
            // sets out menu to the correct state as well as closes the socket
            // mainText.append("---> " + resReader("connectionClosed") + " <---" + '\n');
            write("---> " + resReader("connectionClosed") + " <---" + '\n');
            closeSocket();
            connected = false;
            disconnectMenu();
            
        }
    }
    
    private static String resReader(final String itemTarget) {
        final ResReader reader = new ResReader();
        return reader.langString(JMConfig.BUNDLEBASE, itemTarget);
    }
    
    // Return our text area to who-ever wants to know! */
    /**
     *
     * @return
     */
    public synchronized JMText getTextWindow() {
        return mainText;
    }
    
    /* Return the swing version of our text window */
    /**
     *
     * @return
     */
    public synchronized JMSwingText getSwingTextWindow() {
        return mainSText;
    }
    
    /**
     * Send text out to our MU*
     * @param send 
     */
    public void sendText(final String send) {
        try {
            // if (settings.getUseUnicode()) {
            if (settings.getJMboolean(JMConfig.USEUNICODE)) {
                // Send the string as Unicode
                outStream.writeChars(send + "\r\n");
            } else {
                // Send the string as regular ANSI text
                outStream.writeBytes(send + "\r\n");
            }
        } catch (Exception except) {
            // Most likely we aren't connected to the MU*
            if (DEBUG) {
                System.out.println("MuSocket.sendText exception: " + except);
            }
        }
    }
    
    /** This will set up some of the initial indicators
     * that we're trying to make a connection */
    private void setVisuals(final String title) {
        // mainText.append(resReader("attemptingConnectionTo") + " " + title + "..." + '\n');
        write(resReader("attemptingConnectionTo") + " " + title + "..." + '\n');
    }
    
    private void setFrameTitles() {
        // We should only be able to change the frame titles it this
        // is the active MU*
        if (connHandler.getActiveMUHandle() == this) {
            final MuckMain target = settings.getMainWindowVariable();
            target.setWindowTitle();
            
            final DataIn inWindow = settings.getDataInVariable();
            inWindow.setWindowTitle();
        }
    }
    
    /**
     * Return the status of our connection:<br>
     * @return <CODE>true</CODE> - Connected
     * <CODE>false</CODE> - Not connected
     */
    public boolean isConnectionActive() {
        if (DEBUG) {
            System.err.println("MuSocket.isConnectionActive() returns: " + connected);
        }
        return connected;
    }
    
    /** Set the proper states of our windows, etc.
     * 'cause, woe is us, we are disconnected.  Most often
     * this would be called internally by our class
     */
    private void disconnectMenu() {
        // This is where we actually disconnect the socket.
        // Is there a nicer way of saying good-bye?
        
        // Do we really need to close the socket?!  We should get here because the
        // socket is already closed!  Check this out! XXX
        // This guarantees that we've cleaned up after ourselves, too
        
        // Change the connection notice on the main window if applicable
        final MuckMain window = settings.getMainWindowVariable();
        window.checkDisconnectMenu();
        
    }
    
    /** Allow other classes to close our socket */
    public void closeSocket() {
        /*
        if (DEBUG) {
            StringWriter sw = new StringWriter();
            new Throwable().printStackTrace(
                                            new PrintWriter( sw )
                                           );
            String callStack = sw.toString();
         
            System.out.println("Stack: " + callStack);
        }
         */
        
        try {
            if (connSock != null) {
                connSock.close();
            }
        } catch (Exception closeError) {
            // The only reason we should get here is if the socket was never opened
            // in the first place.  This can happen if we tried to open a bad address
            if (DEBUG) {
                System.out.println("MuSocket hit an error (.disconnectMenu): " + closeError);
            }
        }
        
        // Change the titles of the frames if applicable
        resetTitle();
        
        // return;
    }
    
    /** Set the proper states of our windows as
     * we've probably just connected
     */
    private void connectMenu() {
        final MuckMain main = settings.getMainWindowVariable();
        main.setConnectionMenu();
    }
    
    /**
     * Set the paused status of our text area
     * @param option 
     */
    public void setPaused(final boolean option) {
        paused = option;
        
        if (option) {
            setFrameTitles();
        } else {
            spoolText();
        }
    }
    
    /**
     * Return the fact that our connection is either
     * paused - <pre>true</pre><br>
     * or not paused - <pre>false</pre><br>
     * @return 
     */
    public synchronized boolean isPaused() {
        return paused;
    }
    
    /**
     * Write any "held" text out to the user's display.
     */
    public synchronized void spoolText() {
        // We'll reset this variable, just in case
        paused = false;
        
        // System.out.println("Spooltext, " + heldResponse.size() + " entries.");
        
        // Reset our frame titles
        resetTitle();
        
        // Do a little bounds checking first
        if (heldResponse.size() < 1) {
            /* System.out.println("MuSocket: spoolText, nothing to spool."); */
            return;
        }
        
        // System.out.println("MuSocket.spoolText: we have " + heldResponse.size() + " elements.");
        while (heldResponse.size() > 0) {
            // System.out.println("MuSocket.spoolText: " + heldResponse.elementAt(0).toString());
            // mainText.append(heldResponse.elementAt(0).toString());
            write(heldResponse.elementAt(0).toString());
            if (DEBUG) {
                System.out.println(heldResponse.elementAt(0).toString());
            }
            
            heldResponse.removeElementAt(0);
        }
    }
    
    /** Trim down the string we've received from the MU*, add
     * the minimised line count, and the set the new title */
    private synchronized void addTitleText(final String newText) {
        minLineCount++;
        // System.out.println("MuSocket.addTitleText: " + minLineCount + " " + newText);
        
        final String str = stripEscapes(newText);
        
        // Trim the str down to a managable length
        String trimString;
        if (str.length() > 80) {
            trimString = str.substring(0, 80) + "...";
        } else {
            trimString = str;
        }
        
        final MuckMain target = settings.getMainWindowVariable();
        // target.setWindowTitle("(" + minLineCount + ") " + str);
        target.setWindowTitle("(" + minLineCount + ") " + trimString);
        
    }
    
    /** This method returns the value of a string
     * once it has had any escape characters stripped
     * from it.
     */
    private String stripEscapes(final String token) {
        final StringBuffer workString = new StringBuffer("");
        boolean loop = true;
        int start = 0;
        int end = 0;
        
        while (loop) {
            end = token.indexOf('\u001b', start);
            
            if (end < start) {
                // There are no escapes left
                workString.append(token.substring(start));
                loop = false;
                break;
            }
            
            if (end > 0) {
                workString.append(token.substring(start, end));
            }
            
            // Now set the new 'start'
            start = token.indexOf('m', end) + 1;
            if (start <= end) {
                loop =false;
                System.out.println("Not a proper ANSI escape");
                break;
            }
        }
        
        return workString.toString();
    }
    
    private void closeCleanUp() {
        if (DEBUG) {
            System.err.println("MuSocket.closeCleanUp() calling closeSocket()");
        }
        closeSocket();
        connected = false;
        if (DEBUG) {
            System.err.println("MuSocket.closeCleanUp() calling disconnectMenu()");
        }
        disconnectMenu();
        if (DEBUG) {
            System.err.println("MuSocket.closeCleanUp() completed.");
        }
    }
    
    /** Returns <pre>true</pre> if this connection is the currently active
     * (visible to the user) MU* connection.  <pre>False</pre> is returned
     * otherwise.
     */
    private boolean isActiveMU() {
        return activeStatus;
    }
    
    /**
     * Indicate that this MU active or inactive
     * @param status 
     */
    public void setActiveMU(final boolean status) {
        activeStatus = status;
        // System.out.println("active state: " + status + " for " + getTitle());
        
        // If this MU* is set active then we can reset our activity counter
        if (status) {
            activity = 0;
        }
    }
    
    public void validate() {
        System.out.println("MuSocket.validate called.");
        if (useSwing) {
            mainSText.validate();
        } else {
            mainText.validate();
        }
    }
    
    /**
     * Return the state of our "echo"
     * @return 
     */
    public boolean getEchoState() {
        return echo;
    }
    
    private void setEchoState(final boolean state) {
        echo = state;
        System.out.println("MuSocket echo state: " + echo);
    }
    
    /** Check the string for anything understood as a URL and underline it */
    private String checkURL(final String input) {
        String checkChunk, lCheckChunk;
        final StringBuffer incoming = new StringBuffer(input);
        
        final java.util.StringTokenizer tokens = new java.util.StringTokenizer(input);
        
        while (tokens.hasMoreTokens()) {
            checkChunk = tokens.nextToken();
            lCheckChunk = checkChunk.toLowerCase();
            if (lCheckChunk.indexOf("www.") >= 0 || lCheckChunk.indexOf("http://") >= 0) {
                int match;
                match = (incoming.toString()).indexOf(checkChunk);
                
                // Fix ME XXX!!!
                // Using Swing we may want to change this to a "label" and insert it as a
                // component into our JMSwingText component
                
                // Added back-to-front so we don't have to do extra calculations
                if (!useSwing) {
                    // if (lCheckChunk.indexOf("http.") < 0) {
                    // Add the http:// so that our link conforms to the URL standard
                    // incoming.insert(match, "http://");
                    // }
                    // incoming.insert(match + checkChunk.length(), "</a>");
                    // incoming.insert(match, "<a href=\"" + lCheckChunk + "\">");
                    // } else {
                    incoming.insert(match + checkChunk.length(), "\u001b[24m");
                    incoming.insert(match, "\u001b[4m");
                }
                
            }
        }
        
        return incoming.toString();
    }
    
    /** Write out given text to our text component */
    private synchronized void write(final String output) {
        if (useSwing) {
            mainSText.append(output);
        } else {
            mainText.append(output);
        }
        
        // if logging is enabled, this should be written to the log as well
        if (logging) {
            writeLog(output);
        }
    }
    
    /*
    public void setDoubleBuffer(boolean state) {
        mainText.setDoubleBuffer(state);
    }
     
    public void setFont(Font fontStyle) {
        mainText.setFont(fontStyle);
    }
     
    public void setForeground(Color fgColor) {
        mainText.setForeground(fgColor);
    }
     
    public void setBackground(Color bgColor) {
        mainText.setBackground(bgColor);
    }
     */
    
    /**
     * Custom telnet message receiver for our TelnetEventListener
     * @param event 
     */
    public void telnetMessageReceived(final TelnetEvent event) {
        
        if (DEBUG) {
            System.out.println("MuSocket.telnetMessageReceived has received: " + event);
            System.out.println("With our message being: " + event.getMessage());
        }
        
        final String message = event.getMessage();
        if (message.equals("IAC DO TELOPT_ECHO")) {
            // Turn off the local echo
            setEchoState(false);
        }
        
        if (message.equals("IAC DONT TELOPT_ECHO")) {
            // Turn the local echo back on.
            setEchoState(true);
        }
    }
    
    /**
     * Enable or disable auto-logging
     * @param status 
     */
    public void setLogging(final boolean status) {
        logging = status;
        if (DEBUG) {
            System.err.println("Logging has been set to " + status);
        }
        
        if (!status && logFile != null) {
            // Logging has been disabled.  Close our file handle
            try {
                logWriter.flush();
                logWriter.close();
            } catch (Exception exc) {
                if (DEBUG) {
                    System.err.println("Exception trying to flush and close logWriter in MuSocket.setLogging");
                }
            }
            
            logFile = null;
            
        }
    }
    
    /**
     * Return our auto-logging status
     * @return 
     */
    public boolean isLogging() {
        return logging;
    }
    
    /** Return a string that should be used as a file name for logging this session
     *  If a title does not yet exist, we will create one
     *  Title format: MU* address + timestamp + .txt
     */
    private String fileTitle() {
        String workingTitle;
        
        if (logTitle == null) {
            // Assemble a date string that is human readable from our timestamp
            final java.util.Date workDate = new java.util.Date(timeStamp);
            final java.text.SimpleDateFormat formatter = new java.text.SimpleDateFormat("yy-MM-dd_H-mm-ss");
            
            // Create the title for our file and assign it to the logTitle variable
            final StringBuffer tempStr = new StringBuffer();
            tempStr.append(muckName);
            tempStr.append("_");
            tempStr.append(formatter.format(workDate));
            tempStr.append(".txt");
            
            workingTitle = tempStr.toString();
            
            // Set up our logTitle
            logTitle = workingTitle;
            
            if (DEBUG) {
                System.err.println("Our new logTitle is " + logTitle);
            }
            
            
        } else {
            workingTitle = logTitle;
        }
        
        return workingTitle;
    }
    
    /** Write our output to our log file
     */
    private void writeLog(final String output) {
        if (logFile == null) {
            // Setup our variables for writing
            // We'll keep the append option to true, as we can't tell if the
            // user stopped logging and then perhaps started again.
            // String logPath = settings.getJMString(JMConfig.LOGPATH) + fileTitle();
            
            final String logPath = settings.getJMString(JMConfig.USERDIRECTORY) + "logs";
            final String totalLogFile = logPath + settings.getJMString(JMConfig.PATHSEPARATOR) + fileTitle();
            
            if (DEBUG) {
                System.err.println("Our current logpath is: " + logPath);
                System.err.println("Our total logfile is: " + totalLogFile);
            }
            
            try {
                // java.io.File logFileName = new java.io.File(fileTitle());
                if (!new java.io.File(logPath).exists()) {
                    if (DEBUG) {
                        System.err.println("Attempting to create directory for logs");
                    }
                    new java.io.File(logPath).mkdirs();
                    if (DEBUG) {
                        System.err.println("Successfully created log directory.");
                    }
                }
                
                logFile = new java.io.FileOutputStream(totalLogFile, true);
                logWriter = new java.io.PrintWriter(logFile, true);
            } catch (Exception exc) {
                if (DEBUG) {
                    System.err.println("MuSocket.writeLog: " + exc);
                }
            }
            
        }
        
        final String cleanOut = anecho.gui.TextUtils.stripEscapes(output, false);
        // logWriter.print(output);
        logWriter.print(cleanOut);
        logWriter.flush();
    }
}
