/**
 * Main Muck Output Window
 * $Id: MuckMain.java,v 1.59 2004/08/03 02:06:40 jeffnik Exp $
 */

/* JamochaMUD, a Muck/Mud client program
 * Copyright (C) 1998-2005  Jeff Robinson
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

package anecho.JamochaMUD;

import java.awt.*;
import java.awt.datatransfer.*;
import java.awt.event.*;
// import java.awt.event.ActionListener;
import java.net.*;
import java.io.*;
import java.util.*;

import anecho.gui.SyncFrame;
import anecho.gui.JMText;
import anecho.gui.ResReader;
import anecho.gui.PosTools;

import anecho.JamochaMUD.plugins.PlugInterface;

/**
 * Main Muck Output Window, heart of the program, containing
 * most of the menus and logistics of JamochaMUD
 * @version $Id: MuckMain.java,v 1.59 2004/08/03 02:06:40 jeffnik Exp $
 * @author Jeff Robinson
 */
public class MuckMain implements ActionListener, ComponentListener, KeyListener, ItemListener, WindowListener {
    
    private CheckboxMenuItem splitFramesItem;
    private CheckboxMenuItem tWAltFocus;
    
    static boolean pauseStatus, textWindowStatus;
    static BufferedReader input;
    private Frame muckMainFrame;
    static Socket serverSock;
    protected static Menu tWPlugInMenu;
    public static final int COMBINED = 0;     // Combined frame style
    public static final int SPLIT = 1;        // Split frame style
    private int viewStyle = COMBINED;         // the current style of our display
    private GridBagLayout mainBagLayout;
    private GridBagConstraints constraints;
    private JMConfig settings;                // Our settings
    private static final int NEXT = 1;
    private static final int PREVIOUS = 0;
    private static ResReader reader;
    private boolean listenerSet = false;        // Have we set the listener for JTabbedPane if necessary
    private boolean useSwing = true;   // Whether the interface is swing-based or not
    private boolean useSwingMenu = false;       // Right now we have to use the heavyweight AWT menu bar
    
    private static final boolean DEBUG = false;
    
    /**
     * 
     * @param mainSettings 
     */
    public MuckMain(JMConfig mainSettings) {
        if (mainSettings.getJMboolean(JMConfig.USESWING)) {
            muckMainFrame = (Frame)new anecho.JamochaMUD.MainSwingFrame("JamochaMUD");
            useSwing = true;
            useSwingMenu = true;
        } else {
            useSwing = false;
            useSwingMenu = false;
            muckMainFrame = (Frame)new SyncFrame("JamochaMUD");
        }
        
        this.settings = mainSettings;
        settings.setJMObject(JMConfig.MUCKMAINFRAME, muckMainFrame);
        
        reader = new ResReader();       // Initialise our ResReader to handle our language issues
        
        // Create the windows
        
        muckMainFrame.addComponentListener(this);
        muckMainFrame.addWindowListener(this);
        muckMainFrame.addKeyListener(this);
        
        pauseStatus = false;
        
        if (useSwingMenu) {
            JMainMenu swingMenu = JMainMenu.getInstance();        // Trying out the JMTFKeys singleton
            swingMenu.buildMenu((javax.swing.JFrame)muckMainFrame, this);
        } else {
            anecho.JamochaMUD.legacy.MainMenu mainMenu = anecho.JamochaMUD.legacy.MainMenu.getInstance();
            mainMenu.buildMenu(muckMainFrame, this, settings);
        }
        
        MediaTracker tracker = new MediaTracker(muckMainFrame);
        Class testC = this.getClass();
        Image jMUDImage = Toolkit.getDefaultToolkit().getImage(testC.getResource("kehza.gif"));
        tracker.addImage(jMUDImage, 0);
        try {
            tracker.waitForAll();
        } catch (Exception except) {
            if (DEBUG) {
                System.err.println("Icon load error.  Exception " + except);
            }
        }
        muckMainFrame.setIconImage(jMUDImage);
    }
    
    // This supports multi-line messages
    private static Vector RBL(final String itemTarget) {
        return reader.langVector(JMConfig.BUNDLEBASE, itemTarget);
    }
    
    private static String[] RBA(final String itemTarget) {
        return reader.langArray(JMConfig.BUNDLEBASE, itemTarget);
    }
    
    /**
     * 
     * @param event 
     */
    public void actionPerformed(final ActionEvent event){
        
        final String arg = event.getActionCommand();
        
        if (arg.equals(java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("nextMU"))) {
            advanceMU(NEXT);
        }
        
        if (arg.equals(java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("previousMU"))) {
            advanceMU(PREVIOUS);
        }
        
        // Arguments for the ***FILE**** menu
        
        // Check for 'Dump Output'
        if (arg.equals(java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("dumpOutput"))) {
            dumpOutput();
        }
        
        if (arg.equals(java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("quit"))) {
            // Do the proper exit proceedure
            quitJamochaMUD();
            
            // Close the JVM
            System.exit(1);  // Probably an ugly exit
        }
        
        // Arguments for the **EDIT** menu
        
        /** Copy */
        if (arg.equals(java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("copyFromMainWindow"))) {
            copyToClipboard();
        }
        
        /** Paste */
        if (arg.equals(java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("paste"))) {
            pasteFromClipboard();
        }
        
        /** Find */
        
        // Arguments for the **CONNECTION**** menu
        
        /** Connect to MU* */
        if (arg.equals(java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("connectToMU"))) {
            jmConnectToMU();
        }
        
        /** Reconnect to MU* */
        if (arg.equals(java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("reconnectToMU"))) {
            jmReconnectToMU();
        }
        
        /** Disconnect from MU* */
        if (arg.equals(java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("disconnectFromMU"))) {
            jmDisconnectFromMU();
        }
        
        /** Close the visible MU "window" */
        if (arg.equals(java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("closeThisView"))) {
            closeActiveWindow();
        }
        
        // Arguments for ****OPTIONS**** menu

        if (arg.equals("Configure JamochaMUD")) {
            configureJamochaMUD();
        }
        
        if (arg.equals(java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("fontsAndColours"))) {
            setColours();
        }
        
        if (arg.equals(java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("externalPrograms"))) {
            showExtProgDialogue();
        }
        
        if (arg.equals(java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("serverOptions"))) {
            showProxyDialogue();
        }
        
        if (arg.equals(java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("installPlugin"))) {
            installPlugIn();
        }
        
        // Arguments for ****HELP**** menu
        if (arg.equals(java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("help"))) {
            final StringBuffer tentativeURL = new StringBuffer(java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("jamochaMUDPages"));
            
            // Attempt to use BrowserLauncher!
            try {
                final String browser = settings.getJMString(JMConfig.BROWSER1);
                final String url = tentativeURL.toString();
                edu.stanford.ejalbert.BrowserLauncher.openURL(browser, url);
            } catch (Exception except) {
                // The was an error trying to launch the browser
                if (useSwing) {
                    javax.swing.JOptionPane.showMessageDialog(muckMainFrame,
                            RBA("configureBrowser"),
                            java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("configureExternalProgram"),
                            javax.swing.JOptionPane.ERROR_MESSAGE);
                } else {
                    Vector tempVector = new Vector(0, 1);
                    final anecho.gui.OKBox check = new anecho.gui.OKBox(muckMainFrame, java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("configureExternalProgram"), true);
                    tempVector = RBL("configureBrowser");
                    
                    for (int i = 0; i < tempVector.size(); i++) {
                        check.append((String)tempVector.elementAt(i));
                    }
                    
                    check.showCentered();
                }
            }
        }
        
        if (arg.equals(java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("tinyFugueKeyCodes"))) {
            final JMTFKeys tfKeys = JMTFKeys.getInstance();        // Trying out the JMTFKeys singleton
            tfKeys.showCommands(muckMainFrame, useSwing);
        }
        
        if (arg.equals(java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("aboutJamochaMUD"))) {
            showAboutBox();
        }
        
        if (arg.equals(java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("reportABug"))) {
            
            final String message = anecho.gui.AbstractMessageFormat.wrap(java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("reportABugMessage"));
            
            if (useSwing) {
                javax.swing.JOptionPane.showMessageDialog(muckMainFrame,
                        message,
                        java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("reportABug"),
                        javax.swing.JOptionPane.INFORMATION_MESSAGE);
            } else {
                final anecho.gui.OKBox shoot = new anecho.gui.OKBox(muckMainFrame, java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("reportABug"), true);
                shoot.append(message);
                
                shoot.show();
            }
        }
        
        // Check to see if we've had a request on the plugins menu
        if (arg.startsWith("plugin:") || arg.startsWith("enable:") || arg.startsWith("disable:") || arg.startsWith("description")) {
            
            final int col = arg.indexOf(':');
            
            final String testName = arg.substring(col + 1);
            
            // a request for the properties of a plugin has been made
            try {
                final int selected = EnumPlugIns.plugInName.indexOf(testName);
                final Object plugClass = EnumPlugIns.plugInClass.elementAt(selected);
                final String name = ((PlugInterface)plugClass).plugInName();
                
                if (arg.startsWith("plugin:")) {
                    ((PlugInterface)plugClass).plugInProperties();
                }
                
                if (arg.startsWith("description:")) {
                    if (DEBUG) {
                        System.err.println("Calling description");
                    }
                    
                    final String desc = anecho.gui.AbstractMessageFormat.wrap(((PlugInterface)plugClass).plugInDescription());
                    
                    if (useSwing) {
                        javax.swing.JOptionPane.showMessageDialog(muckMainFrame, desc, name, javax.swing.JOptionPane.INFORMATION_MESSAGE);
                    } else {
                        final anecho.gui.OKBox message = new anecho.gui.OKBox(muckMainFrame, desc);
                        message.setTitle(name);
                        message.showCentered();
                    }
                }
                
                if (arg.startsWith("enable:")) {
                    if (DEBUG) {
                        System.err.println("Calling enable");
                    }
                    
                    final ManagePlugins manplug = new ManagePlugins();
                    manplug.changePlugInState(name, true);
                }
                
                if (arg.startsWith("disable:")) {
                    if (DEBUG) {
                        System.err.println("Calling disable");
                    }
                    
                    final ManagePlugins manplug = new ManagePlugins();
                    manplug.changePlugInState(name, false);
                    
                }
                
            } catch (ArrayIndexOutOfBoundsException except) {
                if (DEBUG) {
                    System.err.println("Error trying to get plug-in options.");
                    System.err.println(except);
                    except.printStackTrace();
                }
                
                if (useSwing) {
                    javax.swing.JOptionPane.showMessageDialog(muckMainFrame, java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("noSettingsAvailable"));
                } else {
                    final anecho.gui.OKBox message = new anecho.gui.OKBox(muckMainFrame, java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("noSettingsAvailable"), true, java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("noPlugInSettings"));
                    message.showCentered();
                }
            }
        }
        
        if (event.getActionCommand().startsWith("ChangeMU:")) {
            // Make sure we tell the ConnHandler we're changing MU*s
            setActiveMU(event.getActionCommand());
            
            // This will show our active MU*
            setVisibleMU();
            
            // Update our connection menu
            updateConnectionMenu();
            
            // Update the Frame's title
            setWindowTitle();
            
            // Make sure the info shows up!
            // validate();
            muckMainFrame.validate();
        }
        
    }
    
    /**
     * Show a dialogue box to allow users to select a file
     * to dump all of the output to for logging/archiving, etc...
     */
    private void dumpOutput() {
        
        // Show the system dump dialogue
        // This will dump *all* the output to the selected file
        final String prevPath = settings.getJMString(settings.LOGPATH);
        
        final FileDialog dumpDialogue = new FileDialog(muckMainFrame, "Dump Output to file", 1);
        if (prevPath != null && !prevPath.equals("")) {
            dumpDialogue.setDirectory(prevPath);
        }
        dumpDialogue.show();
        
        try {
            final String fileName = dumpDialogue.getFile();
            
            // File has been selected
            // now write the output contents to it
            if (fileName != null) {
                final String filePath = dumpDialogue.getDirectory();
                final FileOutputStream outFile = new FileOutputStream((filePath + fileName), false);
                final PrintWriter out = new PrintWriter(outFile, true);
                final CHandler connHandler = settings.getConnectionHandler();
                
                // Strip the text so that only plain text is written
                if (useSwing) {
                    final anecho.gui.JMSwingText activeMU = connHandler.getActiveMUDSwingText();
                    out.println(anecho.gui.TextUtils.stripEscapes(activeMU.getText(), false));
                } else {
                    final JMText activeMU = connHandler.getActiveMUDText();
                    out.println(anecho.gui.TextUtils.stripEscapes(activeMU.getText(), false));
                }
                out.flush();
                outFile.close();
                settings.setJMValue(settings.LOGPATH, filePath);
            }
        } catch (Exception except) {
            // This occurs when there is a 'cancel' instead of a file selection
            // We'll just gracefully fall through
            dumpDialogue.dispose();
        }
        
    }
    
    /**
     * 
     * @param event 
     */
    public void keyPressed(final KeyEvent event){
        boolean autoFocus = false;
        
        if (useSwingMenu) {
            final JMainMenu swingMenu = JMainMenu.getInstance();        // Trying out the JMTFKeys singleton
            autoFocus = swingMenu.isAutoFocus();
        } else {
            final anecho.JamochaMUD.legacy.MainMenu mainMenu = anecho.JamochaMUD.legacy.MainMenu.getInstance();
            autoFocus = mainMenu.isAutoFocus();
        }
        
        if (!event.isAltDown() && !event.isControlDown()) {
            // Check to see if the user wants AutoFocus shift
            // if (!tWAutoFocus.getState()) {
            if (autoFocus) {
                transferFocus(event);
            } else {
                event.consume();
            }
        }
    }
    
    /**
     * 
     * @param event 
     */
    public void keyTyped(final KeyEvent event){}
    
    /**
     * 
     * @param event 
     */
    public void keyReleased(final KeyEvent event){}
    
    /**
     * 
     * @param event 
     */
    public void itemStateChanged(final ItemEvent event) {
        String arg = "";
        boolean itemState = false;
        
        if (useSwingMenu) {
            final Object target = event.getItem();
            arg = ((javax.swing.JCheckBoxMenuItem)target).getText();
        } else {
            arg = (String) event.getItem();
        }
        
        // Determine if our item is now enabled or disabled
        if (event.getStateChange() == ItemEvent.SELECTED) {
            itemState = true;
        }
        
        if (arg.equals(java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("autoFocusInput"))) {
            settings.setJMValue(settings.AUTOFOCUSINPUT, itemState);
        }
        
        if (arg.equals(java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("autoLogging"))) {
            settings.setJMValue(settings.AUTOLOGGING, itemState);
            
            // Tell our different MU*s to start/stop logging
            final CHandler connHandler = settings.getConnectionHandler();
            connHandler.setLogging(itemState);
        }
        
        if (arg.equals(java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("altFocus"))) {
            // settings.setAltFocus(tWAltFocus.getState());
            // settings.setJMValue(JMConfig.ALTFOCUS, tWAltFocus.getState());
            settings.setJMValue(JMConfig.ALTFOCUS, itemState);
        }
        
        /* We current have no MacroBar feature in this version of JamochaMUD */
        /*
        if (arg.equals(java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("showMacroBar"))) {
            settings.setMacroVisible(tWMacro.getState());
            // Hide/Show the macro bars
            // FIx this XXX
            // MuckConn.settings.setMacroVisible(tWMacro.getState());
            // MuckConn.jmMacros.setVisible(tWMacro.getState());
            if (tWMacro.getState()) {
                // Make the frame visible
                // MuckConn.jmMacros.setActiveState(true);
         
                if (tWSyncWindowsItem.getState()) {
                    // Fix this XXX
                    // MuckConn.jmMacros.setSync(true);
                } else {
                    // Fix this XXX
                    // MuckConn.jmMacros.setSync(false);
                }
            }
            // else {
            // // Try and call the subroutine from MuMacros
            //    MuckConn.jmMacros.setActiveState(false);
            //    MuckConn.jmMacros.setVisible(false);
            //}
            }
         */
        
        if (arg.equals(java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("localEcho"))) {
            settings.setJMValue(JMConfig.LOCALECHO, itemState);
        }
        
        if (arg.equals(java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("doubleBuffer"))) {
            setDoubleBuffer(itemState);
        }
        
        if (arg.equals(java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("antiAlias"))) {
            setAntiAliasing(itemState);
        }
        
        if (arg.equals(java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("Low_colour_display"))) {
            setLowColour(itemState);
        }
        
        if (arg.equals(java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("splitFrames"))) {
            settings.setJMValue(JMConfig.SPLITVIEW, itemState);
            // Toggle our layout
            if (viewStyle == SPLIT) {
                viewStyle = COMBINED;
                splitFramesItem.setState(false);
                tWAltFocus.setEnabled(false);
            } else {
                viewStyle = SPLIT;
                splitFramesItem.setState(true);
                tWAltFocus.setEnabled(true);
            }
            
            // Call for new layout
            setMainLayout();
        }
        
        if (arg.equals(java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("syncWindows"))) {
            
            // If the user is not using split frames then we can record this choice,
            // but we should not affect the true nature of the frames
            settings.setJMValue(JMConfig.SYNCWINDOWS, itemState);
            
            if (splitFramesItem.getState()) {
                // The frames are split, so we can go to town!
                setAllSync(itemState);
            }
            
        }
        
        if (arg.equals(java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("showTimers"))) {
            // settings.setTimersVisible(tWTimers.getState());
            // settings.setJMValue(JMConfig.TIMERSVISIBLE, tWTimers.getState());
            settings.setJMValue(JMConfig.TIMERSVISIBLE, itemState);
            // Changed the status of the timers
            // Fix this XXX
            // if (tWTimers.getState()) {
            // The timers have been enabled, start the timer thread
            // timerThread.setActiveState(true);
            // } else {
            // timerThread.setActiveState(false);
            // }
        }
        
        if (arg.equals(java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("useUnicode"))) {
            // Changed the Unicode option
            settings.setJMValue(JMConfig.USEUNICODE, itemState);
        }
        
        if (arg.equals(java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("releasePause"))) {
            settings.setJMValue(JMConfig.RELEASEPAUSE, itemState);
        }
        
        if (arg.equals(java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("tinyFugueKeys"))) {
            settings.setJMValue(settings.TFKEYEMU, itemState);
        }
    }
    
    
    
    /**
     * 
     * @param event 
     */
    public void windowActivated(final WindowEvent event) {
        // Restore the original title to the window
        // settings.setMainWindowIconified(false);
        settings.setJMValue(JMConfig.MAINWINDOWICONIFIED, false);
        setWindowTitle();
    }
    
    /**
     * 
     * @param event 
     */
    public void windowClosed(final WindowEvent event) {
    }
    
    /**
     * 
     * @param event 
     */
    public void windowClosing(final WindowEvent event) {
        // Do a proper shutdown proceedure
        quitJamochaMUD();
    }
    
    /**
     * 
     * @param event 
     */
    public void windowDeactivated(final WindowEvent event) {}
    
    /**
     * 
     * @param event 
     */
    public void windowDeiconified(final WindowEvent event) {
        callSpoolText();
        
        // The text window is visible
        textWindowStatus = true;
        settings.setJMValue(JMConfig.MAINWINDOWICONIFIED, false);
        setWindowTitle();
    }
    
    /**
     * 
     * @param event 
     */
    public void windowIconified(final WindowEvent event) {
        // Write the frame to a the Hashtable
        // settings.setMainWindowIconified(true);
        settings.setJMValue(JMConfig.MAINWINDOWICONIFIED, true);
        // settings.setMainWindow(this.getBounds());
        // settings.setJMValue(JMConfig.MAINWINDOW, this.getBounds());
        settings.setJMValue(JMConfig.MAINWINDOW, muckMainFrame.getBounds());
        if (DEBUG) {
            System.err.println("MuckMain.windowIconified set bounds to: " + muckMainFrame.getBounds());
        }
        
    }
    
    /**
     * 
     * @param event 
     */
    public void windowOpened(final WindowEvent event) {}
    
    // We'll track the movements of our window for when we have to
    // write it out to our .rc file  Each window for himself!!
    /**
     * 
     * @param event 
     */
    public void componentHidden(final ComponentEvent event) {
    }
    
    /**
     * 
     * @param event 
     */
    public void componentResized(final ComponentEvent event) {
        // Do we need anything in this method?  I blocked it out!  XXX
        // settings.setJMValue(JMConfig.MAINWINDOW, muckMainFrame.getBounds());
        // System.err.println("MuckMain.componentResized settings bounds to: " + muckMainFrame.getBounds());
    }
    
    /**
     * 
     * @param event 
     */
    public void componentMoved(final ComponentEvent event) {
        // settings.setMainWindow(this.getBounds());
        // settings.setJMValue(JMConfig.MAINWINDOW, this.getBounds());
        settings.setJMValue(JMConfig.MAINWINDOW, muckMainFrame.getBounds());
    }
    
    /**
     * 
     * @param event 
     */
    public void componentShown(final ComponentEvent event) {
    }
    
    public synchronized void setWindowTitle() {
        // This resets the window's title, depending if the
        // client is connected to a MU* or not
        if (DEBUG) {
            System.err.println("MuckMain.setWindowTitle starts");
        }
        
        final CHandler connection = settings.getConnectionHandler();
        if (DEBUG) {
            System.err.println("MuckMain.setWindowTitle checking for active connection");
        }
        
        final boolean active = connection.isActiveMUDConnected();
        
        if (DEBUG) {
            System.err.println("MuckMain.setWindowTitle got connection status");
        }
        
        if (active) {
            // Now test to see if this MU* output is paused or not
            // System.err.println("MuckMain.setWindowTitle calls connection.getActiveMUHandle");
            final MuSocket mSock = connection.getActiveMUHandle();
            
            if (mSock.isPaused()) {
                // This format is nice, but too long
                // this.setTitle(java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("outputPaused"));
                muckMainFrame.setTitle(java.util.ResourceBundle.getBundle("anecho/JamochaMUD/JamochaMUDBundle").getString("outputPaused"));
            } else {
                // this.setTitle(connection.getActiveTitle());
                muckMainFrame.setTitle(connection.getActiveTitle());
            }
        } else {
            // Connection is inactive, just have program's title
            muckMainFrame.setTitle("JamochaMUD - Not connected");
        }
        
        if (DEBUG) {
            System.err.println("MuckMain.setWindowTitle going to checkCloseMenuState");
        }
        
        checkCloseMenuState();
        
        if (DEBUG) {
            System.err.println("MuckMain.setWindowTitle finished.");
        }
    }
    
    /**
     * A specific instance of setting the MU* title if
     * the MU* is minimised
     * @param title 
     */
    public synchronized void setWindowTitle(final String title) {
        muckMainFrame.setTitle(title);
    }
    
    
    /**
     * Methods called when disconnected from the MUD/MUCK
     * Here, we reset some of the menu flags to the appropriate status
     * and give the user visual notification that they have been disconnected
     * (This is not always called just by the user's 'Disconnect' action)
     */
    public void disconnectMenu() {
        // This changes the flags on the MuckMain menu to 'disconnected'
        // Connect to MU*
        if (useSwingMenu) {
            final JMainMenu swingMenu = JMainMenu.getInstance();        // Trying out the JMTFKeys singleton
            swingMenu.setConnected(false);
        } else {
            // muckMainFrame.setMenuBar(tWMenuBar);
            final anecho.JamochaMUD.legacy.MainMenu mainMenu = anecho.JamochaMUD.legacy.MainMenu.getInstance();
            mainMenu.setConnected(false);
        }
    }
    
    /**
     * We've received notification from one of our connections that it
     * has either terminated.  We'll query the mu* and see if it is
     * the active one.  If so, we'll update our connection menu.
     */
    public void checkDisconnectMenu() {
        final CHandler connHandler = settings.getConnectionHandler();
        // MuSocket mu = connHandler.getActiveMUHandle();
        
        // If our active MU* is not connected, change the connection menu
        if (!connHandler.isActiveMUDConnected()) {
            disconnectMenu();
        }
    }
    
    /**
     * The user has chosen to connect to a MUD/MUCK, so we
     * set the menu items as appropriate, and give the a visual
     * identifier that we are actually trying to make the connection
     */
    public void connectMenu() {
        if (useSwingMenu) {
            final JMainMenu swingMenu = JMainMenu.getInstance();        // Trying out the JMTFKeys singleton
            swingMenu.setConnected(true);
        } else {
            // muckMainFrame.setMenuBar(tWMenuBar);
            final anecho.JamochaMUD.legacy.MainMenu mainMenu = anecho.JamochaMUD.legacy.MainMenu.getInstance();
            mainMenu.setConnected(true);
        }
        
    }
    
    /**
     * Set the 'pause' status on the Main window.
     * Usually done by a single mouse-click, it will stop the
     * text from scrolling, and set the DataBar's title to alert
     * the user that the text is paused
     */
    public static void pauseText() {
        // A single click signals a pause
        // This will pause the output window,
        // the incoming lines held in queue
        pauseStatus = true;
    }

    /**
     * @Deprecated
     */
    public static void refreshPlugInListener() {
        // First, remove the old listener
        // tWPlugInMenu.removeActionListener();
        
        // Now add the Listener to the new list
        // tWPlugInMenu.addActionListener();
    }
    
    /**
     * The user has chosen to connect to a MU*, so we hide all the
     * active windows and show the MuckConn again
     */
    private void jmConnectToMU() {
        final CHandler connHandler = settings.getConnectionHandler();
        if (DEBUG) {
            System.err.println("MuckMain.JMConnectToMU calling connHandler.connectToNewMU()");
        }
        
        connHandler.connectToNewMU();
        
        if (DEBUG) {
            System.err.println("MuckMain.JMConnectToMU complete.");
        }
    }
    
    /**
     * Disconnect from the MU, closing the socket and stopping
     * the thread that listens to that IP address
     */
    private void jmDisconnectFromMU() {
        final CHandler connHandler = settings.getConnectionHandler();
        // System.err.println("MuckMain.JMDisconnectFromMU calls connHandler.getActiveMUHandle.");
        final MuSocket muSock = connHandler.getActiveMUHandle();
        final String stamp = muSock.getTimeStamp() + "";
        connHandler.closeActiveMU(stamp);
        
        // Explicitly set the state of our menus
        setConnectionMenu();
        
    }
    
    /** Reconnect to the MU* we were just connected to! */
    private synchronized void jmReconnectToMU() {
        // We'll remove the current MU* after we grab the name and address
        // It's important to recycle
        final CHandler connHandler = settings.getConnectionHandler();
        
        if (DEBUG) {
            System.err.println("MuckMain.JMReconnectToMU calls CHandler.reconnectToMU()");
        }
        
        connHandler.reconnectToMU();
        
        if (DEBUG) {
            System.err.println("MuckMain.JMReconnectToMU complete");
        }
        
        // Reset titles and stuff
        updateStates();
        
    }
    
    /** Change the shown MU* to our new selection */
    public synchronized void setVisibleMU() {
        
        // Update the menu to show the appropriate MU* as being active
        updateConnectionMenu();
        
        // Make certain that our connection menu is up to date for this MU*
        setConnectionMenu();
        
        
        // Change the title on our data-entry window
        final DataIn inWindow = settings.getDataInVariable();
        inWindow.setWindowTitle();
    }
    
    /** Set the colouring for our components */
    private void setColours() {
        // Grab our settings for the current MU*
        Color fgColor = settings.getJMColor(settings.FOREGROUNDCOLOUR);
        Color bgColor = settings.getJMColor(settings.BACKGROUNDCOLOUR);
        final Font ourFont = settings.getJMFont(settings.FONTFACE);
        
        // Set up our dialogue and display it
        final FontFace fFace = new FontFace(muckMainFrame, fgColor, bgColor, ourFont);
        fFace.setLocation(anecho.gui.PosTools.findCenter(muckMainFrame, fFace));
        fFace.setResizable(false);
        fFace.setVisible(true);
        // Now querry our dialogue and get our new settings
        final Font newStyle = fFace.getFontStyle();
        if (newStyle == null) {
            // The user cancelled, no changes made
            return;
        }
        
        fgColor = fFace.getForegroundColour();
        bgColor = fFace.getBackgroundColour();
        
        // Set up the other components with the proper colour
        // Change the colour/style in our settings
        settings.setJMValue(settings.FONTFACE, newStyle);
        settings.setJMValue(settings.FOREGROUNDCOLOUR, fgColor);
        settings.setJMValue(settings.BACKGROUNDCOLOUR, bgColor);
        
        // Change the colours/styles of our input window
        final JMUD core = settings.getJMCore();
        core.setAllFonts(newStyle);
        core.setAllColours(fgColor, bgColor);
    }
    
    private void showAboutBox() {
        if (settings.getJMboolean(JMConfig.USESWING)) {
            final AboutBox2 temp = new AboutBox2(muckMainFrame);
            temp.setVisible(true);
        } else {
            final AboutBox temp = new AboutBox(muckMainFrame);
            temp.setVisible(true);
        }
    }
    
    private void showExtProgDialogue() {
        final ExternalProgs extP = new ExternalProgs(muckMainFrame);
        extP.setVisible(true);
    }
    
    private void showProxyDialogue() {
        final ProxyBox pSettings = new ProxyBox(muckMainFrame, settings);
        pSettings.pack();
        pSettings.setLocation(PosTools.findCenter(muckMainFrame, pSettings));
        pSettings.show();
    }
    
    /** Perform all the necessary steps to shut down JamochaMUD,
     * most importantly being to save the settings from this session */
    private synchronized void quitJamochaMUD() {
        // "Shut down" and save the details of our other components
        if (useSwing) {
            // Do we need to find a better location for this?  Fix Me XXX
            final int divLoc = ((MainSwingFrame)muckMainFrame).getDividerLocation();

            settings.setJMValue(JMConfig.DIVIDERLOCATION, divLoc);
        }
        
        final JMUD master = settings.getJMCore();
        master.quitJamochaMUD();
    }
    
    private void setAllSync(final boolean state) {
        final JMUD core = settings.getJMCore();
        core.setAllSync(state);
    }
    
    /** transfer focus to our data entry window, taking into consideration
     * an special events that may also need to be passed along. */
    private void transferFocus(final KeyEvent event) {
        // System.err.println("Attempting to write to DataIn: " + event);
        final DataIn target = settings.getDataInVariable();
        
        if (event.getKeyCode() == KeyEvent.VK_ENTER) {
            // The user hit 'ENTER', send the text out
            target.jMSendText();
        }
        
        target.jmGainFocus();
        
        if (event.getKeyCode() == KeyEvent.VK_ENTER) {
            target.jMSendText();
            event.consume();
            return;
        }
        
        if (event.getKeyCode() != KeyEvent.VK_SHIFT && event.getKeyCode() != KeyEvent.VK_CONTROL) {
            if (!event.isControlDown()) {
                target.append(event.getKeyChar() + "");
                // Using keyTyped instead of keyPressed checks our "spool" condition
                target.keyTyped(event);
            }
            event.consume();
        }
    }
    
    /** Copy the selected text from our active MU
     * to the system clipboard */
    private void copyToClipboard() {
        final Clipboard clip = muckMainFrame.getToolkit().getSystemClipboard();
        boolean releasePause = false;
        
        try {
            StringSelection selection = new StringSelection("");
            String str = "";
            final CHandler connHandler = settings.getConnectionHandler();
            
            if (useSwing) {
                final anecho.gui.JMSwingText text = connHandler.getActiveMUDSwingText();
                str = text.getSelectedText();
                text.select(0, 0);
                final JMainMenu swingMenu = JMainMenu.getInstance();        // Trying out the JMTFKeys singleton
                releasePause = swingMenu.isReleasePause();
            } else {
                final JMText text = connHandler.getActiveMUDText();
                str = text.getSelectedText();
                
                // After a copy we should make certain to "unhighlight" our copied area
                text.select(0, 0);
                final anecho.JamochaMUD.legacy.MainMenu mainMenu = anecho.JamochaMUD.legacy.MainMenu.getInstance();
                releasePause = mainMenu.isReleasePause();
            }
            
            selection = new StringSelection(str);
            clip.setContents(selection, selection);
            
            if (releasePause) {
                pauseStatus = false;
                callSpoolText();
            }
        } catch (Exception except) {
            if (DEBUG) {
                System.err.println("Copy exception in MuckMain." + except);
                except.printStackTrace();
            }
        }
        
    }
    
    /** Paste the contents of our system clipboard
     * into the data-entry window */
    private void pasteFromClipboard() {
        // Clipboard c = this.getToolkit().getSystemClipboard();
        final Clipboard clip = muckMainFrame.getToolkit().getSystemClipboard();
        try {
            final Transferable contents = clip.getContents(this);
            final String str = (String)contents.getTransferData(DataFlavor.stringFlavor);
            
            // Now we'll stick it into the dataText-thing
            final DataIn dataBox = settings.getDataInVariable();
            final StringBuffer buff = new StringBuffer(dataBox.getText());
            final int caretPos = dataBox.getCaretPosition();
            buff.insert(caretPos, str);
            dataBox.setText(buff.toString());
        } catch (Exception except) {
            if (DEBUG) {
                System.err.println("Paste exception in MuckMain " + except);
            }
        }
    }
    
    /** Update the connection menu with the proper
     * number of MU's, indicating connected and disconnected
     * status in addition to which is our active MU*
     */
    private void updateConnectionMenu() {
        
        if (useSwingMenu) {
            final JMainMenu swingMenu = JMainMenu.getInstance();
            swingMenu.updateConnectionMenu(this);
        } else {
            // muckMainFrame.setMenuBar(tWMenuBar);
            final anecho.JamochaMUD.legacy.MainMenu mainMenu = anecho.JamochaMUD.legacy.MainMenu.getInstance();
            mainMenu.updateConnectionMenu();
        }
        
    }
    
    /**  Set the state of our connection menu
     * based on our active MU*'s condition
     */
    public void setConnectionMenu() {
        final CHandler connection = settings.getConnectionHandler();
        
        if (connection.isActiveMUDConnected()) {
            connectMenu();
        } else {
            disconnectMenu();
        }
        
        // Update the list of active/inactive MU*s
        updateConnectionMenu();
    }
    
    /** Clear all the plugins from our current menu
     */
    public void removeAllPlugins() {

        if (useSwingMenu) {
            final JMainMenu swingMenu = JMainMenu.getInstance();        // Trying out the JMTFKeys singleton
            swingMenu.removeAllPlugins();
        } else {
            final anecho.JamochaMUD.legacy.MainMenu mainMenu = anecho.JamochaMUD.legacy.MainMenu.getInstance();
            mainMenu.removeAllPlugins();
        }
        
    }
    
    /**
     * Add a new plugin to our plugins menu
     * @param plug 
     */
    // public void addPlugin(String name, boolean state) {
    public void addPlugin(final PlugInterface plug) {
        
        if (useSwingMenu) {
            final JMainMenu swingMenu = JMainMenu.getInstance();        // Trying out the JMTFKeys singleton
            // swingMenu.addPlugin(name, state);
            // swingMenu.addPlugin(itemName, state);
            swingMenu.addPlugin(plug);
        } else {
            final anecho.JamochaMUD.legacy.MainMenu mainMenu = anecho.JamochaMUD.legacy.MainMenu.getInstance();
            // mainMenu.addPlugin(name, state);
            // mainMenu.addPlugin(itemName, state);
            mainMenu.addPlugin(plug);
        }
        
    }
    
    /**
     * This method is used to make sure all the classes
     * properly identify the active MU*
     * @param muName 
     */
    public void setActiveMU(final String muName) {
        // First we take apart our String to get the MU number
        int muNum, split;
        split = muName.indexOf(':');
        split++;
        muNum = Integer.parseInt(muName.substring(split));
        
        final CHandler connHandler = settings.getConnectionHandler();
        connHandler.setActiveMU(muNum);
        
        checkCloseMenuState();
    }
    
    // public synchronized void setMainLayout() {
    public void setMainLayout() {
        final CHandler connHandler = settings.getConnectionHandler();
        
        // To test we're going to avoid the rest of the method altogether!
        // Fix Me XXX
        if (useSwing) {
            // Add the text entry area into our Split-pane
            final DataIn input = settings.getDataInVariable();
            final javax.swing.JScrollPane tempText = (javax.swing.JScrollPane)(input.exportText());
            ((MainSwingFrame)muckMainFrame).setPaneComponent(tempText ,MainSwingFrame.BOTTOM);
            
            // Add the main tabbed panel to our Split-pane
            javax.swing.JTabbedPane tempPane;
            final Object textObj = settings.getJMObject(JMConfig.TEXTPANEL);
            
            if (DEBUG) {
                System.err.println("MuckMain.setMainLayout() successfully received our JTabbedPane.");
            }
            
            // We'll do the lazy initialisation here, as we can't guarantee that we'll have
            // a usable JTabbedPane by the time we get here.
            if (textObj == null) {
                if (DEBUG) {
                    System.err.println("MuckMain.setMainLayout() doing lazy initialisation of JTabbedPane");
                }
                
                tempPane = new javax.swing.JTabbedPane();
                settings.setJMValue(JMConfig.TEXTPANEL, tempPane);
                
                if (DEBUG) {
                    System.err.println("MuckMain.setMainLayout() JTabbedPane successfully initialised and stored.");
                }
            } else {
                // We have an existing tempPane, so we'll use it.
                tempPane = (javax.swing.JTabbedPane)textObj;
                settings.setJMObject(JMConfig.TEXTPANEL, tempPane);
            }
            ((MainSwingFrame)muckMainFrame).setPaneComponent(tempPane ,MainSwingFrame.TOP);
            
            final int divLoc = settings.getJMint(JMConfig.DIVIDERLOCATION);
            
            if (divLoc > 0) {
                ((MainSwingFrame)muckMainFrame).setDividerLocation(settings.getJMint(JMConfig.DIVIDERLOCATION));
            }

            return;
        }
        
        if (DEBUG) {
            System.err.println("MuckMain.setMainLayout...");
        }

        if (useSwing) {
            ((javax.swing.JFrame)muckMainFrame).getContentPane().setLayout(null);
        } else {
            muckMainFrame.setLayout(null);
        }
        
        final int style = viewStyle;
        final DataIn input = settings.getDataInVariable();
        mainBagLayout = new GridBagLayout();
        constraints = new GridBagConstraints();

        if (useSwing) {
            ((javax.swing.JFrame)muckMainFrame).getContentPane().setLayout(mainBagLayout);
        } else {
            muckMainFrame.setLayout(mainBagLayout);
        }
        
        
        if (style == COMBINED) {
            // Set the style for the combined window
            constraints.gridwidth = GridBagConstraints.REMAINDER;
            constraints.gridheight = 5;
            constraints.gridx = 0;
            constraints.gridy = 0;
            constraints.weightx = 1;
            constraints.weighty = 1;
            constraints.insets = new Insets(0, 0, 2, 0);
            constraints.fill = GridBagConstraints.BOTH;
            constraints.anchor = GridBagConstraints.CENTER;

            if (useSwing) {
                if (DEBUG) {
                    System.err.println("MuckMain.setMainLayout() trying to get JTabbedPane.");
                }
                
                javax.swing.JTabbedPane tempPane;
                final Object textObj = settings.getJMObject(JMConfig.TEXTPANEL);
                
                if (DEBUG) {
                    System.err.println("MuckMain.setMainLayout() successfully received our JTabbedPane.");
                }
                
                // We'll do the lazy initialisation here, as we can't guarantee that we'll have
                // a usable JTabbedPane by the time we get here.
                if (textObj == null) {
                    if (DEBUG) {
                        System.err.println("MuckMain.setMainLayout() doing lazy initialisation of JTabbedPane");
                    }
                    
                    tempPane = new javax.swing.JTabbedPane();
                    settings.setJMValue(JMConfig.TEXTPANEL, tempPane);
                    
                    if (DEBUG) {
                        System.err.println("MuckMain.setMainLayout() JTabbedPane successfully initialised and stored.");
                    }
                } else {
                    tempPane = (javax.swing.JTabbedPane)textObj;
                }
                
                mainBagLayout.setConstraints(tempPane, constraints);  // Ugly code Fix me XXX
                ((javax.swing.JFrame)muckMainFrame).getContentPane().add(tempPane);
                
                if (!listenerSet) {
                    
                    tempPane.addChangeListener(new javax.swing.event.ChangeListener() {
                        public void stateChanged(javax.swing.event.ChangeEvent change) {
                            if (DEBUG) {
                                System.err.println("MuckMain stateChanged.  Calling checkCloseMenuState()");
                            }
                            
                            checkCloseMenuState();
                            
                            if (DEBUG) {
                                System.err.println("MuckMain stateChanged.  Calling updateConnectionMenu()");
                            }
                            
                            updateConnectionMenu();
                            // Should change Main Title here, too.  Fix Me XXX
                            if (DEBUG) {
                                System.err.println("MuckMain: Tab change event: " + change);
                            }
                            
                            final javax.swing.JTabbedPane src = (javax.swing.JTabbedPane)(change.getSource());
                            
                            if (DEBUG) {
                                System.err.println("MuckMain... we now have the JTabbedPane object.");
                            }
                            
                            if (src.isValid()) {
                                // We don't want to try and set the title if it
                                // isn't yet a valid object
                                if (DEBUG) {
                                    System.err.println("Our object is valid, so we will set the window title.");
                                }
                                setWindowTitle();
                            }
                        }
                    });
                    if (DEBUG) {
                        System.err.println("MuckMain setting listenerSet to true.");
                    }
                    
                    listenerSet = true;
                    
                }
                
            } else {
                anecho.JamochaMUD.legacy.JMTabPanel tempPane;
                final Object textObj = settings.getJMObject(JMConfig.TEXTPANEL);

                if (textObj == null) {
                    tempPane = new anecho.JamochaMUD.legacy.JMTabPanel();
                    settings.setJMValue(JMConfig.TEXTPANEL, tempPane);
                } else {
                    tempPane = (anecho.JamochaMUD.legacy.JMTabPanel)textObj;
                }
                
                mainBagLayout.setConstraints(tempPane, constraints);  // Ugly code Fix me XXX
                muckMainFrame.add(tempPane);
            }
            
            constraints.gridwidth = GridBagConstraints.REMAINDER;
            constraints.gridheight = GridBagConstraints.REMAINDER;
            constraints.gridx = 0;
            constraints.gridy = 5;
            constraints.weighty = 0;
            constraints.insets = new Insets(0, 0, 0, 0);
            constraints.fill = GridBagConstraints.BOTH;
            constraints.anchor = GridBagConstraints.CENTER;
            // TextArea tempText = in.exportText();
            
            if (settings.getJMboolean(JMConfig.USESWINGENTRY)) {
                final javax.swing.JScrollPane tempText = (javax.swing.JScrollPane)(input.exportText());
                mainBagLayout.setConstraints(tempText, constraints);
                ((javax.swing.JFrame)muckMainFrame).getContentPane().add(tempText);
            } else {
                final TextArea tempText = (TextArea)(input.exportText());
                mainBagLayout.setConstraints(tempText, constraints);
                if (useSwing) {
                    ((javax.swing.JFrame)muckMainFrame).getContentPane().add(tempText);
                } else {
                    muckMainFrame.add(tempText);
                }
            }
            
        } else {
            constraints.gridwidth = GridBagConstraints.REMAINDER;
            constraints.gridheight = GridBagConstraints.REMAINDER;
            constraints.gridx = 0;
            constraints.gridy = 0;
            constraints.weightx = 1;
            constraints.weighty = 1;
            constraints.insets = new Insets(0, 0, 0, 0);
            constraints.fill = GridBagConstraints.BOTH;
            constraints.anchor = GridBagConstraints.CENTER;
            mainBagLayout.setConstraints(connHandler.getTextObject(), constraints);

            if (useSwing) {
                ((javax.swing.JFrame)muckMainFrame).getContentPane().add(connHandler.getTextObject());
            } else {
                muckMainFrame.add(connHandler.getTextObject());
            }
            
            // Now make our input bar visible again
            input.restoreText();
            
        }
        
        mainBagLayout.invalidateLayout(muckMainFrame);
        mainBagLayout.layoutContainer(muckMainFrame);
        muckMainFrame.invalidate();
        muckMainFrame.validate();
        muckMainFrame.doLayout();
        
        
        // Should we try and absolute size the DataIn?
        // Fix this XXX - in the future we want the user to decide how many
        // lines tall our DataIn textarea is!
        input.setRows(3);
        
        viewStyle = style;
        
        updateConnectionMenu();
        settings.setJMValue(JMConfig.MAINLAYOUTVALID, true);
        
    }
    
    /** Remove the active view from our list of MU*s.  Since some Operating Systems
     * seem to synchronize things in ways that I hadn't foresee previously, we'll
     * have to do everything by the time-stamp of the MU, so we don't mess things up!
     */
    private void closeActiveWindow() {
        // remove the MU from our Cardlayout
        final CHandler connHandler = settings.getConnectionHandler();
        final MuSocket muSock = connHandler.getActiveMUHandle();
        
        closeWindow(muSock);
        // connHandler.removeActiveMU(stamp);
    }
    
    /** Close a window based on the passed MuSocket */
    private void closeWindow(final MuSocket muSock) {
        final CHandler connHandler = settings.getConnectionHandler();
        
        connHandler.removeMU(muSock);  // Causes layout problems somewhere down the line XXX
        
        updateStates();
    }
    
    /** Update the window title, set the visible MU* and make
     * certain that our connection menu is showing the proper states
     */
    private void updateStates() {
        setWindowTitle();
        setVisibleMU();
        
        checkCloseMenuState();
    }
    
    /** This method checks to see if the &quot;Close current view"
     * Menu item should be active or not
     */
    private void checkCloseMenuState() {
        // private synchronized void checkCloseMenuState() { // Synchronized causes **HANGS!!**
        if (DEBUG) {
            System.err.println("MuckMain.checkCloseMenuState()");
        }
        
        final CHandler connHandler = settings.getConnectionHandler();
        
        if (DEBUG) {
            System.err.println("MuckMain.checkCloseMenuState() got our connHandler: " + connHandler);
        }
        
        int totalConn = 0;
        boolean closeEnabled = false;
        boolean connectEnabled = false;
        
        if (DEBUG) {
            System.err.println("MuckMain.checkCloseMenuState() getting total connections.");
        }
        
        totalConn = connHandler.totalConnections();
        
        if (DEBUG) {
            System.err.println("MuckMain.checkCloseMenuState() total connections: " + totalConn);
        }
        
        if (connHandler.isActiveMUDConnected()) {
            connectEnabled = true;
        }
        
        if (DEBUG) {
            System.err.println("MuckMain.checkCloseMenuState() activeMUDConnected set to: " + connectEnabled);
        }
        
        if (useSwingMenu) {
            if (DEBUG) {
                System.err.println("MuckMain.checkCloseMenuState() working with Swing Menu.");
            }
            
            final JMainMenu swingMenu = JMainMenu.getInstance();
            
            if (totalConn > 1) {
                closeEnabled = true;
            }
            swingMenu.setCloseMUEnabled(closeEnabled);
            swingMenu.setConnected(connectEnabled);
            
        } else {
            if (DEBUG) {
                System.err.println("MuckMain.checkCloseMenuState() working with AWT Menu.");
            }
            final anecho.JamochaMUD.legacy.MainMenu mainMenu = anecho.JamochaMUD.legacy.MainMenu.getInstance();
            
            if (DEBUG) {
                System.err.println("MuckMain.checkCloseMenuState() checking totalConnections.");
            }
            
            if (totalConn > 1) {
                if (DEBUG) {
                    System.err.println("MuckMain.checkCloseMenuState() more than one connection.");
                }
                closeEnabled = true;
            } else {
                if (DEBUG) {
                    System.err.println("MuckMain.checkCloseMenuState() total connections 1 or less.");
                }
            }
            if (DEBUG) {
                System.err.println("MuckMain.checkCloseMenuState() setCloseMUEnabled to: " + closeEnabled);
            }
            
            mainMenu.setCloseMUEnabled(closeEnabled);
            
            if (DEBUG) {
                System.err.println("MuckMain.checkCloseMenuState() setConnected to: " + connectEnabled);
            }
            mainMenu.setConnected(connectEnabled);
        }
        
        if (DEBUG) {
            System.err.println("MuckMain.checkCloseMenuState() completed.");
        }
        
    }
    
    /** display the next or previous MU, depending on
     * the direction given
     */
    private void advanceMU(final int direction) {
        final CHandler handler = settings.getConnectionHandler();
        
        // change the active MU*
        if (direction == NEXT) {
            handler.nextMU();
        } else {
            handler.previousMU();
        }
        
        // now update our display
        setWindowTitle();
        setVisibleMU();
    }
    
    private void setDoubleBuffer(final boolean state) {
        // settings.setDoubleBuffer(state);
        settings.setJMValue(JMConfig.DOUBLEBUFFER, state);
        final CHandler handler = settings.getConnectionHandler();
        handler.setDoubleBuffer(state);
    }
    
    private void setAntiAliasing(final boolean state) {
        if (DEBUG) {
            System.err.println("MuckMain.setAntiAliasing to : " + state);
        }
        settings.setJMValue(JMConfig.ANTIALIAS, state);
        final CHandler handler = settings.getConnectionHandler();
        handler.setAntiAliasing(state);
        
        // Change our text entry area
        settings.getDataInVariable().setAntiAliasing(state);
    }
    
    private void setLowColour(final boolean state) {
        if (DEBUG) {
            System.err.println("MuckMain settings LOWCOLOUR to: " + state);
        }
        settings.setJMValue(JMConfig.LOWCOLOUR, state);
        final CHandler handler = settings.getConnectionHandler();
        handler.setLowColour(state);
    }
    
    /** Call the SpoolText proceedure that belongs to the correct JMText */
    private void callSpoolText() {
        final CHandler target = settings.getConnectionHandler();
        
        // Check to see if we actually have any connections before we try to spool text
        if (target.totalConnections() > 0) {
            
            final MuSocket mSock = target.getActiveMUHandle();
            if (mSock.isPaused()) {
                // This spools out any paused text
                mSock.spoolText();
            }
        }
    }
    
    private void installPlugIn() {
        final EnumPlugIns plugEnum = EnumPlugIns.getInstance();
        plugEnum.addNewPlugIn();
    }
    
    /** Rebuild the plug-in menu to reflect its current state.
     * This is usually called after activating/deactivating plug-ins
     */
    public void rebuildPlugInMenu() {
        if (DEBUG) {
            System.err.println("MuckMain.rebuildPlugInMenu() called.");
        }
        
        // Clear out all the old entries
        removeAllPlugins();
        
        // Cycle through our list of plug-ins and add them back to the menu
        final Vector pClasses = EnumPlugIns.plugInClass;
        
        final int pcs = pClasses.size();
        
        for (int i = 0; i < pcs; i++) {
            addPlugin((PlugInterface)pClasses.elementAt(i));
        }
        
    }
    
    /** Bring up the configuration dialogue box (Java2 only) */
    private void configureJamochaMUD() {
        final anecho.JamochaMUD.PrefDialogue prefD = new anecho.JamochaMUD.PrefDialogue(muckMainFrame, true);
        prefD.setVisible(true);
    }
}