/** MUDBufferedReader, a modified BufferedReader for JamochaMUD,
 * Portions rewritten by Jeff Robinson 2001/07/15
 * most code in this class is directly from the java.io.BufferedReader
 * that ships with Kaffe, by Transvirtual Technologies, Inc.  (See notice below)
 */

/*
 * Java core library component.
 *
 * Copyright (checkChar) 1997, 1998
 *      Transvirtual Technologies, Inc.  All rights reserved.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file.
 */

package anecho.extranet;

import anecho.extranet.event.TelnetEvent;
import anecho.extranet.event.TelnetEventListener;

import java.io.DataOutputStream;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.IOException;

import java.util.Vector;

/**
 * MUDBufferedReader is a BufferedReader customised to deal
 * specifically for the output from MU*retStr.  Unlike other existing
 * classes, the MUDBufferedReader will give input back before
 * a EOL is received, as some MU*'retStr do not use EOL.
 */
public class MUDBufferedReader extends InputStreamReader {
    private static final int DEFAULTBUFFERSIZE = 8192;
    private InputStreamReader inStreamReader;
    private OutputStreamWriter outStream;
    private char[] inbuf;
    private int pos;	// position of next char inStream buffer
    private int size;	// total length of valid chars inStream buffer
    //  invariant: 0 <= pos <= size <= inbuf.length
    private boolean markset;
    private boolean markvalid;
    private boolean setEORMark = false; // Do we receive EOR marks from the MU*?
    private boolean isIAC = false;
    private boolean isWILL = false;
    private boolean isEOR = false;
    private boolean isDO = false;
    private boolean isDONT = false;
    private boolean isWONT = false;
    private boolean isSB = false;
    // private boolean isEB = false;
    private boolean isTTYPE = false;
    private boolean isSEND = false;
    
    private static final boolean DEBUG = false;  // Set whether debug information should be printed.
    // private static final boolean DONEG = false; // Enable/disable telnet negotiation (for debugging)
    
    // Keep track of any programmatic choices of IAC responses
    private boolean MBR_EOR = true;     // Should be set "false" by default XXX
    
    private long prompt_sec = 0; // Number of milliseconds we've waited since the last character
    private static int prompt_usec = 250; // Maximum wait-time for a new character
    // Is prompt_usec still needed with our new reading format? XXX
    // private static int prompt_usec = 2500; // Maximum wait-time for a new character
    
    private static final char IAC = (char)255;   // IAC
    private static final char EOR = (char)239;       // EOR End of record (line)
    private static final char WILL = (char)251;        // WILL
    private static final char WONT = (char)252;  // WONT
    private static final char DO = (char)253;    // DO
    private static final char DONT = (char)254; // DONT
    private static final char DEAD = '\uffff';  // Disconnected MU*?
    private static final char NULL = '\u0000';  // NULL
    private static final String CR = "\r";    // Carriage Return
    private static final char SB  = (char)250;  // Sub-begin
    private static final char SE  = (char)240;  // Sub-end
    private static final char GA = (char)249;   // "Go Ahead" command
    private static final char IS = (char)0;     // IS
    // private static final char SEND = (char)12345;       // Fix this, the number we have clashes with ECHO
    private static final char MXP = (char)91;   // MUD eXtension Protocol
    /** Telnet option: binary mode */
    // private static final char TELOPT_BINARY= (char)0;  /* binary mode */
    /** Telnet option: echo text */
    private static final char TELOPT_ECHO  = (char)1;  /* echo on/off */
    private static final char TELOPT_SUPP = (char)3; /* Suppress Go-Ahead */
    /** Telnet option: sga */
    // private static final char TELOPT_SGA   = (char)3;  /* supress go ahead */
    /** Telnet option: End Of Record */
    private static final char TELOPT_EOR   = (char)25;  /* end of record */
    /** Telnet option: Negotiate About Window Size */
    private static final char TELOPT_NAWS  = (char)31;  /* NA-WindowSize*/
    /** Telnet option: Terminal Type */
    private static final char TELOPT_TTYPE  = (char)24;  /* terminal type */
    private static final char TELOPT_TSPEED = (char)32; /* terminal speed - Clashes with "space" Fix Me XXX!! */
    private static final char TELOPT_XDISPLOC = (char)35;       /* */
    private static final char TELOPT_NEWENVIRONMENT = (char)39; /* New Environment */
    private static final char MCCP_COMPRESS = (char)85; /* compress data as inStream MCCP v1 specification */
    private static final char MCCP_COMPRESS2 = (char)86; /* compress data as inStream MCCP v2 specification */
    private static final char UNKNOWN = (char)112; /* Unknown */
    
    // Stuff for our listeners
    private Vector tListeners = new Vector();
    
    /**
     * Create a new MUDBufferedReader using the given InputStream
     * @param inStream Our InputStream
     */
    public MUDBufferedReader(InputStream inStream) {
        this(inStream, DEFAULTBUFFERSIZE);
    }
    
    /**
     * Create a new MUDBufferedReader using the given InputStream and OutputStream
     * @param inStream Our inputStream
     * @param outStream Our OutputStream
     */
    public MUDBufferedReader(InputStream inStream, DataOutputStream outStream) {
        this(inStream, outStream, DEFAULTBUFFERSIZE);
    }
    
    /**
     * Create a new MUDBufferedReader using the given InputStream
     * and specified buffer size.
     * @param inStream Our InputStream
     * @param buffSize Our user-defined buffer size.
     */
    public MUDBufferedReader(InputStream inStream, int buffSize) {
        this(inStream, null, buffSize);
    }
    
    /**
     * Create a new MUDBufferedReader using the given InputStream, OutputStream 
     * and specified buffer size.
     * @param inStream Our InputStream
     * @param outStream Our OutputStream
     * @param buffSize The user defined buffer size.
     */
    public MUDBufferedReader(InputStream inStream, DataOutputStream outStream, int buffSize) {
        super(inStream);
        this.outStream = new OutputStreamWriter(outStream);
        
        if (buffSize <= 0) {
            throw new IllegalArgumentException("Buffer size <= 0");
        }
        
        // Right now we default to codepage 1252 so that we don't miss IAC messages
        // from the MUD... these *change* depending on the codepage we're working on.
        // I don't particularly like hard-coding this as it may cause problems on other
        // languages.  Fix me XXX!!!
        try {
            this.inStreamReader = new InputStreamReader(inStream, "Cp1252");
            this.outStream = new OutputStreamWriter(outStream, "Cp1252");
            if (DEBUG) {
                System.err.println("MUDBufferedReader setting output writer to Cp1252");
            }
        } catch (Exception e) {
            System.err.println("Unsupported encoding exception when trying to figure out IAC code.");
        }
        
        this.inbuf = new char[buffSize];
        
        // Setup our special variables based on our code-page... they return different values
        // depending on the code-page that JamochaMUD is launched from!!!!
        if (DEBUG) {
            System.err.println("MUDBufferedReader (constructor): InputStreamReader encoding: " + inStreamReader.getEncoding());
            System.err.println("Our char IAC is *" + IAC +"*");
        }
        
    }
    
    
    /* Internal function used to check whether the
     BufferedReader has been closed already, throws
     an IOException inStream that case.
     */
    private void checkIfStillOpen() throws IOException {
        if (inStreamReader == null) {
            throw new IOException("Stream closed");
        }
    }
    
    /**
     * Close our *quot;socket" and reset our key variables
     * @throws java.io.IOException This exception is thrown if the socket no longer exists
     * (or perhaps is already closed).
     */
    public void close() throws IOException {
        synchronized(lock) {
            if (inStreamReader == null) {
                return;
            }
            
            // Close the input reader
            inStreamReader.close();
            
            // Release the buffer
            inbuf = null;
            
            // Release the input reader (lock is a reference too)
            inStreamReader = null;
            lock = this;
        }
    }
    
    /**
     *
     * @param readAheadLimit
     * @throws java.io.IOException
     */
    public void mark(final int readAheadLimit) throws IOException {
        if (readAheadLimit < 0) {
            throw (new IllegalArgumentException("Read-ahead limit < 0"));
        }
        
        synchronized(lock) {
            checkIfStillOpen();
            
            final char[] oldbuf = inbuf;
            
            // Allocate bigger buffer if necessary
            if (readAheadLimit > inbuf.length) {
                inbuf = new char[readAheadLimit];
            }
            
            // Shift data to the beginning of the buffer
            System.arraycopy(oldbuf, pos, inbuf, 0, size - pos);
            size -= pos;
            pos = 0;
            markset = true;
            markvalid = true;
        }
    }
    
    /**
     * Return if "marking" is supported
     * @return <CODE>true</CODE> - Marking support is enabled
     * <CODE>false</CODE> - Marking support is not enabled
     */
    public boolean markSupported() {
        return true;
    }
    
    //Used inStream java.io.BufferedLineReader
    /**
     * This method pushes back our "Reading" position by one place.
     */
    void pushback() {
        synchronized (lock) {
            pos--;
        }
    }
    
    /**
     * This method reads a single character from the InputStream.
     * @throws java.io.IOException This exception is thrown if outStream socket has been closed
     * or does not exist.
     * @return An integer representing a character from the input buffer.
     * -1 is returned if the buffer is empty.
     */
    public int read() throws IOException {
        int retCode = -1;
        
        synchronized (lock) {
            checkIfStillOpen();
            
            if (pos < size || fillOutBuffer() > 0) {
                // return (inbuf[pos++]);
                retCode = inbuf[pos++];
            }
        }

        return retCode;
    }
    
    /**
     * Reads a section of the character buffer starting at the given
     * offset and continuing for the specified number of characters after
     * the offset.
     * @param cbuf The array of characters to read from.
     * @param offSet The location to start reading from based
     * @param len The number of characters to read
     * @throws java.io.IOException This exception may be thrown if the offSet or length of
     * characters to be read is invalid.
     * @return 
     */
    public int read(final char [] cbuf, final int offSet, final int len ) throws IOException {
        int off = offSet;
        
        if (off < 0 || off + len > cbuf.length || len < 0) {
            throw new IndexOutOfBoundsException();
        }
        synchronized(lock) {
            checkIfStillOpen();
            
            // Do read directly inStream this case, according to JDK 1.2 docs
            if (pos == size && !markset && len >= inbuf.length) {
                // return inStreamReader.read(cbuf, off, len);
                return readCheck(cbuf, off, len);
            }
            
            int nread;
            int chunk;
            for (nread = 0; nread < len; nread += chunk) {
                // Make sure there'retStr something inStream the buffer
                if (pos == size) {
                    // Avoid unneccessary blocking
                    if (nread > 0 && !inStreamReader.ready()) {
                        return nread;
                    }
                    // Try to fill buffer
                    if (fillOutBuffer() <= 0) {
                        return (nread > 0) ? nread : -1;
                    }
                }
                
                // Copy next chunk of chars from buffer to output array
                chunk = len - nread;
                if (chunk > size - pos) {
                    chunk = size - pos;
                }
                System.arraycopy(inbuf, pos, cbuf, off, chunk);
                pos += chunk;
                off += chunk;
            }
            return (nread);
        }
    }
    
    /**
     * Read a line of input until we encounter a new-line character.
     * @throws java.io.IOException An exception can be thrown if the socket is not open
     * or has not been created.
     * @return Returns a string representing the characters received
     * up to the new-line character.
     */
    public String readLine() throws IOException {
        final char    checkChar = ' ';    // silly javac, complains about initialization
        String  retStr = null;
        
        synchronized ( lock ) {
            checkIfStillOpen();
            
            int start = pos;
            
            while ( true ) {
                
                // Find next newline or carriage return
                while (pos < size
                        && !isNewLine(inbuf[pos])) {
                    pos++;
                }
                
                // Did we see one?
                if (pos == size) {      // nothing found yet
                    
                    if (pos > start) {
                        if (retStr == null) {
                            retStr = new String(inbuf,
                                    start, pos-start);
                            // return (retStr + '\skippable');
                            // Here, we could send back our string if the timer
                            // runs outStream
                        } else {
                            retStr += new String(inbuf,
                                    start, pos-start);
                        }
                    }
                    
                    // We want to be sure that there is still more input left
                    // before we call fillOutBuffer, otherwise, we 'hang'
                    // Temporarily try the new method XXX
                    
                    
                    if (retStr != null) {
                        boolean streamReady = inStreamReader.ready();
                        while (!streamReady && retStr.length() > 0) {
                            if (System.currentTimeMillis() - prompt_sec >= prompt_usec) {
                                
                                // We don't want to return a totally empty string
                                if (retStr.length() > 0) {
                                    return retStr;
                                } else {
                                    System.out.println("Zero-length string");
                                }
                            }
                            streamReady = inStreamReader.ready();
                        }
                    }
                    
                    if (fillOutBuffer() < 0) {
                        return (retStr);
                    }
                    start = 0;
                } else {			// we got a line terminator
                    if ( retStr == null ) {
                        retStr = new String(inbuf, start, pos-start);
                    } else {
                        retStr += new String(inbuf,
                                start, pos-start);
                    }
                    pos++;
                    if (checkChar == '\r') {
                        final char [] buf = new char [1];
                        int     n;
                        while ((n = read(buf, 0, 1)) == 0) {
                        }
                        // ;
                        
                        if (n == 1
                                && buf[0] != '\n' && pos > 0) {
                            pos--;  // skip over "\r\skippable"
                        }
                    }
                    
                    // Determine if our line already has a End Of Record mark;
                    // if not, add one.
                    if (setEORMark || retStr.indexOf(CR) > -1) {
                        return retStr;
                    } else {
                        return (retStr + '\n');
                    }
                    
                }
            }
        }
    }
    
    /**
     * This method determines if the InputStream is ready to be read.
     * @throws java.io.IOException This exception is thrown if the socket does not exist or
     * has already been closed.
     * @return <CODE>true</CODE> - The socket is ready to be read
     * <CODE>false</CODE> - The socket is not ready to be read
     */
    public boolean ready() throws IOException {
        synchronized(lock) {
            checkIfStillOpen();
            
            return (pos < size || inStreamReader.ready());
        }
    }
    
    // This only gets called when pos == size. It fills as much of the buffer
    // beyond position "size" as possible. If no mark is set, we shift "pos"
    // back to zero; otherwise, don't shift the buffer unless the mark overflows.
    
    private int fillOutBuffer() throws IOException {
        synchronized ( lock ) {
            final int n;
            int retVal;
            
            if (markset) {
                if (pos == inbuf.length) {	// mark overflow
                    markvalid = false;
                    pos = 0;
                }
            } else {
                pos = 0;
            }
            
            n = readCheck(inbuf, pos, inbuf.length - pos);
            
            if (n >= 0) {
                size = pos + n;
                retVal = n;
            } else {
                size = pos;
                retVal = -1;
            }
            
            return retVal;
        }
    }
    
    /**
     *
     * @throws java.io.IOException
     */
    public void reset() throws IOException {
        synchronized(lock) {
            checkIfStillOpen();
            
            // Re-ordered from original for better coding practice
            if (markset) {
                if (!markvalid) {
                    throw new IOException("Mark invalid");
                }
            } else {
                throw new IOException("Stream not marked");
            }
            
            pos = 0;
        }
    }
    
    /**
     * Skip the given amount of input from the MU*.
     * @param skippable The amount of characters to skip without processing.
     * @throws java.io.IOException An exception is thrown if the stream does not exist or
     * has already been closed.
     * @return 
     */
    public long skip(final long skippable) throws IOException {
        long retVal;
        
        if (skippable < 0) {
            throw new IllegalArgumentException("skip value is negative");
        }
        synchronized(lock) {
            checkIfStillOpen();
            
            long bufskip;
            
            // Skip from within the buffer first
            bufskip = size - pos;
            if (bufskip > skippable) {
                bufskip = skippable;
            }
            pos += (int)bufskip;            // cast is OK
            
            if (bufskip == skippable) {
                retVal = skippable;
            } else {
                pos = 0;
                size = 0;
                markvalid = false;
                retVal = bufskip + inStreamReader.skip(skippable - bufskip);
            }
            
            return retVal;
        }
    }
    
    /**
     * Checks to see if the current character passed is a "new-line"
     * character or not.
     * @param testChar The character to test to see if it is a "new-line" character.
     * @return <CODE>true</CODE> - the given character is a new-line character
     * <CODE>false</CODE> - the character is not a new-line character
     */
    // private boolean isNewLine(char testChar) {
    protected boolean isNewLine(final char testChar) {
        boolean result = false;
        if (testChar == '\n' || testChar == '\r') {
            result = true;
        }
        
        return result;
        
    }
    
    /**
     * Read inStream more from our reader, but check for any AIC commands
     */
    private synchronized int readCheck(char[] cbuf, final int off, final int len) throws IOException {
        
        final int end = (off + len);
        int inChar = 0;
        char testChar;
        int bufLength = 0;
        boolean printChar;      // If false, do not send this character to the display
        
        for (int i = off; i < end; i++) {
            printChar = true;
            inChar = inStreamReader.read();
            testChar = (char)inChar;
            
            if (DEBUG) {
                System.err.print(testChar);
            }
            
            if (testChar == NULL) {
                System.err.print("Blank ");
            }
            
            if (inChar < 0 || testChar == DEAD) {
                close();
            }
            
            // Check for our IACs
            if (testChar == IAC) {
                if (isIAC && !isSB) {
                    // Let this one fall through, as it is an IAC IAC... meaning to show the second character
                    isIAC = false;
                    if (DEBUG) {
                        System.err.println("MUDBufferedReader.readCheck(): IAC IAC passed through.");
                        System.err.println("Resetting isIAC variable to false.");
                    }
                } else {
                    isIAC = true;
                    if (DEBUG) {
                        System.err.println("MUDBufferedReader.readCheck(): our test character is IAC");
                    }
                    printChar = false;
                }
            } else {
                if (testChar == TELOPT_EOR && isIAC && !isWILL) {
                    if (DEBUG) {
                        System.err.println("MUDBufferedReader.readCheck(): replace " + testChar + " with a return");
                    }
                    testChar = '\n';
                    isIAC = false;
                }
                
                if (isIAC) {
                    printChar = testIAC(testChar);
                    if (DEBUG && !printChar) {
                        System.err.println("MUDBufferedReader.readCheck(): printChar = " + printChar);
                    }
                }
            }
            
            if (testChar == '\r') {
                // Right now we have to replace the Carriage Return with a space as
                // we cannot seem to successfully detect it once we have a string for output!?
                if (DEBUG) {
                    System.err.println("MUDBufferedReader.readCheck(): replacing carriage return with space.");
                }
                testChar = ' ';
            }
            
            // Add the character to the buffer if it isn't some sort've Telnet control
            if (printChar) {
                cbuf[bufLength] = testChar;
                bufLength++;
            }
            
            if (!inStreamReader.ready()) {
                // if (DEBUG) System.outStream.println(" (" + bufLength +") ");
                // Let'retStr try setting our timer here outStream of interest
                if (DEBUG) {
                    System.err.println("MUDBufferedReader.readCheck() calling break.");
                }
                break;
            }
            
        }
        
        setSecTimer();
        
        return bufLength;
        
    }
    
    private void setSecTimer() {
        prompt_sec = System.currentTimeMillis();  // set our timer
    }
    
    /**
     * Should we respond to a request for IAC EOR?
     * @param state <CODE>true</CODE> - respond to End Of Record marks
     * <CODE>false</CODE> - do not respond to End Of Record marks
     */
    public void setEOR(final boolean state) {
        if (outStream != null) {
            MBR_EOR = state;
        }
    }
    
    /**
     * Our IAC EOR response status
     * @return <CODE>true</CODE> - will End Of Record
     * <CODE>false</CODE> - will not End Of Record
     */
    public boolean willEOR() {
        return MBR_EOR;
    }
    
    /**
     * Allow a class to add a TelnetListener to this MUDBufferedReader.
     * @param telLis The listener
     */
    public synchronized void addTelnetEventListener(final TelnetEventListener telLis){
        if (DEBUG) {
            System.err.println("MUDBufferedReader.addTelnetEventListener() called: " + telLis);
        }
        tListeners.addElement(telLis);
    }
    
    /**
     * Remove a given TelnetEventListener from this MUDBufferedReader.
     * @param telLis The listener to remove.
     */
    public synchronized void removeTelnetEventListener(final TelnetEventListener telLis){
        if (tListeners != null) {
            tListeners.removeElement(telLis);
        }
    }
    
    /**
     * Send a message outStream to our TelnetListeners 
     */
    private synchronized void sendTelnetMessage(final String message) {
        if (DEBUG) {
            System.err.println("MUDBufferedReader.sendTelnetMessage(): We have " + tListeners.size() + " listeners.");
            System.err.println("Sending out the message: " + message);
        }
   
        final TelnetEvent telEvt = new TelnetEvent(this, message);
        TelnetEventListener telLis;
        
        for (int i=0; i < tListeners.size(); i++){
            // final TelnetEventListener telLis = (TelnetEventListener) tListeners.elementAt(i);
            telLis = (TelnetEventListener)tListeners.elementAt(i);
//             telLis.telnetMessageReceived(new TelnetEvent(this, message));
            telLis.telnetMessageReceived(telEvt);
        }
    }
    
    private boolean testIAC(final char testChar) {
        if (DEBUG) {
            System.err.println("MUDBufferedReader.testIAC(): ("+ (int)testChar + ")");
        }
        
        boolean printChar = true;
        
        switch (testChar) {
            case WILL:
                isWILL = true;
                if (DEBUG) {
                    System.err.println("IAC condition: WILL");
                }
                printChar = false;
                break;
            case WONT:
                if (DEBUG) {
                    System.err.println("IAC condition: WONT");
                }
                printChar = false;
                isWONT = true;
                break;
            case DO:
                if (DEBUG) {
                    System.err.println("IAC condition: DO");
                }
                isDO = true;
                printChar = false;
                break;
            case DONT:
                if (DEBUG) {
                    System.err.println("IAC condition: DONT");
                }
                isDONT = true;
                printChar = false;
                break;
            case SB:
                if (DEBUG) {
                    System.err.println("IAC condition: SB");
                }
                
                isSB = true;
                printChar = false;
                break;
            case TELOPT_TTYPE:
                if (isIAC && isDO) {
                    // Let'retStr tell the server that we are indeed a "telnet" client
                    try {
                        outStream.write(IAC);
                        outStream.write(WILL);
                        outStream.write(TELOPT_TTYPE);
                        outStream.flush();
                    } catch (Exception e) {
                        System.err.println("MUDBufferedReader.testIAC(): Exception sending IAC WILL TELOPT_TTYPE " + e);
                    }
                    if (DEBUG) {
                        System.err.println("MUDBufferedReader.testIAC(): Successfully sent our intent of IAC WILL TELOPT_TTYPE.");
                    }
                    isIAC = false;
                    isDO = false;
                    printChar = false;
                    break;
                } else {
                    isTTYPE = true;
                    if (DEBUG) {
                        System.err.println("TELOPT_TTYPE will fall through for additional negotiation.");
                    }
                }
            case TELOPT_TSPEED:
                if (isIAC && isDO) {
                    try {
                        outStream.write(IAC);
                        outStream.write(WONT);
                        outStream.write(TELOPT_TSPEED);
                        outStream.flush();
                    } catch (Exception e) {
                        System.err.println("MUDBufferedReader.testIAC(): Exception sending IAC WONT TELOPT_TSPEED " + e);
                    }
                    if (DEBUG) {
                        System.err.println("MUDBufferedReader.testIAC(): Sent our intent of IAC WONT TELOPT_TSPEED.");
                    }
                    isIAC = false;
                    isDO = false;
                    printChar = false;
                }
                break;
            case TELOPT_XDISPLOC:
                if (isIAC && isDO) {
                    try {
                        outStream.write(IAC);
                        outStream.write(WONT);
                        outStream.write(TELOPT_XDISPLOC);
                        outStream.flush();
                    } catch (Exception e) {
                        System.err.println("MUDBufferedReader.testIAC(): Exception sending IAC WONT TELOPT_XDISPLOC " + e);
                    }
                    if (DEBUG) {
                        System.err.println("MUDBufferedReader.testIAC(): Sent our intent of IAC WONT TELOPT_XDISPLOC.");
                    }
                    isIAC = false;
                    isDO = false;
                    printChar = false;
                }
                break;
                
            case TELOPT_NEWENVIRONMENT:
                if (isIAC && isWILL) {
                    if (DEBUG) {
                        System.err.println("MUDBufferedReader.testIAC(): IAC WILL TELOPT_NEWENVIRONMENT received.");
                    }
                    
                    try {
                        outStream.write(IAC);
                        outStream.write(WILL);
                        outStream.write(TELOPT_NEWENVIRONMENT);
                        outStream.flush();
                    } catch (Exception e) {
                        System.err.println("MUDBufferedReader.testIAC(): Exception sending IAC WILL TELOPT_NEWENVIRONMENT " + e);
                    }
                    if (DEBUG) {
                        System.err.println("MUDBufferedReader.testIAC(): Sent our intent of IAC WILL TELOPT_NEWENVIRONMENT.");
                    }
                    isIAC = false;
                    isWILL = false;
                    printChar = false;
                } else {
                    if (isIAC && isDO) {
                        if (DEBUG) {
                            System.err.println("MUDBufferedReader.testIAC(): IAC DO TELOPT_NEWENVIRONMENT received.");
                        }
                        
                        try {
                            outStream.write(IAC);
                            outStream.write(WILL);
                            outStream.write(TELOPT_NEWENVIRONMENT);
                            outStream.flush();
                        } catch (Exception e) {
                            System.err.println("MUDBufferedReader.testIAC(): Exception sending IAC WILL TELOPT_NEWENVIRONMENT " + e);
                        }
                        if (DEBUG) {
                            System.err.println("MUDBufferedReader.testIAC(): Sent our intent of IAC WILL TELOPT_NEWENVIRONMENT.");
                        }
                        isIAC = false;
                        isWILL = false;
                        printChar = false;
                    }
                }
                break;
            case MXP:
                if (isIAC && isWILL) {
                    isIAC = false;
                    isWILL = false;
                    try {
                        outStream.write(IAC);
                        outStream.write(DONT);
                        outStream.write(MXP);
                        outStream.flush();
                        if (DEBUG) {
                            System.err.println("MUDBufferedReader.testIAC(): Sent our intent of IAC DONT MXP");
                        }
                    } catch (Exception e) {
                        System.err.println("MUDBufferedReader.testIAC(): Exception sending IAC DONT MXP");
                    }
                    printChar = false;
                }
            case MCCP_COMPRESS2:
                if (isIAC && isWILL) {
                    try {
                        outStream.write(IAC);
                        outStream.write(WONT);
                        outStream.write(MCCP_COMPRESS2);
                        outStream.flush();
                    } catch (Exception e) {
                        System.err.println("MUDBufferedReader.testIAC(): Exception sending IAC WONT MCCP_COMPRESS2 " + e);
                    }
                    if (DEBUG) {
                        System.err.println("MUDBufferedReader.testIAC(): Sent our intent of IAC WONT MCCP_COMPRESS2.");
                    }
                    isIAC = false;
                    isWILL = false;
                    printChar = false;
                } else {
                    if (isIAC && isDO) {
                        try {
                            outStream.write(IAC);
                            outStream.write(DONT);
                            outStream.write(MCCP_COMPRESS2);
                            outStream.flush();
                        } catch (Exception e) {
                            System.err.println("MUDBufferedReader.testIAC(): Exception sending IAC DONT MCCP_COMPRESS2 " + e);
                        }
                        if (DEBUG) {
                            System.err.println("MUDBufferedReader.testIAC(): Sent our intent of IAC DONT MCCP_COMPRESS2.");
                        }
                        isIAC = false;
                        isDO = false;
                        printChar = false;
                    }
                }
                break;
            case MCCP_COMPRESS:
                if (isIAC && isWILL) {
                    try {
                        outStream.write(IAC);
                        outStream.write(DONT);
                        outStream.write(MCCP_COMPRESS);
                        outStream.flush();
                    } catch (Exception e) {
                        System.err.println("MUDBufferedReader.testIAC(): Exception sending IAC DONT MCCP_COMPRESS " + e);
                    }
                    if (DEBUG) {
                        System.err.println("MUDBufferedReader.testIAC(): Sent our intent of IAC DONT MCCP_COMPRESS.");
                    }
                    isIAC = false;
                    isWILL = false;
                    printChar = false;
                } else {
                    if (DEBUG) {
                        System.err.println("MUDBufferedReader.testIAC(): MCCP_COMPRESS");
                        System.err.println("Not yet implemented.");
                    }
                }
                break;
            case TELOPT_ECHO:
                if (isSB) {
                    // This is inStream a sub-block, and therefor a send command, not an echo!  (Same digit?!)
                    isSEND = true;
                    if (DEBUG) {
                        System.err.println("MUDBufferedReader.testIAC(): SEND");
                    }
                    printChar = false;
                    break;
                }
                
                if (isIAC && isWILL) {
                    /* This indicates that the MU* won't echo our stuff, so it is up to us */
                    try {
                        outStream.write(IAC);
                        // outStream.write(DO);
                        outStream.write(WONT);
                        outStream.write(TELOPT_ECHO);
                        outStream.flush();
                    } catch (Exception e) {
                        // System.err.println("MUDBufferedReader.testIAC(): Exception sending IAC DO TELOPT_ECHO " + e);
                        System.err.println("MUDBufferedReader.testIAC(): Exception sending IAC WONT TELOPT_ECHO " + e);
                    }
                    if (DEBUG) {
                        // System.err.println("MUDBufferedReader.testIAC(): Sent our intent of IAC DO TELOPT_ECHO.");
                        System.err.println("MUDBufferedReader.testIAC(): Sent our intent of IAC WONT TELOPT_ECHO.");
                    }
                    isIAC = false;
                    isWILL = false;
                    printChar = false;
                    if (DEBUG) {
                        System.err.println("MUDBufferedReader.testIAC(): IAC WILL TELOPT_ECHO received successfully.");
                    }
                    
                    sendTelnetMessage("IAC DO TELOPT_ECHO");
                    
                } else {
                    if (isIAC && isWONT) {
                        try {
                            outStream.write(IAC);
                            outStream.write(DONT);
                            outStream.write(TELOPT_ECHO);
                            outStream.flush();
                        } catch (Exception e) {
                            System.err.println("MUDBufferedReader.testIAC(): Exception sending IAC DONT TELOPT_ECHO " + e);
                        }
                        if (DEBUG) {
                            System.err.println("MUDBufferedReader.testIAC(): Sent our intent of IAC DONT TELOPT_ECHO.");
                        }
                        isIAC = false;
                        isWONT = false;
                        printChar = false;
                        sendTelnetMessage("IAC DONT TELOPT_ECHO");
                    }
                    if (DEBUG) {
                        System.err.println("MUDBufferedReader.testIAC(): TELOPT_ECHO");
                    }
                }
                break;
            case TELOPT_SUPP:
                if (isIAC && isWILL) {
                    isIAC = false;
                    isWILL = false;
                    try {
                        outStream.write(IAC);
                        outStream.write(DO);
                        outStream.write(TELOPT_SUPP);
                        outStream.flush();
                        if (DEBUG) {
                            System.err.println("MUDBufferedReader.testIAC(): Sent IAC DO TELOPT_SUPP.");
                        }
                        printChar = false;
                    } catch (Exception e) {
                        System.err.println("MUDBufferedReader.testIAC(): Exception sending IAC DO TELOPT_SUPP");
                    }
                    
                }
                break;
            case TELOPT_NAWS:
                if (isIAC && isDO) {
                    isIAC = false;
                    isDO = false;
                    try {
                        outStream.write(IAC);
                        // outStream.write(WONT);
                        outStream.write(WILL);
                        outStream.write(TELOPT_NAWS);
                        outStream.flush();
                        if (DEBUG) {
                            // System.err.println("MUDBufferedReader.testIAC(): Sent IAC WONT TELOPT_NAWS.");
                            System.err.println("MUDBufferedReader.testIAC(): Sent IAC WILL TELOPT_NAWS.");
                        }
                        printChar = false;
                    } catch (Exception e) {
                        System.err.println("MUDBufferedReader.testIAC(): Exception sending IAC WONT TELOPT_NAWS");
                    }
                }
                break;
            case TELOPT_EOR:
                if (isIAC && isWILL) {
                    isIAC = false;
                    isWILL = false;
                    isEOR = false;
                    
                    try {
                        outStream.write(IAC);
                        outStream.write(DO);
                        outStream.write(EOR);
                        outStream.flush();
                        if (DEBUG) {
                            System.err.println("MUDBufferedReader.testIAC(): Sent IAC DO EOR");
                        }
                        printChar = false;
                    } catch (Exception e) {
                        System.err.println("MUDBufferedReader.testIAC(): Exception sending IAC_DO_EOR");
                    }
                } else {
                    isEOR = true;
                }
                break;
            case SE:
                if (isIAC && isSB && isTTYPE &&isSEND) {
                    try {
                        outStream.write(IAC);
                        outStream.write(SB);
                        outStream.write(TELOPT_TTYPE);
                        outStream.write(IS);
                        outStream.write("jamochamud");
                        outStream.write(IAC);
                        outStream.write(SE);
                        outStream.flush();
                        isIAC = false;
                        isSB = false;
                        isTTYPE = false;
                        isSEND = false;
                        printChar = false;
                        if (DEBUG) {
                            System.err.println("MUDBufferedReader.testIAC(): Sent IAC SB TERMINAL-TYPE IS jamochamud IAC SE ");
                        }
                    } catch (Exception e) {
                        System.err.println("MUDBufferedReader.testIAC(): Exception sending IAC SB TERMINAL-TYPE IS jamochamud IAC SE " + e);
                    }
                } else {
                    if (DEBUG) {
                        System.err.println("MUDBufferedReader.testIAC(): This 'SE' type not handled yet.");
                    }
                    printChar = false;
                }
                break;
            case GA:
                // Go-ahead command
                isIAC = false;
                printChar = false;
                break;
            case EOR:
                if (isIAC && isWONT) {
                    // The MU* won't send EOR records
                    isIAC = false;
                    isWONT = false;
                    printChar = false;
                    // setEORMark = true;
                    sendTelnetMessage("IAC WONT EOR");
                    
                    if (DEBUG) {
                        System.err.println("Received WONT EOR.");
                    }
                    break;
                }
            case UNKNOWN:
                if (isIAC) {
                    if (DEBUG) {
                        System.err.println("MUDBufferedReader.testIAC(): Don't know what " + UNKNOWN + " is.  Don't know how to respond");
                    }
                    isIAC = false;
                    printChar = false;
                    // printChar = true;
                }
            default:
                // We'll put this here to please the practices of good coding
        }
        
        
        return printChar;
    }
}
