/**
 * Socks5socket.java; a class to handle Socks5 sockets, 
 * gratiously donated and Copyright(C) 2000 Bjoern Weber
 * $Id: Socks5socket.java,v 1.4 2001/05/26 18:00:03 jeffnik Exp $
 */

/* JamochaMUD, a Muck/Mud client program
 * Copyright (C) 1998-2000  Jeff Robinson
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

package anecho.extranet;

import java.io.*;
import java.net.*;

/**
 * Extension of the standard Socket class to enable connections through
 * a socks5 server.
 * <p>
 * Does NO authentification yet!
 * <p>
 * This class is a straightforward implementation of the SOCKS5 client protocol
 * as described in RFC....
 * Unfortunately it is just capable of identifying itself with username and
 * password since I was not able to find a Java GSSAPI and furthermore was
 * the administrator of the SOCKS server myself so I could configure it to
 * accept username password authentification.
 * <p>
 * <b>Usage:</b>
 * <pre>
 *	private void tconnect(String host, int port, boolean socks){
 *		Socket s=null;
 *		String buff=null;
 *		try{
 *			if (socks) s=(Socket)new Socks5socket(host, port);
 *			else s=new Socket(host, port);
 *			...
 *		}catch (IOException e){
 *			output.append("Connection refused.\n"+e+"\n");
 *		}finally{
 *			output.append("Connection closed.\n");
 *			if (s != null) try{s.close();}catch(IOException ce){;}
 *		}
 *	}
 * </pre>
 * 
 * @author Bjoern Weber
 * @version 0.5
 */
public class Socks5socket extends Socket {

    /**
     * the default SOCKS server address, unless passed
     * specifically by the constructor.
     */
    private static final String SOCKS5HOSTNAME = "socks";

	/**
	 * the default SOCKS server port, unless passed
	 * specifically by the constructor
	 */
	private static final int    SOCKS5PORT = 119;

	/**
	 * the remote hostname.
	 */
	private String              shost = null;

	/**
	 * the remote port.
	 */
	private int                 sport = 0;

	/**
	 * the standardconstructor.
	 * <p>
	 * This constructor feels exactly like a standard Socket, it just opens
	 * a connection to the SOCKS server first, negotiates the credentials
	 * and requests a connect to the remote host on success.
	 *
	 * @param hostname the name of the remote host to connect to.
	 * @param port the portnumber on the remote host to connect to.
	 * 
	 * @throws java.net.UnknownHostException if the host could not be resolved.
	 * @throws java.io.IOException if any error occured on connection.
	 *
         */
        public Socks5socket(String hostname, int port) throws java.net.UnknownHostException, java.io.IOException  {
            this(hostname, port, SOCKS5HOSTNAME, SOCKS5PORT);
        }

	/**
	 * 
	 * @param hostname 
	 * @param port 
	 * @param s5Host 
	 * @param s5Port 
	 * @throws java.net.UnknownHostException 
	 * @throws java.io.IOException 
	 */
	public Socks5socket(String hostname, int port, String s5Host, int s5Port)
			throws java.net.UnknownHostException, java.io.IOException {
		// Open a standardconnection to the SOCKS server.
		// This will be the connection we're talking to.
                super(s5Host, s5Port);

		// Allocate a bytebuffer for negotiation.
		byte[] octets = new byte[1024];

		debug("Connected to " + SOCKS5HOSTNAME);
		OutputStream S5OutStream = this.getOutputStream();
		InputStream  S5InStream = this.getInputStream();

		debug("Got IOStreams");
		// Set up the ID-String for our client.
		octets[0] = 5;    // SOCKS Version
		octets[1] = 3;    // Number of authentification methods
		octets[2] = 0;    // None - no authentification
		octets[3] = 1;    // GSSAPI - authentification via Kerberos (not implemented!)
		octets[4] = 2;    // Username/Passwd
		
		// Send the id to the SOCKS server
		S5OutStream.write(octets, 0, 5);
		debug("Sent ID request");

		// Recieve the answer from the server
		// The answer consists of two bytes, the version of the SOCKS server
		// (hopefully 5) and the requested authentification method.
		S5InStream.read(octets, 0, 2);
		debug("Recieved version " + octets[0]);
		switch (octets[1]) {
			case 0:
				debug("No authentification required");
				break;
			case 1:
				debug("Authentification via GSSAPI");
				throw new java.io.IOException("Socks5 GSSAPI not implemented");
			case 2:
				debug("Authentification via Username/Password");
				octets[0] = 5;    // Version

				// Hopefully the username is the one we find in the properties.
				// Eventually this should move to an extra property, together
				// with SOCKS server name and port.
				byte[] namebuff = System.getProperty("user.name").getBytes();

				octets[1] = (new Integer(namebuff.length)).byteValue();
				for (int i = 0; i < namebuff.length; i++) {
					octets[1 + i] = namebuff[i];
				}

				// The password... here just the word 'pass'. Apply the same 
				// thoughts here as found in SOCKS host/port and username.
				byte[] passbuff = "pass".getBytes();

				octets[1 + namebuff.length] = 
					(new Integer(passbuff.length)).byteValue();
				for (int i = 0; i < passbuff.length; i++) {
					octets[1 + namebuff.length + i] = passbuff[i];
				}
				
				// Send authentification to the server and cross fingers...
				S5OutStream.write(octets, 0, 
								  1 + namebuff.length + passbuff.length);
				debug("Sent " + System.getProperty("user.name") 
					  + " with passwd pass!");

				// Get the answer.
				S5InStream.read(octets, 0, 2);
				if (octets[1] != 0) {
					debug("Authentification failed with " + octets[1]);
					throw new java.io.IOException("U/P authentification failed!");
				} 
				break;
			case -1:
				debug("Authentification denied!");
				throw new java.io.IOException("Socks5 authentification denied!");
			default:
				debug("Unknown answer " + octets[1]);
				throw new java.io.IOException("Socks5 unknown authentification required!");
		}
		
		// After the handshake was completed, we'll ask for a connection to
		// the remote host.
		octets[0] = 5;    // Socks version
		octets[1] = 1;    // command: Connect
		octets[2] = 0;    // Reserved 0
		octets[3] = 3;    // Domainname is following:
		byte[] namebuff = hostname.getBytes();

		octets[4] = (new Integer(namebuff.length)).byteValue();
		for (int i = 0; i < namebuff.length; i++) {
			octets[5 + i] = namebuff[i];
		}

		// since the port may be 16Bit wide and we're working with bytes we 
		// need to split the port accordingly.
		int uport = port / 256;
		int lport = port % 256;

		debug("Portid: " + port + "=" + uport + "*256+" + lport);
		octets[5 + namebuff.length + 0] = (new Integer(uport)).byteValue();
		octets[5 + namebuff.length + 1] = (new Integer(lport)).byteValue();

		// request connection...
		S5OutStream.write(octets, 0, 5 + namebuff.length + 2);
		debug("Connection Request sent");

		// retrieve the answer
		S5InStream.read(octets);
		switch (octets[1]) {
			case 0:
				debug("Success!");
				break;
			case 1:
				debug("general failure");
				throw new java.io.IOException("Socks5 server failure");
			case 2:
				debug("connection refused by ruleset");
				throw new java.io.IOException("Socks5 refused by ruleset");
			case 3:
				debug("network unreachable");
				throw new java.io.IOException("Socks5 network unreachable");
			case 4:
				debug("host unreachable");
				throw new java.io.IOException("Socks5 host unreachable");
			case 5:
				debug("connection refused");
				throw new java.io.IOException("Socks5 connection refused");
			case 6:
				debug("TTL expired");
				throw new java.io.IOException("Socks5 TTL expired");
			case 7:
				debug("Command not supported");
				throw new java.io.IOException("Socks5 command not supported");
			case 8:
				debug("Address type not supported");
				throw new java.io.IOException("Socks5 address type not supported");
			default:
				debug("Unknown answer " + octets[1]);
				throw new java.io.IOException("Socks5 unknown error!");
		}
		// hopefully we now get an IP-address and a port which describes the
		// connection which is routed through the SOCKS server to our
		// destination.
		switch (octets[3]) {
			case 1:
				debug("IP V4 address");
				shost = unsign(octets[4]) + "." + unsign(octets[5]) + "." 
						+ unsign(octets[6]) + "." + unsign(octets[7]);
				sport = unsign(octets[8]) * 256 + unsign(octets[9]);
				break;
			case 3:
				debug("DOMAINNAME");
				throw new java.io.IOException("Socks5 Domainname encoding not implemented");
			case 4:
				debug("IP V6 address");
				throw new java.io.IOException("Socks5 IP V6 encoding not implemented");
			default:
				debug("Unknown hostname encoding!");
				throw new java.io.IOException("Socks5 server sent unknown hostname!");
		}
		debug("-> " + shost + ":" + sport);
	}

	/**
	 * Returns the IP address of the Socks server.
	 * <p>
	 * This just prints out the IP-Address that was returned after the
	 * request. Eventually we need this in a cascaded proxy environment
	 * since the SOCKS definition is capable of load balancing connections
	 * between different servers.
	 *
	 * @return the IP-address in a String.
	 *
	 */
	public String getHost() {
		return this.shost;
	} 

	/**
	 * Returns the portnumber of the Socks server.
	 * <p>
	 * This just prints out the portnumber that was returned after the
	 * request. Eventually we need this in a cascaded proxy environment
	 * since the SOCKS definition is capable of load balancing connections
	 * between different servers.
	 *
	 * @return the portnumber.
	 *
	 */
	public int getPort() {
		return this.sport;
	} 

	/**
	 * Internal routine to transform a signed (8-bit) Byte into
	 * an unsigned integer value.
	 *
	 * @param bval bytevalue
	 *
	 * @return the according unsigned representation.
	 *
	 */
	private int unsign(final byte bval) {
		int ival = (new Byte(bval)).intValue();
                
		if (ival < 0) {
			// return 256 + ival;
                    ival = ival + 256;
		}
//                else {
//			return ival;
//		}
                return ival;
	} 

	/**
	 * Internal debugging!
	 * <p>
	 * If you're sure that everything is alright, just comment the
	 * System.out.println() line.
	 *
	 * @param code Debugmessage
	 *
	 */
	private void debug(final String code) {
		System.out.println("S5: " + code);
	} 

}


