/* JMSwingEntry, a Swing-based text entry area with anti-aliasing effects
 * Copyright (C) 2005 Jeff Robinson
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

package anecho.gui;

import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.RenderingHints;

/**
 * JMSwingEntry is a Swing-based area used for inputting text.
 */
public class JMSwingEntry extends javax.swing.JTextArea implements java.awt.event.MouseListener {
    
    private boolean antiAlias = false;
    
    /**
     * JMSwingEntry is a Swing based component that allows a user to 
     * input text.
     */
    public JMSwingEntry() {
        super();
        addMouseListener(this);
    }
    
    /**
     * JMSwingEntry is a Swing based component that allows a user to 
     * input text.
     * @param doc Our document (used by the parent class)
     */
    public JMSwingEntry(javax.swing.text.Document doc)  {
        super(doc);
        addMouseListener(this);
    }
    
    /**
     * JMSwingEntry is a Swing based component that allows a user to 
     * input text.
     * @param doc Our document, used by the parent class.
     * @param text Text to be initially shown after our object is created
     * @param rows Number of rows for our component to display
     * @param columns Number of columns for our component to display.
     */
    public JMSwingEntry(javax.swing.text.Document doc, String text, int rows, int columns) {
        super(doc, text, rows, columns);
        addMouseListener(this);
    }
    
    /**
     * JMSwingEntry is a Swing-based area used for inputting text.
     * @param rows Number of rows to show in our component
     * @param columns Number of columns to show in our component
     */
    public JMSwingEntry(int rows, int columns) {
        super(rows, columns);
        addMouseListener(this);
    }
    
    /**
     * JMSwingEntry is a Swing-based area used for inputting text.
     * @param text Initial text to be displayed after our component is created
     */
    public JMSwingEntry(String text) {
        super(text);
        addMouseListener(this);
    }
    
    /**
     * JMSwingEntry is a Swing-based area used for inputting text.
     * @param text The initial text to be displayed after our component is created.
     * @param rows Number of rows for our component to show.
     * @param columns Number of columns for our component to show.
     */
    public JMSwingEntry(String text, int rows, int columns) {
        super(text, rows, columns);
        addMouseListener(this);
    }
    
    /**
     * This method allows anti-aliasing of text to be enabled or disabled
     * @param state true - Enable anti-aliasing on text (smooth characters)
     * false - Disable anti-aliasing on text
     */
    public void setAntiAliasing(final boolean state) {
        antiAlias = state;
    }
    
    /**
     * Over-ridden paintComponent proceedure to set the Rendering Hints
     * @param graphItem The Graphic object that we will render to.
     */
    public void paintComponent(final Graphics graphItem ) {
        
        final Graphics2D graph2D = (Graphics2D)graphItem;
        
        if (antiAlias) {
            graph2D.setRenderingHint( RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_ON );
        } else {
            graph2D.setRenderingHint( RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_OFF );
            graph2D.setRenderingHint( RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_OFF );
        }
        
        super.paintComponent(graphItem);
    }
    
    /**
     * Subclassed for proper cursor appearance
     * @param evt Our mouse event
     */
    public void mousePressed(final java.awt.event.MouseEvent evt) {
        this.setCaretColor(this.getForeground());
    }
    
    /**
     * Empty event
     * @param evt Mouse Event
     */
    public void mouseReleased(final java.awt.event.MouseEvent evt) {
        
    }
    
    /**
     * Empty event
     * @param evt mouse event
     */
    public void mouseEntered(final java.awt.event.MouseEvent evt) {
        
    }
    
    /**
     * Empty event
     * @param evt Mouse Event
     */
    public void mouseExited(final java.awt.event.MouseEvent evt) {
        
    }
    
    /**
     * Empty event
     * @param evt Mouse Event
     */
    public void mouseClicked(final java.awt.event.MouseEvent evt) {
        
    }
    
}

