/**
 * JMSwingText, a Swing-based text area for ANSI colours (originally developed for JamochaMUD)
 * $Id: JMSwingText.java,v 1.6 2004/07/28 02:43:42 jeffnik Exp $
 */

/* JMSwingText, a Swing-based text area for ANSI colours
 * Copyright (C) 2003-2005 Jeff Robinson
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

package anecho.gui;

import java.awt.AWTEvent;
import java.awt.AWTEventMulticaster;
import java.awt.Color;
import java.awt.Font;
// import java.awt.Graphics;
import java.awt.Graphics2D;
// import java.awt.RenderingHints;
import java.awt.SystemColor;
import java.awt.event.MouseListener;
import java.awt.event.MouseEvent;
import java.awt.event.KeyListener;
import java.awt.event.KeyEvent;

import javax.swing.JScrollPane;
// import javax.swing.JTextPane;
import javax.swing.text.DefaultStyledDocument;
import javax.swing.text.SimpleAttributeSet;
import javax.swing.text.StyleConstants;
import javax.swing.event.HyperlinkListener;
import javax.swing.event.HyperlinkEvent;

/**
 * JMSwingText, a Swing-based text area for ANSI colours (originally developed for JamochaMUD)
 */
public class JMSwingText extends JScrollPane implements MouseListener, KeyListener, HyperlinkListener {
    
    // private JTextPane tPane;
    private JMSwingTextPane tPane;
    private DefaultStyledDocument textDoc;
    // private javax.swing.text.html.HTMLDocument textDoc;
    private SimpleAttributeSet attr;
    private Color tempColour, fgColour, bgColour;
    private Font standardFont;  // Our basic font style
    private MouseListener mListener;	// A mouselistener
    private KeyListener keyL;
    
    private boolean paintBG = false;  // OS/2 JVM 1.1.8 + Swing screws up background painting
    private static final boolean DEBUG = false;
    
    private int totalLength;  // Total length of the current text area
    private boolean isBold = false;     // keep track of the "bold" state of our letters
    
    private boolean boldNotBright;  // set in our constructor
    private Color[] privPal;    // Mappings for our private palette
    public static final int PAGEUP = 0;
    public static final int PAGEDOWN = 1;
    
    Graphics2D graph2D; // Our 2D graphics item (for anti-aliasing support)
    
    /** The constructor for JMSwingText.
     * There is no guarantee that this component will be fully initialised
     * before a program calls some of its methods.  Therefore, we will
     * check our colours, fonts, etc to see if they have already been
     * set.  If not, we will set them to our defaults
     */
    public JMSwingText() {
        textDoc = new DefaultStyledDocument();
        // javax.swing.text.html.HTMLEditorKit kit = new javax.swing.text.html.HTMLEditorKit();
        // textDoc = (javax.swing.text.html.HTMLDocument) (kit.createDefaultDocument());
        attr = new SimpleAttributeSet();
        tPane = new JMSwingTextPane(textDoc);
        
        tPane.addMouseListener(this);
        tPane.addKeyListener(this);
        tPane.addHyperlinkListener(this);
        this.setViewportView(tPane);
        
        
        // tPane.setEditorKit(kit);
        // tPane.setContentType("text/html");
        
        if (DEBUG) {
            System.err.println("Our text area is using the EditorKit: " + tPane.getEditorKit());
        }
        
        // Set the standard font
        if (standardFont == null) {
            standardFont = new Font("SanSerif", Font.PLAIN, 16);
        }
        StyleConstants.setFontFamily(attr, "SansSerif");
        StyleConstants.setFontSize(attr, 16);
        
        // Background color
        if (bgColour == null) {
            // bgColour = new Color(0, 0, 0);
            bgColour = new Color(SystemColor.text.getRGB());
        }
        
        // Foreground color
        if (fgColour == null) {
            // fgColour = new Color(255, 255, 255);
            fgColour = new Color(SystemColor.textText.getRGB());
        }
        
        tPane.setBackground(bgColour);
        tPane.setForeground(fgColour);
        StyleConstants.setForeground(attr, fgColour);
        if (paintBG) {
            StyleConstants.setBackground(attr, bgColour);
        }
        
        this.setBoldNotBright(boldNotBright);
        
    }
    
    /**
     * Append text to our JMSwingText
     * Revised version of method follows
     * @param input The input to be added to the current JMSwingText area.
     */
    /*
    public synchronized void append(String input) {
     
        // int totalLength, newPos;
        int len = input.length();
        totalLength = textDoc.getLength();
     
        try {
            // This is an ugly hack-test to see if things can be sped up.
            // Fix this XXX!!!
            // Ideally, this should be extended to add "chunks" of Strings
            // that occur inbetween escapes.
            if (input.indexOf('\u001b') < 0) {
                textDoc.insertString(totalLength, input, attr);
                tPane.setCaretPosition(totalLength + len);
                return;
            }
     
            for (int i = 0; i < len; i++) {
                inputChar = input.charAt(i);
                // if (input.charAt(i) == '\u001b') {
                if (inputChar == '\u001b') {
                    // int newPos = checkEscape(input, i);
                    newPos = checkEscape(input, i);
                    i = newPos;
                } else {
                    if (DEBUG) {
                        // System.err.print(input.charAt(i));
                        System.err.print(inputChar);
                    }
                    // totalLength = textDoc.getLength();
                    // textDoc.insertString(textDoc.getLength(), input.charAt(i) + "", attr);
                    // textDoc.insertString(totalLength, input.charAt(i) + "", attr);
                    textDoc.insertString(totalLength, inputChar + "", attr);
                    totalLength++;
                }
            }
     
            // tPane.setCaretPosition(textDoc.getLength());
            tPane.setCaretPosition(totalLength);
     
        } catch (Exception e) {
            System.err.println("JMSwingText.append(): Error inserting text." + e);
            if (DEBUG) {
                e.printStackTrace();
            }
        }
     
    }
     */
    
    /**
     * Append text to our JMSwingText.
     * Revised method supplied by Stephane Boisjoli 2004-11-26
     * @param input The input to be added to the current JMSwingText area.
     */
    public synchronized void append(final String input) {
        
        final int len = input.length();
        totalLength = textDoc.getLength();
        
        try {
            
            int doneTo=0, escPos;
            while (  (escPos=input.indexOf('\u001b',doneTo)) > - 1) {
                // There are special escape chars that need to be processed.
                // First ditch any non-special stuff
                if (  escPos - doneTo > 0) {
                    textDoc.insertString(totalLength, input.substring(doneTo, escPos) , attr);
                    totalLength += escPos - doneTo;
                    
                    if (DEBUG) {
                        System.err.println(input.substring(doneTo, escPos));
                        System.err.println("totalLength is: " + totalLength);
                    }
                }
                // Now process the Escape.
                doneTo = checkEscape(input, escPos) + 1;
            }
            // If anything remains it is just icing, I mean, regular text
            if (doneTo < len) {
                textDoc.insertString(totalLength, input.substring(doneTo) , attr);
                totalLength += len - doneTo;
                
                if (DEBUG) {
                    System.err.println(input.substring(doneTo));
                    System.err.println("totalLength is: " + totalLength);
                }
            }
            
            tPane.setCaretPosition(totalLength);
            
        } catch (Exception e) {
            System.err.println("JMSwingText.append(): Error inserting text." + e);
            if (DEBUG) {
                e.printStackTrace();
            }
        }
        
    }
    
    
    /**
     * This sets the text area editable or non-editable.
     * @param state <CODE>true</CODE> sets the text area editable.
     * <CODE>false</CODE> sets the text area non-editable.
     */
    public synchronized void setEditable(final boolean state) {
        tPane.setEditable(state);
    }
    
    /** Here, we check for an ANSI colour escape in the 'token', starting at position 'i'.
     * If this is indeed a real colour escape, we will return i as the end of the escape,
     * modifying the 'currentAttrib' and "eating" the escape
     * @param token is a string that is to be checked to see if it is an ANSI colour escape
     * @param startPos is the starting position of the token, incase the token is more than an &quot;escape"
     * @return This integer represents to end of the &quot;escape" if there is one.
     */
    protected synchronized int checkEscape(final String token, final int startPos) {
        int end = 0;
        // int split = token.indexOf('\u001b') + 2;
        int split = 2;  // Just exclude our escape character and the square bracket
        int newSplit = 0;
        int token1 = -1;
        // int token2 = -1;
        boolean loop = true;
        
        end = token.indexOf('m', startPos);	// look for the closing 'm' from the offset of 'startPos'
        
        if (DEBUG) {
            System.err.println("JMSwingText.checkEscape() token: " + token + " start: " + startPos + " end: " + end);
            System.err.println("Snipped token: " + token.substring(startPos, end));
        }
        
        // if (end == 0) {
        if (end < 0) {
            return startPos + 2;       // Could be a simple escape like "BELL"
            // return i; 	// This token doesn't qualify as a colour
        }
        final int mPlace = token.indexOf('m', startPos);
        if (mPlace < 0) {
            return token.length();
        }
        final String lilToken = token.substring(startPos, mPlace + 1);
        
        if (DEBUG) {
            System.err.println("After chopping, lilToken is now: " + lilToken);
        }
        
        // This should continue a loop until our last element, which does
        // not end with a ';' character.  We'll grab that last.
        try {
            while (loop) {
                if (DEBUG) {
                    System.err.println("Our 'split' is: " + split);
                }
                
                newSplit = lilToken.indexOf(';', split);
                
                if (DEBUG) {
                    System.err.println("Presence of semi-colon at: " + newSplit);
                }
                
                if (newSplit > end || newSplit < 1 || split >= newSplit) {
                    if (DEBUG) {
                        System.err.println("Our newSplit is greater than end, terminating loop.");
                    }
                    loop = false;
                    break;
                }
                
                token1 = Integer.parseInt(lilToken.substring(split, newSplit));
                if (DEBUG) {
                    System.err.println("Token: " + lilToken.substring(split, newSplit) + " creates:" + token1 + ":");
                }
                buildColourAttr(token1);
                split = lilToken.indexOf(';', split);
                split++;
            }
        } catch (Exception e) {
            if (DEBUG) {
                System.err.println("Token-loop caught: " + e);
                System.err.println("Start (split) " + split + " end split (newSplit) " + newSplit);
            }
            loop = false;
        }
        
        // if (lilToken.indexOf(';') > 0) {
        if (lilToken.indexOf(';') > -1) {
            if (DEBUG) {
                System.err.println("JMSwingText.checkEscape() has found a semi-colon.");
            }
            final int endStart = lilToken.lastIndexOf(';') + 1;
            final int endEnd = lilToken.lastIndexOf('m');
            try {
                if (endStart > 0) {
                    token1 = Integer.parseInt(lilToken.substring(endStart, endEnd));
                    if (DEBUG) {
                        System.err.println("JMSwingText.checkEscape() token: " + token1);
                    }
                } else {
                    // This is just a single colour token
                    token1 = Integer.parseInt(token.substring(startPos + 2, end));
                    if (DEBUG) {
                        System.err.println("We think we've just got a little token.");
                    }
                }
                
                // if (token1 == 0) token1 = 10;  What did this do?  Fix this XXX
            } catch (NumberFormatException nfe) {
                System.err.println(nfe);
                System.err.println("lilToken " + lilToken);
                System.err.println("Token " + token);
                // System.exit(0);
                return (endEnd + 1);
            } catch (Exception e) {
                if (DEBUG) {
                    System.err.println("Error trying to get closing token." + e);
                    // This seems to happen if we're missing a final 'm'
                    System.err.println("Our entire token was: " + token);
                    System.err.println("The lilToken was: " + lilToken);
                }
                final int start = lilToken.lastIndexOf(';');
                final int stop = lilToken.indexOf('m', start);
                token1 = Integer.parseInt(lilToken.substring(start + 1, stop));
            }
        } else {
            // System.err.println("Our token: " + token.substring(i + 2, end));
            try {
                if (DEBUG) {
                    System.err.println("Checking for escape in :" + token.substring(startPos + 2, end) + ":");
                }
                
                if (token.substring(startPos + 2, end).equals("")) {
                    // This is a "reset" token... ESC[m;
                    if (DEBUG) {
                        System.err.println("Caught 'reset' escape.");
                    }
                    buildColourAttr(0);
                } else {
                    token1 = Integer.parseInt(token.substring(startPos + 2, end));
                }
                
            } catch (Exception e) {
                if (DEBUG) {
                    System.err.println("JMSwingText.checkEscape() Nastybad token: " + token);
                    System.err.println("JMSwingText.checkEscape() caught exception " + e);
                    e.printStackTrace();
                }
                
                if (DEBUG) {
                    System.err.println("Our working token is: *" + token.trim() + "*");
                }
                
                // See if this works.  Fix me XXX
                // if (token.trim().length() >= 7 && token.trim().startsWith("[")) {
                if (token.trim().length() >= 7) {
                    // String hexToken = token.trim().substring(startPos + 2, end - 1);
                    final String hexToken = token.substring(startPos + 2, end);
                    
                    if (DEBUG) {
                        System.err.println("JMSwingText falling back to hex colour using token: " + token);
                        System.err.println("We've come up with: " + hexToken);
                    }
                    
                    // System.err.println("Should've changed with ChangeAttrib");
                    final Color tempColour = hexToColour(hexToken);
                    if (DEBUG) {
                        System.err.println("Going to set foreground.");
                    }
                    StyleConstants.setForeground(attr, tempColour);
                    if (DEBUG) {
                        System.err.println("Exited.");
                    }
                    return end;
                } else {
                    System.err.println("Token isn't 6 long? *" + token.trim() + "*");
                }
            }
        }
        buildColourAttr(token1);
        
        // return (end + 1);
        return end;
    }
    
    /**
     * A method to sift through ANSI colour numbers and assign
     * them to the proper "area" (foreground, background, etc...)
     * @param colAt The ANSI colour code to be processed by this method
     */
    protected synchronized void buildColourAttr(final int colAt) {
        if (colAt == 0) {
            StyleConstants.setFontFamily(attr, standardFont.getName());
            StyleConstants.setFontSize(attr, standardFont.getSize());
            
            // Bold actually means "brighter", not "bolder"
//            if (standardFont.isBold()) {
//                StyleConstants.setBold(attr, true);
//            } else {
//                StyleConstants.setBold(attr, false);
//            }
            StyleConstants.setBold(attr, standardFont.isBold());
            
//            if (standardFont.isItalic()) {
//                StyleConstants.setItalic(attr, true);
//            } else {
//                StyleConstants.setItalic(attr, false);
//            }
            StyleConstants.setItalic(attr,  standardFont.isItalic());
            
            StyleConstants.setUnderline(attr, false);
            StyleConstants.setStrikeThrough(attr, false);
            StyleConstants.setForeground(attr, fgColour);
            if (paintBG) {
                StyleConstants.setBackground(attr, bgColour);
            }
            
            if (DEBUG) {
                System.err.println("Reset styles");
            }
            // return;
        }
        
        if (colAt < 30) {
            switch (colAt) {
                case 0:
                    isBold = false;
                    if (DEBUG) {
                        System.err.println("Resetting isBold to false.");
                    }
                    break;
                case 1:
                    if (boldNotBright) {
                        StyleConstants.setBold(attr, true);
                        if (DEBUG) {
                            System.err.println("Setting BOLD true.");
                        }
                    } else {
                        if (DEBUG) {
                            System.err.println("We're setting bold, but boldNotBright is set " + boldNotBright);
                        }
                    }
                    isBold = true;
                    break;
                case 3:
                    StyleConstants.setItalic(attr, true);
                    break;
                case 4:
                    if (DEBUG) {
                        System.err.println("Set Underline.");
                    }
                    StyleConstants.setUnderline(attr, true);
                    break;
                case 9:
                    StyleConstants.setStrikeThrough(attr, true);
                    break;
                case 22:
                    if (DEBUG) {
                        System.err.println("Setting BOLD false.");
                    }
                    StyleConstants.setBold(attr, false);
                    isBold = false;
                    break;
                case 24:
                    StyleConstants.setUnderline(attr, false);
                    break;
                default:
                    /* Nothing */
            }
        }
        
        /*
         if (ca < 30) {
         if (ca == 1) ChangeAttrib("<bold>");
         if (ca == 2) ChangeAttrib("<faint>");
         if (ca == 3) ChangeAttrib("<i>");
         if (ca == 4) ChangeAttrib("<u>");
         if (ca == 7) ChangeAttrib("<negative>");
         if (ca == 9) ChangeAttrib("<s>");
         if (ca == 22) ChangeAttrib("</bold>");
         if (ca == 24) ChangeAttrib("</u>");
         return;
         }
         */
        
        /* Real colour codes (JamochaMUD sort've fudges them right now
         * Atributes
         * 0  Normal text
         * 1  Bold   (This gives the color higher intensity)
         * 4  Underline
         * 5  Blinking
         * 7  Reverse (Foreground -> Background, Background -> Foreground)
         * 8  Hidden
         *
         * Foreground color
         * 30 Black
         * 31 Red
         * 32 Green
         * 33 Yellow
         * 34 Blue
         * 35 Magenta
         * 36 Cyan
         * 37 White
         *
         * Background color
         * 39 Reset background colour
         * 40 Black
         * 41 Red
         * 42 Green
         * 43 Yellow
         * 44 Blue
         * 45 Magenta
         * 46 Cyan
         * 47 White
         */
        
        if (colAt > 29 && colAt < 50) {
            int cInt = colAt;
            
            if (colAt > 29 && colAt < 40) {
                cInt = cInt - 30;
            } else {
                cInt = cInt - 40;
            }
            
            tempColour = privPal[cInt];
            if (DEBUG) {
                System.err.println("JMSwingText.BuildColourAttr colour is " + tempColour);
            }
            
            // Make colour brighter if it is supposed to be bold
            if (isBold) {
                if (cInt == 0) {
                    tempColour = Color.gray;
                } else {
                    tempColour = privPal[cInt + 8];
                }
            }
            
            if (colAt > 29 && colAt < 38) {
                StyleConstants.setForeground(attr, tempColour);
            }
            
            if (colAt > 39 && colAt < 48 && paintBG) {
                //if (paintBG) {
                StyleConstants.setBackground(attr, tempColour);
                //}
            }
            
            // if (colAt == 39) {
            if (colAt == 39 && paintBG) {
                // attr = sContext.removeAttribute(SimpleAttributeSet.EMPTY, StyleConstants.Background);
                // attr = sContext.addAttribute(SimpleAttributeSet.EMPTY, StyleConstants.Background, new Color(255, 255, 255));
                // if (paintBG) {
                StyleConstants.setBackground(attr, bgColour);
                // }
            }
            
        }
        
    }
    
    /**
     * Tell us whether it is alright to paint the background colour or not.
     * Some combinations of Java 1.1.x and Swing seem to have trouble with
     * background painting and just paint a solid coloured square instead of
     * text!!
     * @param state <CODE>true</CODE> indicates we should handle painting the background colour
     * <CODE>false</CODE> indicates we should not paint the background colour ourselves
     */
    public void setPaintBackground(final boolean state) {
        paintBG = state;
        if (DEBUG) {
            System.err.println("JMSwingText.setPaintBackground now set to: " + paintBG);
        }
    }
    
    /**
     * Set the foreground (text) colour of our display
     * @param newColour The colour to set our foreground (text)
     */
    public void setForeground(final Color newColour) {
        fgColour = newColour;
        if (tPane != null) {
            tPane.setForeground(fgColour);
            StyleConstants.setForeground(attr, fgColour);
        }
    }
    
    /**
     * Set the background colour of our text area
     * @param newColour The colour to set our background
     */
    public void setBackground(final Color newColour) {
        bgColour = newColour;
        if (tPane != null) {
            tPane.setBackground(bgColour);
            if (paintBG) {
                StyleConstants.setBackground(attr, bgColour);
            }
        }
    }
    
    /**
     * Set the font to be used in our display.
     * @param newFont The java.awt.Font to use for our display
     */
    public void setFont(final Font newFont) {
        int oldSize = 0;
        if (standardFont != null) {
            oldSize = standardFont.getSize();
        }
        
        standardFont = newFont;
        if (tPane != null) {
            StyleConstants.setFontFamily(attr, standardFont.getName());
            StyleConstants.setFontSize(attr, standardFont.getSize());
            
            if (standardFont.isBold()) {
                StyleConstants.setBold(attr, true);
            }
            
            if (standardFont.isItalic()) {
                StyleConstants.setItalic(attr, true);
            }
            
            // If the old font size and new font size are different, we're going
            // to have to change the size of the font in the entire panel
            if (oldSize != newFont.getSize()) {
                if (DEBUG) {
                    System.err.println("Old font size: " + oldSize);
                    System.err.println("New font size: " + newFont.getSize());
                    // tPane.setFont(newFont);  Doesn't affect styles
//                    SimpleAttributeSet newAttr = new SimpleAttributeSet();
//                    StyleConstants.setFontFamily(newAttr, standardFont.getFamily());
//                    StyleConstants.setFontSize(newAttr, standardFont.getSize());
//
//                    setCharacterAttributes(tPane, )
                }
            }
        }
    }
    
    /**
     * Return all of our text
     * @return Return all the text current in our text widget.
     */
    public String getText() {
        return tPane.getText();
    }
    
    /**
     * Return any texted that is currently selected
     * @return returns any text currently selected in our text widget
     */
    public String getSelectedText() {
        if (DEBUG) {
            System.err.println("JMSwingText returning text: " + tPane.getSelectedText());
        }
        return tPane.getSelectedText();
    }
    
    /**
     * Select text marked by the start and end integers
     * @param start The starting point of our selection.  This number
     * is the count of characters from the beginning of our
     * entire text.
     * @param end The end point of our selection.  This number is the
     * count of characters from the beginning of our text.
     */
    public void select(final int start, final int end) {
        final int caretPos = tPane.getCaretPosition();
        tPane.select(start, end);
        tPane.setCaretPosition(caretPos);
    }
    
    /**
     * We need to make our own mouse-listener that'll report back
     * to any listeners that may've registered to this component.  Or
     * something.  This may not make a lotta sense, I'm tired.
     * @param mouse The mouselistener for this widget
     */
    public synchronized void addMouseListener(final MouseListener mouse) {
        mListener = AWTEventMulticaster.add(mListener, mouse);
        enableEvents(AWTEvent.MOUSE_EVENT_MASK);
    }
    
    /**
     * Remove given mouse listener from this widget
     * @param mouse The mouselistener to remove
     */
    public synchronized void removeMouseListener(final MouseListener mouse) {
        if (mListener != null) {
            mListener = AWTEventMulticaster.remove(mListener, mouse);
        }
    }
    
    // Mouse events
    /**
     * Capture any mouse clicks so that we can snoop them if need be.
     * @param event The captured mouse event
     */
    public void mouseClicked(final MouseEvent event) {
        
        // Fix me!  This has to be replaced to avoid a Null Pointer Exception
        // if (event.getClickCount() >= 2) {
        if (event.getClickCount() == 2) {
            if (DEBUG) {
                System.err.println("JMSwingText.mouseClicked() (twice) event " + event);
            }
            
            final java.awt.Point newPos = new java.awt.Point(event.getX(), event.getY());
            final int pos = tPane.viewToModel(newPos);
            
            if (DEBUG) {
                System.err.println("Our position is: " + pos);
            }
            
            // javax.swing.text.Element elem = textDoc.getCharacterElement(pos);
            // String word;
            
            try {
                // String word = textDoc.getText(elem.getStartOffset(), elem.getEndOffset() - elem.getStartOffset());
                final int start = javax.swing.text.Utilities.getWordStart(tPane, pos);
                final int end = javax.swing.text.Utilities.getWordEnd(tPane, pos);
                
                if (DEBUG) {
                    System.err.println("JMSwingText attempting to set selection: " + start + " to " + (end - start));
                }
                
                // Our selection doesn't seem to work if we set it manually, but can we
                // depend on other operating systems to highlight the word?!
                // Fix Me XXX!
                // select(start, end - start);
                
                if (DEBUG) {
                    System.err.println("Selection complete.");
                }
                // word = textDoc.getText(start, end - start);
                
                //if (DEBUG) {
                // System.err.println("Our word is: " + word);
                // }
            } catch (Exception exc) {
                System.err.println("Exception trying to get word in JMSwingText.");
            }
            
        }
        
        if (mListener != null) {
            mListener.mouseClicked(event);
        }
        
    }
    
    /**
     * Empty event
     * @param event Mouse Event
     */
    public void mouseEntered(final MouseEvent event) {
        /*
        // Change the mouse cursor to an I-beam
        setCursor(new Cursor(Cursor.TEXT_CURSOR));
         */
    }
    
    /**
     * Empty Event
     * @param event Mouse Event
     */
    public void mouseExited(final MouseEvent event) {
        /*
        // Return mouse cursor back to standard pointer
        setCursor(new Cursor(Cursor.DEFAULT_CURSOR));
         */
    }
    
    /**
     * Empty Event
     * @param event Mouse Event
     */
    public void mousePressed(final MouseEvent event) {
        /*
        // The mouse has been pressed, we will record its array position
        textBegin = pointToArray(new Point(event.getX(), event.getY()));
        textEnd = new Point(textBegin.x, textBegin.y);
         
        // Check to see if we have to deselect any old selections
        if (selectBegin.x > 0 || selectBegin.y > 0 || selectEnd.x > 0 || selectEnd.y > 0) {
            paintSelection(selectBegin, selectEnd, drawText.getBackground());
        }
         
        // And now we'll record the 'corrected position'; the character's position
        selectBegin = arrayToPoint(textBegin);
        selectEnd = new Point(selectBegin.x, selectBegin.y);
         */
    }
    
    /**
     * Empty Event
     * @param event Mouse Event
     */
    public void mouseReleased(final MouseEvent event) {
        /*
        if(selectBegin.x == selectEnd.x && selectBegin.y == selectEnd.y) {
            repaint();
            // No need for selection... start and end points are the same
            return;
            }
         */
    }
    
    /**
     * Empty Event
     * @param event Mouse Event
     */
    public void mouseDragged(final MouseEvent event) {
        /*
        // First we'll nab our selected character location
        Point tempPoint = pointToArray(new Point(event.getX(), event.getY()));
        int selectedColumn = tempPoint.x;
        int selectedRow = tempPoint.y;
         
        // First we'll grab our FontMetrics
        // FontMetrics fMetric = getFontMetrics(getFont());
         
        int oldx = selectEnd.x, oldy = selectEnd.y;
        // Figure out the coordinates of this selection
         
        tempPoint = arrayToPoint(new Point(selectedColumn, selectedRow));
        int x = tempPoint.x;
        int y = tempPoint.y;
         
        if((x < selectBegin.x && y < selectBegin.y) &&
           (x < selectEnd.x && y < selectEnd.y)) {
         
            selectBegin.x = x;
            selectBegin.y = y;
            textBegin.x = selectedColumn;
            textBegin.y = selectedRow;
        } else {
            selectEnd.x = x;
            selectEnd.y = y;
            textEnd.x = selectedColumn;
            textEnd.y = selectedRow;
        }
         
        if(oldx != x || oldy != y) {
            // erase our old selection first
            paintSelection(selectBegin, new Point(oldx, oldy), drawText.getBackground());
            // Then draw our new selection
            paintSelection(selectBegin, selectEnd, drawText.getBackground());
            }
         */
    }
    
    /**
     * Empty Event
     * @param evt Mouse Event
     */
    public void mouseMoved(final MouseEvent evt) {
    }
    
    // KeyListener events
    /**
     * Allow KeyListeners to be added to our JMSwingText.
     * @param listener The listener to be added
     */
    public void addKeyListener(final KeyListener listener) {
        keyL = AWTEventMulticaster.add(keyL, listener);
        enableEvents(AWTEvent.KEY_EVENT_MASK);
    }
    
    /**
     * Remove registered KeyListeners from our JMSwingText class
     * @param listener The KeyListener to be removed
     */
    public void removeKeyListener(final KeyListener listener) {
        keyL = AWTEventMulticaster.remove(keyL, listener);
    }
    
    /**
     * Send out any key events to our listeners
     * @param event The received keyPressed event
     */
    public void keyPressed(final KeyEvent event) {
        if(keyL != null) {
            keyL.keyPressed(event);
        }
    }
    
    /**
     * Related our keyReleased event to our registered KeyListener
     * @param event received KeyEvent
     */
    public void keyReleased(final KeyEvent event) {
        if(keyL != null) {
            keyL.keyReleased(event);
        }
    }
    
    /**
     * Pass our keyTyped events to our KeyListener
     * @param event Our KeyEvent
     */
    public void keyTyped(final KeyEvent event) {
        if (DEBUG) {
            System.err.println("keyTyped on JMSwingText.");
        }
        if(keyL != null) {
            keyL.keyTyped(event);
        }
    }
    
    /**
     * Methods for handling hyper-links in text.  To be completed.  Fix Me XXX.
     * @param evt Hyperlink event
     */
    public void hyperlinkUpdate(final HyperlinkEvent evt) {
        
        // If we don't check this in versions of Java later than 1.2 the link
        // will be activated every time the mouse moves over it!
        // Fix this XXX
//        if (evt.getEventType() == HyperlinkEvent.EventType.ACTIVATED) {
//
//            if (DEBUG) {
//                System.err.println("JMSwingText.hyperlinkUpdated(): Received a HyperlinkEvent " + evt);
//            }
//        }
    }
    
    /** Convert the given hex value into a Java Color object */
    private Color hexToColour(final String initHexVal) {
        // private Color hexToColour(String hexVal) {
        if (DEBUG) {
            System.err.println("Entering hexToColour with: " + initHexVal);
        }
        
        String hexVal;
        int red, green, blue;
        
        // if (hexVal.startsWith("#")) {
        // if (initHexVal.startsWith("#")) {
        final int hashPoint = initHexVal.indexOf('#');
        if (hashPoint > -1) {
            // strip the proceeding # symbol
            // hexVal = new String(hexVal.substring(1, 7));
            // hexVal = new String(initHexVal.substring(1, 7));
            hexVal = new String(initHexVal.substring(hashPoint + 1, 7));
        } else {
            hexVal = initHexVal;
        }
        
        if (DEBUG) {
            System.err.println("JMSwingText hexVal: " + hexVal);
        }
        
        try {
            red = Integer.parseInt(hexVal.substring(0, 2), 16);
            green = Integer.parseInt(hexVal.substring(2, 4), 16);
            blue = Integer.parseInt(hexVal.substring(4, 6), 16);
        } catch (Exception colourErr) {
            // Not a valid number
            if (DEBUG) {
                System.err.println("JMSwingText, not a valid colour number, setting to white.");
            }
            
            red = 255;
            green = 255;
            blue = 255;
        }
        
        return new Color(red, green, blue);
        
    }
    
    /**
     * Turn anti-aliasing of our letters on or off
     * @param status The variable that tells us to either turn on
     * anti-aliasing with <CODE>true</CODE>, or turn off anti-aliasing
     * with <CODE>false</CODE>.
     */
    public synchronized void setAntiAliasing(final boolean status) {
        
        tPane.setAntiAliasing(status);
        
        if (DEBUG) {
            System.err.println("JMSwingText.setAntiAliasing() setting antialiasing to " + status);
        }
        
    }
    
    /** Copy any selected text to the system clipboard */
    public void copy() {
        tPane.copy();
    }
    
    /**
     * There are two different ways to interpret the "bold" feature on MU*'s.
     *  The first technique takes the "bold" command and makes the text bold.
     *  The second technique uses a palette at about half-brightness, and when
     *  the "bold" command is received, it turns the brightness up to full.
     * @param state <CODE>true</CODE> - Create the characters as bold, using a limited palette
     * <CODE>false</CODE> - Use the 16 colour palatte instead of bolding characters
     */
    public void setBoldNotBright(final boolean state) {
        if (DEBUG) {
            System.err.println("setBoldNotBright set to: " + state);
        }
        
        boldNotBright = state;
        // Color tempCol;
        privPal = new Color[16];
        
        // Remap our palette of colours
        if (boldNotBright) {
            if (DEBUG) {
                System.err.println("Set to 8 colour palette");
            }
            setStandardPalette();
            // We are going to using the bolding technique
//            privPal[0] = Color.black;
//            privPal[1] = Color.red;
//            privPal[2] = Color.green;
//            privPal[3] = Color.yellow;
//            privPal[4] = Color.blue;
//            privPal[5] = Color.magenta;
//            privPal[6] = Color.cyan;
//            privPal[7] = Color.white;
//            // upper colours
//            privPal[8] = Color.gray;
//            privPal[9] = Color.red;
//            privPal[10] = Color.green;
//            privPal[11] = Color.yellow;
//            privPal[12] = Color.blue;
//            privPal[13] = Color.magenta;
//            privPal[14] = Color.cyan;
//            privPal[15] = Color.white;
            
        } else {
            if (DEBUG) {
                System.err.println("Colours set to fancy palette");
            }
            // We will use the fancy palette technique
            setFancyPalette();
//            privPal[0] = Color.black;
//            privPal[1] = new Color(205, 0, 0);
//            privPal[2] = new Color(110, 139, 61);
//            privPal[3] = new Color(238, 238, 0);
//            privPal[4] = new Color(0, 0, 129);
//            privPal[5] = new Color(205, 10, 76);
//            privPal[6] = new Color(122, 197, 205);
//            privPal[7] = new Color(169, 169, 169);
//
//            // upper colours
//            privPal[8] = Color.gray;
//            privPal[9] = Color.red;
//            privPal[10] = Color.green;
//            privPal[11] = Color.yellow;
//            privPal[12] = Color.blue;
//            privPal[13] = Color.magenta;
//            privPal[14] = Color.cyan;
//            privPal[15] = Color.white;
            
        }
    }
    
    /**
     * This returns the state of our colour rendering, if we are using the
     * 8 colour (bold) palette, or the 16 colour (bright) palette.
     * @return <CODE>true</CODE> - We are using the 8 colour (bold) palette
     * <CODE>false</CODE> - We are using the 16 colour (bright) palette
     */
    public boolean isBoldNotBright() {
        return boldNotBright;
    }
    
    /**
     * Move the scrollbars either one page up or one page down
     * @param direction 
     */
    public void movePage(final int direction) {
        // Get a handle on our scrollbar
        // int pos = tPane.get
        final javax.swing.JScrollBar tempScroll = this.getVerticalScrollBar();
        // Get the unit to move our scrollbar
        final int inc = tempScroll.getBlockIncrement(tempScroll.VERTICAL);
        final int pos = tempScroll.getValue();
        int newPos = 0;
        // Move the durn thing!
        if (direction == anecho.gui.JMSwingText.PAGEDOWN) {
            if (DEBUG) {
                System.err.println("JMSwingText.java: PageDown.");
            }
            newPos = pos + inc;
        } else {
            if (DEBUG) {
                System.err.println("JMSwingText.java: PageUp.");
            }
            newPos = pos - inc;
        }
        if (DEBUG) {
            System.err.println("Our increment is " + inc);
            System.err.println("Our position is " + pos);
            System.err.println("Our new position should be " + newPos);
        }
        tempScroll.setValue(newPos);
    }
    
    /**
     * Return an array of colours representing our current palette
     * @return 
     */
    public Color[] getCurrentPalette() {
        if (privPal == null) {
            setStandardPalette();
        }
        return privPal;
    }
    
    /**
     * 
     * @param newPal 
     * @throws java.lang.Exception 
     */
    public void setPalette(final Color[] newPal) throws Exception {
        if (newPal != null) {
            privPal = newPal;
        }
    }
    
    private void setStandardPalette() {
        privPal = new Color[16];
        if (DEBUG) {
            System.err.println("Standard palette set via setStandardPalette()");
        }
//        privPal[0] = Color.black;
//        privPal[1] = Color.red;
//        privPal[2] = Color.green;
//        privPal[3] = Color.yellow;
//        privPal[4] = Color.blue;
//        privPal[5] = Color.magenta;
//        privPal[6] = Color.cyan;
//        privPal[7] = Color.white;
//        // upper colours
//        privPal[8] = Color.gray;
//        privPal[9] = Color.red;
//        privPal[10] = Color.green;
//        privPal[11] = Color.yellow;
//        privPal[12] = Color.blue;
//        privPal[13] = Color.magenta;
//        privPal[14] = Color.cyan;
//        privPal[15] = Color.white;
        privPal = this.getStandardPalette();
        
    }
    
    private void setFancyPalette() {
        privPal = new Color[16];
        if (DEBUG) {
            System.err.println("Fancy palette set via setFancyPalette()");
        }
        // We will use the fancy palette technique
//        privPal[0] = Color.black;
//        privPal[1] = new Color(205, 0, 0);
//        privPal[2] = new Color(110, 139, 61);
//        privPal[3] = new Color(238, 238, 0);
//        privPal[4] = new Color(0, 0, 129);
//        privPal[5] = new Color(205, 10, 76);
//        privPal[6] = new Color(122, 197, 205);
//        privPal[7] = new Color(169, 169, 169);
//        
//        // upper colours
//        privPal[8] = Color.gray;
//        privPal[9] = Color.red;
//        privPal[10] = Color.green;
//        privPal[11] = Color.yellow;
//        privPal[12] = Color.blue;
//        privPal[13] = Color.magenta;
//        privPal[14] = Color.cyan;
//        privPal[15] = Color.white;
        
        privPal = this.getFancyPalette();
    }
    
    /**
     * 
     * @return 
     */
    public Color[] getStandardColours() {
        Color[] retPal = new Color[16];
        if (boldNotBright) {
            retPal = getStandardPalette();
        } else {
            retPal = getFancyPalette();
        }
        
        return retPal;
    }

        private Color[] getStandardPalette() {
        Color[] stdPal = new Color[16];
        if (DEBUG) {
            System.err.println("Standard palette set via setStandardPalette()");
        }
        stdPal[0] = Color.black;
        stdPal[1] = Color.red;
        stdPal[2] = Color.green;
        stdPal[3] = Color.yellow;
        stdPal[4] = Color.blue;
        stdPal[5] = Color.magenta;
        stdPal[6] = Color.cyan;
        stdPal[7] = Color.white;
        // upper colours
        stdPal[8] = Color.gray;
        stdPal[9] = Color.red;
        stdPal[10] = Color.green;
        stdPal[11] = Color.yellow;
        stdPal[12] = Color.blue;
        stdPal[13] = Color.magenta;
        stdPal[14] = Color.cyan;
        stdPal[15] = Color.white;
        
        return stdPal;
    }

    private Color[] getFancyPalette() {
        Color[] fancyPal = new Color[16];
        if (DEBUG) {
            System.err.println("Fancy palette set via setFancyPalette()");
        }
        // We will use the fancy palette technique
        fancyPal[0] = Color.black;
        fancyPal[1] = new Color(205, 0, 0);
        fancyPal[2] = new Color(110, 139, 61);
        fancyPal[3] = new Color(238, 238, 0);
        fancyPal[4] = new Color(0, 0, 129);
        fancyPal[5] = new Color(205, 10, 76);
        fancyPal[6] = new Color(122, 197, 205);
        fancyPal[7] = new Color(169, 169, 169);
        
        // upper colours
        fancyPal[8] = Color.gray;
        fancyPal[9] = Color.red;
        fancyPal[10] = Color.green;
        fancyPal[11] = Color.yellow;
        fancyPal[12] = Color.blue;
        fancyPal[13] = Color.magenta;
        fancyPal[14] = Color.cyan;
        fancyPal[15] = Color.white;
        
        return fancyPal;
    }
}