/* Icon Plus
 * Copyright (C) 2001 Dmitry A.Steklenev
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. All advertising materials mentioning features or use of this
 *    software must display the following acknowledgment:
 *    "This product includes software developed by Dmitry A.Steklenev".
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Dmitry A.Steklenev".
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR OR CONTRIBUTORS "AS IS"
 * AND ANY EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * AUTHOR OR THE CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: ic_bitmap.cpp,v 1.5 2001/05/31 06:28:20 glass Exp $
 */

#include "ic_bitmap.h"
#include <stdlib.h>
#include <memory.h>
#include <stdio.h>

/*------------------------------------------------------------------
 * Constructs the empty generic bitmap
 *------------------------------------------------------------------*/
ICBitmap::ICBitmap()

: map_width         ( 0 ),
  map_height        ( 0 ),
  map_bits_per_pixel( 0 ),
  map_byte_width    ( 0 ),
  map_size          ( 0 ),
  map_bits          ( 0 )
{}

/*------------------------------------------------------------------
 * Constructs the generic bitmap
 *------------------------------------------------------------------*/
ICBitmap::ICBitmap( size_type width, size_type height, size_type bits_per_pixel )

: map_width         ( width          ),
  map_height        ( height         ),
  map_bits_per_pixel( bits_per_pixel )
{
  map_byte_width = (width*bits_per_pixel+7)/8;
  map_size = map_byte_width*height;
  map_bits = new BYTE[map_size];
  memset( map_bits, 0, map_size);
}

/*------------------------------------------------------------------
 * Constructs the generic bitmap from another bitmap object
 *------------------------------------------------------------------*/
ICBitmap::ICBitmap( const ICBitmap& bitmap )
{
  map_width          = bitmap.map_width;
  map_height         = bitmap.map_height;
  map_bits_per_pixel = bitmap.map_bits_per_pixel;
  map_byte_width     = bitmap.map_byte_width;
  map_size           = bitmap.map_size;

  map_bits = new BYTE[map_size];
  memcpy( map_bits, bitmap.map_bits, map_size);
}

/*------------------------------------------------------------------
 * Destructs the generic bitmap
 *------------------------------------------------------------------*/
ICBitmap::~ICBitmap()
{
  delete map_bits;
}

/*------------------------------------------------------------------
 * Assignment operator
 *------------------------------------------------------------------*/
ICBitmap& ICBitmap::operator=( const ICBitmap& bitmap )
{
  if( &bitmap != this )
  {
    delete map_bits;

    map_width          = bitmap.map_width;
    map_height         = bitmap.map_height;
    map_bits_per_pixel = bitmap.map_bits_per_pixel;
    map_byte_width     = bitmap.map_byte_width;
    map_size           = bitmap.map_size;

    map_bits = new BYTE[map_size];
    memcpy( map_bits, bitmap.map_bits, map_size);
  }

  return *this;
}

/*------------------------------------------------------------------
 * Returns the padded bitmap size in bytes
 *------------------------------------------------------------------*/
ICBitmap::size_type ICBitmap::padded_size( size_type boundary ) const
{
  size_type width = byte_width();

  while( width % boundary != 0 )
    width++;

  return width * height();
}

/*------------------------------------------------------------------
 * Returns the specified pixel
 *------------------------------------------------------------------*/
UINT32 ICBitmap::get_pixel( size_type row, size_type col ) const
{
  // The first byte of the specified location.
  size_type pos = row*map_byte_width + col*map_bits_per_pixel/8;
  // The first bit in first byte of the specified location.
  size_type bit = col*map_bits_per_pixel%8;

  UINT32 pixel = 0;
  UINT32 mask  = ((1 << map_bits_per_pixel) - 1);

  for( int i = 0; i <= (int)((map_bits_per_pixel-1)/8); i++ )
  {
    pixel = (pixel << 8) | map_bits[ pos + i ];
  }

  pixel >>= ((map_bits_per_pixel+7)/8)*8 - bit - map_bits_per_pixel;
  return pixel & mask;
}

/*------------------------------------------------------------------
 * Sets the specified pixel
 *------------------------------------------------------------------*/
void ICBitmap::set_pixel( size_type row, size_type col, UINT32 pixel )
{
  // The first byte of the specified location.
  size_type pos = row*map_byte_width + col*map_bits_per_pixel/8;
  // The first bit in first byte of the specified location.
  size_type bit = col*map_bits_per_pixel%8;

  UINT32 mask= ((1 << map_bits_per_pixel) - 1);

  pixel <<= ((map_bits_per_pixel+7)/8)*8 - bit - map_bits_per_pixel;
  mask  <<= ((map_bits_per_pixel+7)/8)*8 - bit - map_bits_per_pixel;

  for( int i = (map_bits_per_pixel-1)/8; i >= 0; i-- )
  {
    map_bits[pos+i] = map_bits[pos+i] & ~(mask & 0xFF ) | (pixel & 0xFF );
    pixel >>= 8;
    mask  >>= 8;
  }
}

/*------------------------------------------------------------------
 * Flips the generic bitmap
 *------------------------------------------------------------------*/
void ICBitmap::flip()
{
  BYTE*  flipped = new BYTE[map_size];
  BYTE*  dst_row = flipped;
  BYTE*  src_row = map_bits + map_size - map_byte_width;

  size_type i;

  while( src_row >= map_bits )
  {
    for( i = 0; i < map_byte_width; i++ )
      *dst_row++ = *src_row++;

    src_row -= 2*map_byte_width;
  }

  memcpy( map_bits, flipped, map_size );
  delete flipped;
}

/*------------------------------------------------------------------
 * Dumps the generic bitmap
 *------------------------------------------------------------------*/
void ICBitmap::dump()
{
  char format[8];
  size_type row, col;

  sprintf( format, "%%0%uX ", map_bits_per_pixel/4 );

  for( row = 0; row < map_height; row++ )
  {
    printf( "\t" );

    for( col = 0; col < map_width; col++ )
      printf( format, get_pixel( row, col ));

    printf( "\n" );
  }
}
