/* Icon Plus
 * Copyright (C) 2001 Dmitry A.Steklenev
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. All advertising materials mentioning features or use of this
 *    software must display the following acknowledgment:
 *    "This product includes software developed by Dmitry A.Steklenev".
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Dmitry A.Steklenev".
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR OR CONTRIBUTORS "AS IS"
 * AND ANY EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * AUTHOR OR THE CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: ic_bitmap.h,v 1.3 2001/05/31 05:34:52 glass Exp $
 */

#ifndef IC_BITMAP_H
#define IC_BITMAP_H

#include "ic_config.h"

/**
 * Icon Plus generic bitmap class
 *
 * The ICBitmap class creates and manages a bitmap image.
 * @author Dmitry A Steklenev
 */

class ICBitmap
{
  public:
    /** An unsigned integral type */
    typedef UINT32 size_type;

  public:
    /** Constructs the empty generic bitmap */
    ICBitmap();
    /** Constructs the generic bitmap */
    ICBitmap( size_type width, size_type height, size_type bits_per_pixel );
    /** Constructs the generic bitmap from another bitmap object */
    ICBitmap( const ICBitmap& );
    /** Destructs the generic bitmap */
   ~ICBitmap();

  public:
    /** Returns the bitmap size in bytes */
    size_type size() const;
    /** Returns the padded bitmap size in bytes */
    size_type padded_size( size_type boundary ) const;
    /** Returns the bitmap width in pixels */
    size_type width() const;
    /** Returns the bitmap height in pixels */
    size_type height() const;
    /** Returns the number of bits per pixel */
    size_type bits_per_pixel() const;
    /** Returns the bitmap width in bytes */
    size_type byte_width() const;
    /** Returns the bitmap pointer */
    BYTE* map();

    /** Returns the specified pixel */
    UINT32 get_pixel( size_type row, size_type col ) const;
    /** Sets the specified pixel */
    void set_pixel( size_type row, size_type col, UINT32 );

  public:
    /** Flips the generic bitmap */
    void flip();
    /** Dumps the generic bitmap */
    void dump();

  public:
    /** Assignment operator */
    ICBitmap& operator=( const ICBitmap& );

  private:
    BYTE*     map_bits;
    size_type map_width;
    size_type map_height;
    size_type map_bits_per_pixel;
    size_type map_byte_width;
    size_type map_size;
};

/**
 * Icon Plus generic bitmap mask class
 *
 * The ICBitMask class creates and manages a black & white bitmap image.
 * @author Dmitry A Steklenev
 */

class ICBitMask : public ICBitmap
{
  public:
    /** Constructs the empty generic mask */
    ICBitMask() : ICBitmap() {}
    /** Constructs the generic mask */
    ICBitMask( size_type width, size_type height ) : ICBitmap( width, height, 1 ) {}
};

/*------------------------------------------------------------------
 * Returns the bitmap size in bytes
 *------------------------------------------------------------------*/
inline ICBitmap::size_type ICBitmap::size() const
{
  return map_size;
}

/*------------------------------------------------------------------
 * Returns the bitmap width in pixels
 *------------------------------------------------------------------*/
inline ICBitmap::size_type ICBitmap::width() const
{
  return map_width;
}

/*------------------------------------------------------------------
 * Returns the bitmap width in pixels
 *------------------------------------------------------------------*/
inline ICBitmap::size_type ICBitmap::byte_width() const
{
  return map_byte_width;
}

/*------------------------------------------------------------------
 * Returns the bitmap height in pixels
 *------------------------------------------------------------------*/
inline ICBitmap::size_type ICBitmap::height() const
{
  return map_height;
}

/*------------------------------------------------------------------
 * Returns the number of bits per pixel
 *------------------------------------------------------------------*/
inline ICBitmap::size_type ICBitmap::bits_per_pixel() const
{
  return map_bits_per_pixel;
}

/*------------------------------------------------------------------
 * Returns the bitmap pointer
 *------------------------------------------------------------------*/
inline BYTE* ICBitmap::map()
{
  return map_bits;
}

#endif
