/* Icon Plus
 * Copyright (C) 2001 Dmitry A.Steklenev
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. All advertising materials mentioning features or use of this
 *    software must display the following acknowledgment:
 *    "This product includes software developed by Dmitry A.Steklenev".
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Dmitry A.Steklenev".
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR OR CONTRIBUTORS "AS IS"
 * AND ANY EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * AUTHOR OR THE CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: ic_file.cpp,v 1.4 2001/05/31 05:57:56 glass Exp $
 */

#include "ic_file.h"
#include <stdlib.h>
#include <string.h>

/*------------------------------------------------------------------
 * Constructs the file object
 *------------------------------------------------------------------*/
ICFile::ICFile( const char* filename, const char* path )
{
  file_handle = 0;

  if( !path )
    file_name = strdup( filename );
  else
  {
    file_name = (char*)malloc( strlen(filename) + strlen(path) + sizeof(PS) + 1 );
    strcpy( file_name, path     );
    strcat( file_name, PS       );
    strcat( file_name, filename );
  }
}

/*------------------------------------------------------------------
 * Destructs the file object
 *------------------------------------------------------------------*/
ICFile::~ICFile()
{
  if( is_open()) close();
  free( file_name );
}

/*------------------------------------------------------------------
 * Opens file for reading
 *------------------------------------------------------------------*/
int ICFile::open_for_read()
{
  file_handle = fopen( file_name, IC_OPEN_BIN_READ );
  return is_open();
}

/*------------------------------------------------------------------
 * Opens file for writing
 *------------------------------------------------------------------*/
int ICFile::open_for_write()
{
  file_handle = fopen( file_name, IC_OPEN_BIN_WRITE );
  return is_open();
}

/*------------------------------------------------------------------
 * Close open file
 *------------------------------------------------------------------*/
int ICFile::close()
{
  int rc = fclose( file_handle );

  if( rc == 0 )
  {
    file_handle = 0;
    return 1;
  }
  else
    return 0;
}

/*------------------------------------------------------------------
 * Changes the current file position
 *------------------------------------------------------------------*/
long ICFile::seek( long offset )
{
  clearerr( file_handle );
  return fseek( file_handle, offset, SEEK_SET );
}

/*------------------------------------------------------------------
 * Returns the current file position
 *------------------------------------------------------------------*/
long ICFile::tell()
{
  return ftell( file_handle );
}

/*------------------------------------------------------------------
 * Reads one byte
 *------------------------------------------------------------------*/
BYTE ICFile::read_byte()
{
  BYTE bytes;

  if( fread( &bytes, 1, 1, file_handle ) == 1 )
    return bytes;
  else
    return 0;
}

/*------------------------------------------------------------------
 * Reads 16bit big-endian integer
 *------------------------------------------------------------------*/
UINT16 ICFile::read_int16()
{
  BYTE bytes[2];

  if( fread( &bytes, 1, 2, file_handle ) == 2 )
    return (bytes[1] << 8) + (bytes[0]);
  else
    return 0;
}

/*------------------------------------------------------------------
 * Reads 24bit big-endian integer
 *------------------------------------------------------------------*/
UINT32 ICFile::read_int24()
{
  BYTE bytes[3];

  if( fread( &bytes, 1, 3, file_handle ) == 3 )
    return (bytes[2] << 16) + (bytes[1] << 8) + (bytes[0]);
  else
    return 0;
}

/*------------------------------------------------------------------
 * Reads 32bit big-endian integer
 *------------------------------------------------------------------*/
UINT32 ICFile::read_int32()
{
  BYTE bytes[4];

  if( fread( &bytes, 1, 4, file_handle ) == 4 )
    return (bytes[3] << 24) + (bytes[2] << 16) + (bytes[1] << 8) + (bytes[0]);
  else
    return 0;
}

/*------------------------------------------------------------------
 * Write one byte
 *------------------------------------------------------------------*/
void ICFile::write_byte( BYTE data )
{
  if( fwrite( &data, 1, 1, file_handle ) != 1 )
  {
    perror( "Write error" );
    exit  ( 1 );
  }
}

/*------------------------------------------------------------------
 * Write 16bit big-endian integer
 *------------------------------------------------------------------*/
void ICFile::write_int16( UINT16 data )
{
  BYTE bytes[2];

  bytes[1] = (data >> 8     );
  bytes[0] = (data & 0x00FF );

  if( fwrite( &bytes, 1, 2, file_handle ) != 2 )
  {
    perror( "Write error" );
    exit  ( 1 );
  }
}

/*------------------------------------------------------------------
 * Write 24bit big-endian integer
 *------------------------------------------------------------------*/
void ICFile::write_int24( UINT32 data )
{
  BYTE bytes[3];

  bytes[2] = (data & 0x00FF0000UL ) >> 16;
  bytes[1] = (data & 0x0000FF00UL ) >>  8;
  bytes[0] = (data & 0x000000FFUL );

  if( fwrite( &bytes, 1, 3, file_handle ) != 3 )
  {
    perror( "Write error" );
    exit  ( 1 );
  }
}

/*------------------------------------------------------------------
 * Write 32bit big-endian integer
 *------------------------------------------------------------------*/
void ICFile::write_int32( UINT32 data )
{
  BYTE bytes[4];

  bytes[3] = (data >> 24 );
  bytes[2] = (data & 0x00FF0000UL ) >> 16;
  bytes[1] = (data & 0x0000FF00UL ) >>  8;
  bytes[0] = (data & 0x000000FFUL );

  if( fwrite( &bytes, 1, 4, file_handle ) != 4 )
  {
    perror( "Write error" );
    exit  ( 1 );
  }
}

