/* Icon Plus
 * Copyright (C) 2001 Dmitry A.Steklenev
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. All advertising materials mentioning features or use of this
 *    software must display the following acknowledgment:
 *    "This product includes software developed by Dmitry A.Steklenev".
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Dmitry A.Steklenev".
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR OR CONTRIBUTORS "AS IS"
 * AND ANY EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * AUTHOR OR THE CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: ic_main.cpp,v 1.10 2002/07/10 16:20:54 glass Exp $
 */

#include "ic_os2.h"
#include "ic_windows.h"
#include "ic_icl.h"
#include "ic_icon.h"
#include <stdlib.h>
#include <direct.h>

#ifdef IC_GETOPT_NEEDED
#include <getopt.h>
#endif

ICFile*    load_file;
ICFile*    save_file;
char*      save_path;
int        load_dump;
int        maps_dump;
icon_type  save_type = os2_v12;
icon_type  load_type = unknown;
ICIconList icon_list;

/*------------------------------------------------------------------
 * Returns type of the input file
 *------------------------------------------------------------------*/
static icon_type query_load_type()
{
  UINT16 type   = load_file->read_int16();
  UINT32 cbsize = load_file->read_int16();

  load_file->seek(0);

  switch( type )
  {
    case 0:
      if( cbsize == 0x01 )
        return win_ico;
      break;

    case OS2_BFT_BITMAPARRAY:   /* OS/2 Bitmap array file */
      if( cbsize == 0x5C )      /* version 2.0 */
        return os2_v20;
      if( cbsize == 0x28 )      /* version 1.2 */
        return os2_v12;
      break;

    case OS2_BFT_COLORICON:     /* OS/2 Color icon file   */
    case OS2_BFT_ICON:          /* OS/2 B&W   icon file   */
      if( cbsize == 0x4E )      /* version 2.0 */
        return os2_v20;
      if( cbsize == 0x1A )      /* version 1.2 */
        return os2_v12;
      break;

    case 0x5A4D:                /* 'MZ' executable */
    {
      UINT16 offset;
      UINT16 type;

      load_file->seek( 0x3C   );
      offset = load_file->read_int32();
      load_file->seek( offset );
      type   = load_file->read_int16();
      load_file->seek( 0x00   );

      if( type == 0x454E )      /* 'NE'executable */
        return win_icl;

      break;
    }
  }
  return unknown;
}

/*------------------------------------------------------------------
 * Generates program error
 *------------------------------------------------------------------*/
void static error( const char* msg )
{
  fprintf( stderr, "%s\n", msg );
  exit   ( 1 );
}

/*------------------------------------------------------------------
 * Display help
 *------------------------------------------------------------------*/
void static help()
{
  printf( "Usage: icplus [<options>...] <load_file> [<save_file>|-p <save_path>]\n" );
  printf( "Where:\n"
          "  <load_file>  is an icon file in OS/2 format, MS Windows format or\n"
          "               MS Windows icon library file.\n"
          "  <save_file>  is the name of the OS/2 icon file that will be created.\n"
          "  <save_path>  is the name of the directory where file(s) will be created.\n"
          "  <options>    -2 write an OS/2 v2.0 icon file rather than a v1.2 icon file.\n"
          "               -w write a Windows icon file rather than an OS/2 icon file.\n"
          "               -d dump all loaded structures.\n"
          "               -b dump all loaded bitmaps and structures.\n"
        );
  exit(1);
}

/*------------------------------------------------------------------
 * Writes icon file
 *------------------------------------------------------------------*/
void static save_icon()
{
  printf( "%d icons in file.\n", icon_list.count());

  if( !save_file->open_for_write())
    error( "Cannot create output file."  );

  switch( save_type )
  {
    case os2_v12:
    case os2_v20:
      ICOS2().save( save_file, save_type );
      break;

    case win_ico:
      ICWindows().save( save_file );
      break;

    default:
      error( "Can't wrote that type of file." );
  }
}

/*------------------------------------------------------------------
 * Program entry point
 *------------------------------------------------------------------*/
int main( int argc, char **argv )
{
  int c;

  printf( "Icon Plus Version 1.0.1. Icon conversion utility.\n"
          "Copyright (C) 2001-2002 Dmitry A.Steklenev\n\n" );

  if( argc < 2 )
    help();

  while(( c = getopt( argc, argv, "?dbp:w2" )) != EOF )
  {
    switch(c)
    {
      case '?': help();
                break;
      case 'd': load_dump = 1;
                break;
      case 'b': maps_dump = 1;
                load_dump = 1;
                break;
      case '2': save_type = os2_v20;
                break;
      case 'w': save_type = win_ico;
                break;
      case 'p': save_path = optarg;
                break;
      default:
        fprintf( stderr, "Illegal option ignored.\n" );
        break;
     }
  }

  for( ; optind < argc; optind++ )
  {
    if( !load_file )
      load_file = new ICFile( argv[optind] );
    else if( !save_file )
      save_file = new ICFile( argv[optind] );
    else
      error( "Too many filenames." );
  }

  if( !load_file )
    error( "You must specify input filename." );
  if( !load_file->open_for_read())
    error( "Cannot read input file." );

  load_type = query_load_type();

  if( save_path )
    mkdir( save_path );

  if( load_type != win_icl )
  {
    if( !save_file && !save_path )
      error( "You must specify input and output filenames." );

    if( save_path )
    {
      delete save_file;
      save_file = new ICFile( load_file->name(), save_path );
    }
  }

  switch( load_type )
  {
    case os2_v12:
    case os2_v20: ICOS2().load( load_file );
                  save_icon();
                  break;

    case win_ico: ICWindows().load( load_file );
                  save_icon();
                  break;
    case win_icl:
    {
      ICICL filter( load_file );
      while(( save_file = filter.find( save_path )) != 0 )
      {
        save_icon();
        icon_list.clear();
        delete save_file;
      }
      break;
    }

    default:
      error( "Can't read that type of file." );
  }

  delete load_file;
  delete save_file;
  return 0;
}