/* Icon Plus
 * Copyright (C) 2001 Dmitry A.Steklenev
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. All advertising materials mentioning features or use of this
 *    software must display the following acknowledgment:
 *    "This product includes software developed by Dmitry A.Steklenev".
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Dmitry A.Steklenev".
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR OR CONTRIBUTORS "AS IS"
 * AND ANY EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * AUTHOR OR THE CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: ic_palette.cpp,v 1.4 2001/05/31 05:39:08 glass Exp $
 */

#include "ic_palette.h"
#include <stdlib.h>
#include <memory.h>

const ICRGB ICRGB::black( 0x00, 0x00, 0x00 );
const ICRGB ICRGB::white( 0xFF, 0xFF, 0xFF );

const ICPalette::size_type ICPalette::npos = ICPalette::size_type(-1);

/*------------------------------------------------------------------
 * Constructs the empty color palette
 *------------------------------------------------------------------*/
ICPalette::ICPalette()
{
  pal_size = 0;
  pal_list = 0;
  pal_true = 0;
}

/*------------------------------------------------------------------
 * Constructs the color palette
 *------------------------------------------------------------------*/
ICPalette::ICPalette( size_type bits_per_pixel )
{
  if( bits_per_pixel >= 24 )
  {
    pal_true = 1;
    pal_size = 0;
    pal_list = 0;
  }
  else
  {
    pal_true = 0;
    pal_size = 1 << bits_per_pixel;
    pal_list = new ICRGB[pal_size];
  }
}

/*------------------------------------------------------------------
 * Constructs the color palette from another palette object
 *------------------------------------------------------------------*/
ICPalette::ICPalette( const ICPalette& palette )
{
  pal_size = palette.pal_size;
  pal_true = palette.pal_true;

  if( pal_size )
  {
    pal_list = new ICRGB[pal_size];
    memcpy( pal_list, palette.pal_list, sizeof(ICRGB)*pal_size );
  }
  else
    pal_list = 0;
}

/*------------------------------------------------------------------
 * Destructs the color palette
 *------------------------------------------------------------------*/
ICPalette::~ICPalette()
{
  delete[] pal_list;
}

/*------------------------------------------------------------------
 * Returns the RGB color index
 *------------------------------------------------------------------*/
ICPalette::size_type ICPalette::find( const ICRGB& color ) const
{
  if( pal_true )
    return color;
  else
  {
    for( size_type i = 0; i < count(); i++ )
      if( pal_list[i] == color )
        return i;
  }

  return npos;
}

/*------------------------------------------------------------------
 * Assignment operator
 *------------------------------------------------------------------*/
ICPalette& ICPalette::operator=( const ICPalette& palette )
{
  if( &palette != this )
  {
    delete[] pal_list;

    pal_size = palette.pal_size;
    pal_true = palette.pal_true;

    if( pal_size )
    {
      pal_list = new ICRGB[pal_size];
      memcpy( pal_list, palette.pal_list, sizeof(ICRGB)*pal_size );
    }
    else
      pal_list = 0;
  }

  return *this;
}

/*------------------------------------------------------------------
 * Compares this palette to another palette object for equality
 *------------------------------------------------------------------*/
int ICPalette::operator==( const ICPalette& palette ) const
{
  if(( pal_size != palette.pal_size ) ||
     ( pal_true != palette.pal_true )  )
  {
    return 0;
  }

  for( size_type i = 0; i < pal_size; i++ )
    if( pal_list[i] != palette[i] )
      return 0;

  return 1;
}

/*------------------------------------------------------------------
 * Compares this palette to another palette object for inequality
 *------------------------------------------------------------------*/
int ICPalette::operator!=( const ICPalette& palette ) const
{
  return !operator==(palette);
}

/*------------------------------------------------------------------
 * Returns standard black and white color palette
 *------------------------------------------------------------------*/
ICPalette ICPalette::default_bw()
{
  ICPalette palette(1);
  palette[0] = ICRGB::black;
  palette[1] = ICRGB::white;

  return palette;
}

