//----------------------------------------------------------------------------
//
// C++ Objects for Allegro's gui
//
// Douglas Eleveld (D.J.Eleveld@anest.azg.nl)
//
//----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Functions for the main menu
//----------------------------------------------------------------------------
#include <stdio.h>
#include <string.h>
#include "degui.h"
#include "build_mu.h"
#include "build_di.h"
#include "descript.h"

int driver = GFX_AUTODETECT;
int xres = 640;
int yres = 480;

//-----------------------------------------------------------------------------
// Export to a dialog
int dialog_export (void)
   {
   // Get an export filename
   char filename[80];
   strcpy(filename,"");
   int ret = file_select("Export C++ objects to file",filename,"cc");

   // If it wasn't cancelled then export the thing
   if(ret!=0) gui_builder->export(filename);

   return D_O_K;
   }
//----------------------------------------------------------------------------
// Save and load a partially done dialog
int save_dialog (void)
   {
   // Get an save filename
   char filename[80];
   strcpy(filename,"");
   int ret = file_select("Save to file",filename,"adf");

   // If it wasn't cancelled then save the thing
   if(ret!=0)
      {
      delete_file(filename);
      gui_builder->save(filename);
      }
   return D_O_K;
   }
//----------------------------------------------------------------------------
int load_dialog (void)
   {
   // Get an load filename
   char filename[80];
   strcpy(filename,"");
   int ret = file_select("Load from file",filename,"adf");

   // If it wasn't cancelled then load the thing
   if(ret!=0)
      {
      // Try to load the file
      if(gui_builder->load(filename)==false)
         {
         // Tell the user if it didn't work
         alert(NULL,"Problem loading file",NULL,"&Ok",NULL,'o',0);
         }
      gui_builder->redraw();
      }
   return D_O_K;
   }
//----------------------------------------------------------------------------
int append_dialog (void)
   {
   // Get an load filename
   char filename[80];
   strcpy(filename,"");
   int ret = file_select("Append from file",filename,"adf");

   // If it wasn't cancelled then load the thing
   if(ret!=0)
      {
      // Try to load the file
      if(gui_builder->load(filename,true)==false)
         {
         // Tell the user if it didn't work
         alert(NULL,"Problem appending file",NULL,"&Ok",NULL,'o',0);
         }
      }
   gui_builder->redraw();
   return D_O_K;
   }

//----------------------------------------------------------------------------
// Close the dialog from the menu
int main_dialog_close (void)
   {
   // Check if there are objects to save
   if(gui_builder->number_objects()>0)
      // Ask if they want to save at the exit
      if(alert(NULL,"Save dialog before exit?",NULL,"&Yes","&No",'n','n')==1)
         save_dialog();

   // As the main dialog to close itself
   gui_builder->exit();
   return D_O_K;
   }

//----------------------------------------------------------------------------
// General dialog procedures
//----------------------------------------------------------------------------
int set_builder_palette (void)
   {
   char filename[80];
   strcpy(filename,"");
   if(file_select("Choose file to load palette from",filename,"bmp;tga;lbm;pcx")!=0)
      {
      // Find the type of file it is and load it
      BITMAP *b;
      PALETTE pal;
      if(stricmp(get_extension((char*)filename),"bmp")==0) b = load_bmp((char*)filename,pal);
      else if(stricmp(get_extension((char*)filename),"tga")==0) b = load_tga((char*)filename,pal);
      else if(stricmp(get_extension((char*)filename),"lbm")==0) b = load_lbm((char*)filename,pal);
      else b = load_pcx((char*)filename,pal);

      // Adjust the gui for the best colors
      set_palette(pal);
      reset_default_object_colors(false);

      destroy_bitmap(b);
      gui_builder->redraw();
      }
   return D_O_K;
   }
//----------------------------------------------------------------------------
int center_dialog (void)
   {
   gui_builder->center(3);
   gui_builder->redraw();
   return D_O_K;
   }
//----------------------------------------------------------------------------
int placement_granularity (void)
   {
   char gran_string[10];
   sprintf(gran_string,"%i",gui_builder->get_granularity());

   // Object definitions
   panel_raised_object object_0;
   button_object object_1("&Ok",0,true);
   button_object object_2("&Cancel",0,true);
   text_object title("Set placemet grid");
   text_object grid_size("Grid size:");
   edittext_object gran_edit(gran_string,4);

   // Add Objects to the dialog
   dialog the_dialog;
   the_dialog.add(object_0,165,160,305,160);
   the_dialog.add(object_1,215,285,80,20);
   the_dialog.add(object_2,350,285,80,20);
   the_dialog.add(title,244,170,136,8);
   the_dialog.add(grid_size,192,224,80,8);
   the_dialog.add(gran_edit,273,224,40,8);
   the_dialog.center();

   // Ask the user to change the granularity
   if(the_dialog.popup()==&object_1)
      {
      // Change the granularity according to the user
      int gran = atoi(gran_edit.get_text());
      if(gran<1) gran = 1;

      gui_builder->set_granularity(gran);
      gui_builder->redraw();
      }
   return D_O_K;
   }
//----------------------------------------------------------------------------
int choose_resolution (void)
   {
   int res_c = driver, res_w = xres, res_h = yres;
   if(gfx_mode_select(&res_c,&res_w,&res_h)!=0)
      {
      show_mouse(NULL);
      PALETTE pal;
      get_palette(pal);
      if(set_gfx_mode(res_c,res_w,res_h,0,0)!=0)
         {
         // Confirm bad mode setting
         set_gfx_mode(driver,xres,yres,0,0);
         set_palette(pal);
         alert("Could not set graphics mode",allegro_error,0,"Ok",0,0,0);
         }
      else
         {
         // Change graphics mode so it gets saved in the INI file
         driver = res_c;
         xres = res_w;
         yres = res_h;
         }
      set_palette(pal);
      show_mouse(screen);
      gui_builder->redraw();
      }
   return D_O_K;
   }

//----------------------------------------------------------------------------
// Help procedures
//----------------------------------------------------------------------------
// Uhh... it's a bit difficult to read like this....
char *menutext = "
MENU DESCRIPTION

Here is a short description of each element of the builder main menu:

File menu

Load
- Load a DEGUI builder file (ADF)

Append
- Append an ADF file to the current objects

Save
- Save the current objects to an ADF file

Export
- Export the current object to C++ code

Exit
- Quit the DEGUI builder


Object menu

Box
- Create a new box object

Shadow Box
- Create a new shadow box object

Panel raised
- Create a raised panel object

Panel sunken
- Create a sunken panel object

Panel ridge
- Create a panel with a ridge around it

Panel groove
- Create a panel with a groove around it

Bitmap
- Create a new bitmap object

Text
- Create a new text object

Centerd text
- Create a new centerd text object

Editable text
- Create a new editable text object

Button
- Create a new button object

Checkbox
- Create a new checkbox object

Radio button
- Create a new radio button object

List Box
- Create a new listbox object

Textbox
- Create a new textbox object

Progress bar
- Create a new progress bar object

User
- Create a new user defined object


Option menu

Palette
- Load a palette from a file and use it.  The default object colors will be adjusted to the closest colors in the new palette

Center
- Center all the object to the center of the screen

Granularity
- Change the movement grid spacing

Resolution
- Change the graphics resolution


Help menu

Help
- You are reading it

About
- Information about the program

";
char *keytext = "
KEY ASSIGNMENTS

Here are the keys that may be useful during the building of your dialog.  Put the mouse over the object you want to change and then press one of the following keys.  Note that you DON'T have to click the mouse.

F1 
Move object - Click to put the object back down

F2 
Multi-move object - This moves the object pointed to and all the objects that are inside it. - Click to put the object(s) back down

F3 
Resize object - Click when the object is the desired size

F4 
Push object back - This pushes the object lower in the dialog and it will be redraw earlier whenever the dialog is redrawn.  It sort of pushes the object behind other objects.

F5
Pull object to front - This pull the object higher in the dialog and it will be redraw later whenever the dialog is redrawn.  It sort of pulls the object in front of other objects.

F10
Delete an object - This removes the object from the current dialog
\n";

textbox_object *textobj;
button_object *menubutt;
button_object *keybutt;

// Set the buttons and text when clicked on the menu button
int menucallback (void)
   {
   textobj->set_text(menutext);
   menubutt->select();
   keybutt->deselect();
   return D_O_K;
   }

// Set the buttons and text when clicked on the keys button
int keycallback (void)
   {
   textobj->set_text(keytext);
   menubutt->deselect();
   keybutt->select();
   return D_O_K;
   }

int help_procedure (void)
   {
   // Object definitions - DEGUI 1.2 beta
   panel_raised_object object_1;
   textbox_object helptext(keytext,true,true);
   button_object object_3("Done",0,true);
   button_object object_4("Keys",0,false);
   text_object title("DEGUI Help");
   button_object object_6("Menus",0,false);

   // Set the pointers for the callback
   textobj = &helptext;
   menubutt = &object_6;
   keybutt = &object_4;

   // Add Objects to the dialog
   dialog the_dialog;
   the_dialog.add(object_1,105,92,430,295);
   the_dialog.add(helptext,108,136,424,248);
   the_dialog.add(object_3,480,114,52,18);
   the_dialog.add(object_4,108,114,52,18);
   the_dialog.add(title,270,97,80,8);
   the_dialog.add(object_6,164,114,60,18);

   // Set the callbacks
   object_4.set_key_callback(keycallback);
   object_6.set_key_callback(menucallback);
   object_4.set_click_callback(keycallback);
   object_6.set_click_callback(menucallback);

   object_4.select();

   the_dialog.center();
   the_dialog.popup();

   return D_O_K;
   }
//----------------------------------------------------------------------------
int about_procedure (void)
   {
   // Save the palette so we can replace it later
   PALETTE pal;
   get_palette(pal);

   char *filename = "about.pcx";

   char version[50];
   sprintf(version,"DEGUI %s",DEGUI_VERSION_STR);

   // Object definitions
   panel_raised_object object_0;
   bitmap_object picture(filename);
   text_object object_2("Doug Eleveld's");
   text_object object_3(version);
   text_object object_4("for Allegro");
   button_object object_5("Ok",13,true);
   text_object object_6("C++ GUI builder");

   // Setup the correct palette for the about bitmap
   PALETTE about;
   BITMAP *bit = load_pcx(filename,about);
   set_palette(about);
   reset_default_object_colors(false);
   gui_builder->redraw(true);

   // Add Objects to the dialog
   dialog the_dialog;
   the_dialog.add(object_0,234,129,171,264);
   the_dialog.add(picture,241,223,157,118);
   the_dialog.add(object_2,261,150,112,8);
   the_dialog.add(object_3,256,203,128,8);
   the_dialog.add(object_4,276,177,88,8);
   the_dialog.add(object_5,267,354,110,26);
   the_dialog.add(object_6,258,164,120,8);
   the_dialog.center();

   // Execute the dialog
   the_dialog.popup();
   destroy_bitmap(bit);

   // Put the palette back to what it was
   set_palette(pal);
   reset_default_object_colors(false);
   gui_builder->redraw();

   return D_O_K;
   }

//----------------------------------------------------------------------------
// Object adding procedures
//----------------------------------------------------------------------------
int add_box_object (void)
   {
   return gui_builder->add_box_object();
   }
//----------------------------------------------------------------------------
int add_shadow_box_object (void)
   {
   return gui_builder->add_shadow_box_object();
   }
//----------------------------------------------------------------------------
int add_panel_raised_object (void)
   {
   return gui_builder->add_panel_raised_object();
   }
//----------------------------------------------------------------------------
int add_panel_sunken_object (void)
   {
   return gui_builder->add_panel_sunken_object();
   }
//----------------------------------------------------------------------------
int add_panel_ridge_object (void)
   {
   return gui_builder->add_panel_ridge_object();
   }
//----------------------------------------------------------------------------
int add_panel_groove_object (void)
   {
   return gui_builder->add_panel_groove_object();
   }
//----------------------------------------------------------------------------
int add_bitmap_object (void)
   {
   return gui_builder->add_bitmap_object();
   }
//----------------------------------------------------------------------------
int add_text_object (void)
   {
   return gui_builder->add_text_object();
   }
//----------------------------------------------------------------------------
int add_centerd_text_object (void)
   {
   return gui_builder->add_centerd_text_object();
   }
//----------------------------------------------------------------------------
int add_edittext_object (void)
   {
   return gui_builder->add_edittext_object();
   }
//----------------------------------------------------------------------------
int add_button_object (void)
   {
   return gui_builder->add_button_object();
   }
//----------------------------------------------------------------------------
int add_checkbox_object (void)
   {
   return gui_builder->add_checkbox_object();
   }
//----------------------------------------------------------------------------
int add_radio_button_object (void)
   {
   return gui_builder->add_radio_button_object();
   }
//----------------------------------------------------------------------------
int add_list_object (void)
   {
   return gui_builder->add_list_object();
   }
//----------------------------------------------------------------------------
int add_textbox_object (void)
   {
   return gui_builder->add_textbox_object();
   }
//----------------------------------------------------------------------------
int add_progress_bar_object (void)
   {
   return gui_builder->add_progress_bar_object();
   }
//----------------------------------------------------------------------------
int add_window_object (void)
   {
   return gui_builder->add_window_object();
   }
//----------------------------------------------------------------------------
int add_user_object (void)
   {
   return gui_builder->add_user_object();
   }
//----------------------------------------------------------------------------
int add_hidden_list_object (void)
   {
   return gui_builder->add_hidden_list_object();
   }
//----------------------------------------------------------------------------

