//----------------------------------------------------------------------------
//
// C++ Objects for Allegro's gui
//
// Douglas Eleveld (D.J.Eleveld@anest.azg.nl)
//
//----------------------------------------------------------------------------

/*

OBJECT DERIVATION EXAMPLE

  Here is an example of using object derivation to interface with
  the dialog objects.  Just derive an object from the most similar base
  object and then fill in the virtual functions that you need to define
  the behaviour that you want.

  This is better for data hiding than using callback functions because
  you can hide data inside your derived objects.  Here the
  DATAFILE *datafile is hidden inside the object, wheras in ex2.cc
  the datafile is global.  It's better to have the datafile hidden in
  the object because if there is something going wrong with the datafile
  you know that the problem must be in the derived class.  If the datafile
  is global then the problem could be anywhere in the entire program.
  Keeping the data hidden helps debugging and you can write more complex
  programs faster.

*/

#include <stdlib.h>
#include <stdio.h>

#include "allegro.h"
#include "example.h"

// Include the dialog objects
#include "degui.h"

//----------------------------------------------------------------------------
// A custom dialog object for the 'change font' button
class font_button_object
		:public button_object
		{
		private:
            /* we need to load example.dat to access the big font */
            DATAFILE *datafile;

            /* since we change the font, we need to store a copy of the original one */
            FONT *original_font;

            // Actually switch the fonts
            void switch_fonts (void);

		protected:
				virtual void msg_key (void);
				virtual void msg_click (void);

		public:
            // Constructor
				font_button_object (const char*, const char);
            // Destructor
            virtual ~font_button_object (void);
		};

// Font button object constructor
font_button_object::font_button_object (const char *tex, const char c)
   :button_object(tex,c,false)
   {
   // Remember the font
   original_font = font;

   // Load the data file
   datafile = load_datafile("example.dat");
   if(!datafile)
      {
      allegro_exit();
      printf("Error loading example.dat!\n\n");
      exit(1);
      }
   }

// Font button object destructor
font_button_object::~font_button_object (void)
   {
   // Release the datafile
   unload_datafile(datafile);
   }

// Function to change the font when a key is pressed on the font object
void font_button_object::msg_key (void)
	{
   switch_fonts();
	}

// Also change the font on a click
void font_button_object::msg_click (void)
	{
   switch_fonts();

	// Wait for the button up
	while(mouse_b){};
	}

// Actually do the work of switching the fonts
void font_button_object::switch_fonts (void)
   {
	if(font==original_font)
		font = (FONT*)datafile[BIG_FONT].dat;
	else
		font = original_font;

   // Now we need to update the whole screen
   parent()->redraw();
   }

//----------------------------------------------------------------------------
int main(void)
   {
   /* initialise everything */
   allegro_init();
   install_keyboard(); 
   install_mouse();
   install_timer();

   // Set up the graphics mode
   set_gfx_mode(GFX_AUTODETECT,640,480,0,0);
   set_pallete(desktop_pallete);

   set_default_object_colors(0,235,236,237,238,239,240,241);

   // Declare the dialog and some objects
   dialog the_dialog;
   clear_screen_object  clear_obj;
   text_object			   test_text("Some text");
   button_object			ok_button("OK",0,true);

   // Here is our derived object
   font_button_object	change_font("Change &font",'f');

   // Add the objects to the dialog
   // The color arguments have defaults so we don't have to fill then in
                              // (x)   (y)   (w)   (h)   (fg)  (bg)
   the_dialog.add(clear_obj);
   the_dialog.add(test_text,      80,   182,  160,  48 );
   the_dialog.add(change_font,    80,   240,  160,  48 );
   the_dialog.add(ok_button,      360,  400,  160,  48 );

   // Do the dialog
   the_dialog.execute();

   show_mouse(NULL);
   set_gfx_mode(GFX_TEXT,80,25,0,0);
   allegro_exit();
   return 0;
   }
//----------------------------------------------------------------------------
