//----------------------------------------------------------------------------
//
// C++ Objects for Allegro's gui
//
// Douglas Eleveld (D.J.Eleveld@anest.azg.nl)
//
//----------------------------------------------------------------------------
#include "degui.h"
#include "internal.h"

//----------------------------------------------------------------------------
// Stuff for handling the default behaviour in callback functions
int dialog_object::default_message = 0;
dialog_object* dialog_object::default_object = NULL;
int dialog_object::default_data = 0;
int dialog_object::default_answer = D_O_K;

//----------------------------------------------------------------------------
// The main dispatching function i.e. the interface between the C and C++ object
int dialog_object::object_message_proc (int msg, DIALOG *d, int c)
   {
   // Make sure that there IS an object there
   if((d==NULL)||(d->dp==NULL)) return D_O_K;

   // Possibly problematic cast
   dialog_object *object = (dialog_object*)(d->dp);
   if(object==NULL) return D_O_K;  // Just some protection from wierdness

   // Tell the object what it's dialog is
   // Normally this won't happen, but We'll check just in case
   if(object->d!=d) object->connect(d);

   // Possibly do a callback on every message
   if(object->all_callback!=NULL) object->all_callback();

   // Setup any default behaviour
   dialog_object::default_message = msg;
   dialog_object::default_object = object;
   dialog_object::default_data = c;
   dialog_object::default_answer = D_O_K;

   bool request_close = false;
   int callback_ret = D_O_K;

   // Construct the Allegro style answer
   // If we called a callback function then the dialog_object::default_answer
   // will be set correctly else it will just be D_O_K like we set it at the
   // beginning of this function
   int ret = dialog_object::default_answer;

   // Convert the allegro message to a call of a member function
   // or the call of a callback function
   switch(msg)
      {
      case(MSG_START):
         if(object->start_callback!=NULL) callback_ret = object->start_callback();
         else object->msg_start();
         break;
      case(MSG_END):
         if(object->end_callback!=NULL) callback_ret = object->end_callback();
         else object->msg_end();
         break;
      case(MSG_DRAW):
         if(object->draw_callback!=NULL) callback_ret = object->draw_callback();
         else object->msg_draw();
         object->_redraw = false;
         break;
      case(MSG_CLICK):
         if(object->click_callback!=NULL) callback_ret = object->click_callback();
         else object->msg_click();

			// Check if we exit on click
			if(object->d->flags&D_EXIT) request_close = true;
         break;
      case(MSG_DCLICK):
         if(object->dclick_callback!=NULL) callback_ret = object->dclick_callback();
         else object->msg_dclick();
         break;
      case(MSG_KEY):
         if(object->key_callback!=NULL) callback_ret = object->key_callback();
         else object->msg_key();
         if(object->demand_focus==true) ret |= D_WANTFOCUS;

			// Check if we exit on keypress
			if(object->d->flags&D_EXIT) request_close = true;
         break;
      case(MSG_CHAR):
         if(object->char_callback!=NULL) callback_ret = object->char_callback(c);
         else if(object->msg_char(c)==true) ret = D_USED_CHAR;
         break;
      case(MSG_XCHAR):
         if(object->xchar_callback!=NULL) callback_ret = object->xchar_callback(c);
         else if(object->msg_xchar(c)==true) ret = D_USED_CHAR;
         break;
      case(MSG_WANTFOCUS):
         if(object->wantfocus_callback!=NULL) callback_ret = object->wantfocus_callback();
         else if(object->msg_wantfocus()==true) ret = D_WANTFOCUS;
         if(object->demand_focus==true) ret =  D_WANTFOCUS;
         break;
      case(MSG_GOTFOCUS):
         if(object->gotfocus_callback!=NULL) callback_ret = object->gotfocus_callback();
         else object->msg_gotfocus();
         break;
      case(MSG_LOSTFOCUS):
         if(object->lostfocus_callback!=NULL) callback_ret = object->lostfocus_callback();
         else object->msg_lostfocus();
         if(object->demand_focus==true) ret |= D_WANTFOCUS;
         break;
      case(MSG_GOTMOUSE):
         if(object->gotmouse_callback!=NULL) callback_ret = object->gotmouse_callback();
         else object->msg_gotmouse();
         break;
      case(MSG_LOSTMOUSE):
         if(object->lostmouse_callback!=NULL) callback_ret = object->lostmouse_callback();
         else object->msg_lostmouse();
         break;
      case(MSG_IDLE):
         if(object->idle_callback!=NULL) callback_ret = object->idle_callback();
         else object->msg_idle();
         break;
      case(MSG_RADIO):
         if(object->radio_callback!=NULL) callback_ret = object->radio_callback(c);
         else object->msg_radio(c);
         break;
      default:
         if(object->unknown_callback!=NULL) callback_ret = object->unknown_callback(msg);
         else object->msg_unknown(msg);
         break;
      }
   // Check if the object needs a redraw
   if(object->_redraw==true)
      {
      show_mouse(NULL);
      if(object->draw_callback!=NULL) callback_ret = object->draw_callback();
      else object->msg_draw();
      object->_redraw = false;
      show_mouse(screen);
      }

   // Make sure that any wierd calls to the default behaviour don't do anything
   dialog_object::default_message = MSG_IDLE;
   dialog_object::default_object = NULL;
   dialog_object::default_data = 0;

   // Modify the return value depending on what the
   // callback funtion has asked for
   ret |= callback_ret;

   if(request_close==true) ret |= D_CLOSE;

   return ret;
   }

//----------------------------------------------------------------------------
// Dialog base class object functions
//----------------------------------------------------------------------------
// Constructor
dialog_object::dialog_object (void)
   :d(new DIALOG),
   connected(false),
   in_c_dialog(false),
   _parent(NULL),
   _window(NULL),
   _redraw(false),
   all_callback(NULL),
   start_callback(NULL),
   end_callback(NULL),
   draw_callback(NULL),
   click_callback(NULL),
   dclick_callback(NULL),
   key_callback(NULL),
   char_callback(NULL),
   xchar_callback(NULL),
   wantfocus_callback(NULL),
   gotfocus_callback(NULL),
   lostfocus_callback(NULL),
   gotmouse_callback(NULL),
   lostmouse_callback(NULL),
   idle_callback(NULL),
   radio_callback(NULL),
   unknown_callback(NULL),
   demand_focus(false),
   color(NULL),
   dataptr(NULL)
   {
   // Check the d pointer
   if(d==NULL) degui_no_memory();

   d->proc = object_message_proc;
   d->x = 0;
   d->y = 0;
   d->w = 0;
   d->h = 0;
   d->fg = gui_fg_color;
   d->bg = gui_bg_color;
   d->key = 0;
   d->flags = 0;
   d->d1 = 0;
   d->d2 = 0;
   d->dp = this;
   }
//----------------------------------------------------------------------------
// Destructor
dialog_object::~dialog_object (void)
   {
   if(connected==false) delete d;
   }
//----------------------------------------------------------------------------
// Basic message passing functions

// Initialization of the dialog
void dialog_object::msg_start (void)
   {
   }
// De-initialization of the dialog
void dialog_object::msg_end (void)
   {
   }
// Tell the object to draw itself
void dialog_object::msg_draw (void)
   {
   _redraw = false;
   }
// Tell the object to deal with a mouse click default is the same as a keypress
void dialog_object::msg_click (void)
   {
   msg_key();
   while(mouse_b);
   }
// Tell the object to deal with a double click
void dialog_object::msg_dclick (void)
   {
   }
// Tell the object that a shortcut key was pressed
void dialog_object::msg_key (void)
   {
   }
// Tell the object that a key was pressed while it had the focus
// This should return true if the key was used
bool dialog_object::msg_char (const int)
   {
   return false;
   }
// Tell the object that a key was pressed while it did noty have the focus
// This should return true if the key was used
bool dialog_object::msg_xchar (const int)
   {
   return false;
   }
// Ask the object if they want the focus
bool dialog_object::msg_wantfocus (void)
   {
   return false;
   }
// Tell the object that the got or lost the focus
void dialog_object::msg_gotfocus (void)
   {
   }
void dialog_object::msg_lostfocus (void)
   {
   }
// Tell the object that they got or lost the mouse
void dialog_object::msg_gotmouse (void)
   {
   }
void dialog_object::msg_lostmouse (void)
   {
   }
// Tell the object that the object manager is bored
void dialog_object::msg_idle (void)
   {
   }
// Deselect grouped radio buttons
void dialog_object::msg_radio (const int)
   {
   }
// Get an hitherto unknown message
void dialog_object::msg_unknown (const int)
   {
   }
//-----------------------------------------------------------------------------
// Create and use a DIALOG strucure for an object
DIALOG dialog_object::create_DIALOG (const int x, const int y, const int w, const int h, const int fg, const int bg)
   {
   // Place the object
   if(x>=0) d->x = x;
   if(y>=0) d->y = y;
   if(w>=0) d->w = w;
   if(h>=0) d->h = h;
   if(fg>=0) d->fg = fg;
   if(bg>=0) d->bg = bg;

   in_c_dialog = true;

   return *d;
   }
//----------------------------------------------------------------------------
// Connect and disconnect the dialog_object from a DIALOG
// When not connected, it has it's own allocated DIALOG
void dialog_object::connect (DIALOG *newd)
   {
   // Make sure we don't connect to ourselves
   if(d==newd) return;

   if(in_c_dialog==false)
      {
      newd->x = d->x;
      newd->y = d->y;
      newd->w = d->w;
      newd->h = d->h;
      newd->key = d->key;
      newd->flags = d->flags;
      newd->d1 = d->d1;
      newd->d2 = d->d2;
      newd->dp = this;
      }

   // Set the new d pointer
   if(connected==false)
      {
      delete d;
      connected = true;
      in_c_dialog = false;
      }
   d = newd;
   }
//----------------------------------------------------------------------------
void dialog_object::disconnect (void)
   {
   if(connected==true)
      {
      // Make the new d pointer
      DIALOG *newd = new DIALOG;

      // Copy init info over
      newd->x = d->x;
      newd->y = d->y;
      newd->w = d->w;
      newd->h = d->h;
      newd->key = d->key;
      newd->flags = d->flags;
      newd->d1 = d->d1;
      newd->d2 = d->d2;
      newd->dp = this;

      // Set the new d pointer
      d = newd;
      connected = false;
      in_c_dialog = false;
      _parent = NULL;
      }
   }
//----------------------------------------------------------------------------
// Send a message through the passing procedure
int dialog_object::send_message (int msg, int c)
   {
   return object_message_proc(msg,d,c);
   }

//----------------------------------------------------------------------------
// The default behaviour in callback functions
//----------------------------------------------------------------------------
void dialog_object::default_behaviour (void)
   {
   // Make sure we have been called properly
   if(default_object==NULL) return;

	// Convert the allegro message to a call of a member function
   switch(default_message)
      {
      case(MSG_START):
         default_object->msg_start();
         break;
      case(MSG_END):
         default_object->msg_end();
         break;
      case(MSG_DRAW):
         default_object->msg_draw();
         break;
      case(MSG_CLICK):
         default_object->msg_click();

			// Check if we exit on keypress
			if(default_object->d->flags&D_EXIT) dialog_object::default_answer |= D_CLOSE;
         break;
      case(MSG_DCLICK):
         default_object->msg_dclick();
         break;
      case(MSG_KEY):
         default_object->msg_key();

			// Check if we exit on keypress
			if(default_object->d->flags&D_EXIT) dialog_object::default_answer |= D_CLOSE;
         break;
      case(MSG_CHAR):
         if(default_object->msg_char(default_data)==true) dialog_object::default_answer = D_USED_CHAR;
         break;
      case(MSG_XCHAR):
         if(default_object->msg_xchar(default_data)==true) dialog_object::default_answer = D_USED_CHAR;
         break;
      case(MSG_WANTFOCUS):
         if(default_object->msg_wantfocus()==true) dialog_object::default_answer = D_WANTFOCUS;
         break;
      case(MSG_GOTFOCUS):
         default_object->msg_gotfocus();
         break;
      case(MSG_LOSTFOCUS):
         default_object->msg_lostfocus();
         break;
      case(MSG_GOTMOUSE):
			default_object->msg_gotmouse();
         break;
      case(MSG_LOSTMOUSE):
         default_object->msg_lostmouse();
         break;
      case(MSG_IDLE):
         default_object->msg_idle();
         break;
      case(MSG_RADIO):
         default_object->msg_radio(default_data);
         break;
      default:
         default_object->msg_unknown(default_message);
         break;
      }
   }
//----------------------------------------------------------------------------



