#ifndef M3D_H_
#define M3D_H_

#include <allegro.h>
#include <stdlib.h>
#include <math.h>

/* total max # of faces/lights */
#define TOTAL_FACES         16384
#define TOTAL_LIGHTS        256

/* total number of textures to support in the 8bpp conversion code */
#define TOTAL_TEXTURES      1024

/* new polygon type */
#define POLYTYPE_WIRE       31          /* wire frame */

/* light flags */
#define LIGHT_DIRECTIONAL   0x00000001  /* is it a directional light? */
#define LIGHT_FADE          0x00000002  /* does it fade? */
#define LIGHT_INVERSE       0x00000004  /* dark lights (makes scenes darker) */

typedef struct _face {
    V3D_f v[3];             /* verteces */
    float norm[3];          /* face normal */
    BITMAP *texture;        /* texture for face */
    unsigned char shadetype,/* what type of shading? */
                  backface; /* draw the backface? */
} face;

typedef struct _obj {
    int     nfaces;         /* number of faces */
    face    *faces;         /* the faces */
    float   Xa,Ya,Za,       /* rotations */
            Xp,Yp,Zp;       /* positions */
    int     nchild;         /* number of children */
    struct _obj *child;     /* child object */
} obj;

typedef struct _light {
    float   Xa,Ya,Za,       /* (direction) angle of the light */
            Xp,Yp,Zp,       /* where is it? */
            fadedist,       /* dist at which power halves */
            intensity,      /* intensity of light */
            falloff;        /* falloff exponent.  you should normally use
                             * 2.0 but if you want spot lights you can use
                             * higher values.  Around 75.0 gives a neat
                             * spot light */
    int     flags;          /* flags for the light */
} light;

/* Starts the rendering process.  You pass it 'dest' which is where it should
 * render the graphics and 'camera' which is an appropriately setup camera
 * matrix
 */
void m3d_Begin(BITMAP *dest, MATRIX_f *camera);

/* Adds a light to the render queue.  All objects rendered before this
 * will not take this new light into account */
void m3d_RenderLight(light *l);

/* Adds an object to the render queue. */
void m3d_RenderObj(obj *o);

/* Draws the objects to the surface (set 'zbuffered' if the objects are
 * all zbufferable)
 */
void m3d_Draw(int zbuffered);

/* Allocates memory for an object */
obj *m3d_CreateObj(int nofaces, int nochild);

/* Frees all memory from an object, does not free textures */
void m3d_DeleteObj(obj *o);

/* Allocates ram for a new object then copies from a source */
obj *m3d_CopyObj(obj *src);

/* Allocates ram for a new object and makes the faces from a .VTX file
 * The .VTX file must have under 50,000 points/faces per faceset and it doest
 * read textures (only ambient color).  You can apply textures afterwards
 * since it does read u/v coords
 */
obj *m3d_LoadVTX(char *filename);

/* Export an object as a simple VTX file.  This is to be loaded only by
 * the m3d_LoadVTX function.  It doesn't save colors or textures but does
 * preserve verteces, texture u/v coords and faces.  It also makes files
 * larger then their original since it doesn't collapse verteces.
 */
void m3d_ExportVTX(obj *o, char *name);

/* Changes the shading type of all faces and all child faces */
void m3d_ChangeShading(obj *o, int newtype);

/* Changes the backface drawing flag of an object and it's childs */
void m3d_SetBackface(obj *o, int newsetting);

/* Maps a texture onto an object.  The u/v coords of all verteces must be
 * setup using a 16.16 fixed point system (as done by the VTX reader).
 */
void m3d_ApplyTexture(obj *o, BITMAP *txt);

/* Maps a texture onto a single face.  The u/v coords must be setup as
 * described above.
 */
void m3d_ApplyTextureFace(face *f, BITMAP *txt);

/* Changes the color of the verteces to the specified color */
void m3d_ApplyColor(obj *o, unsigned long color);

/* Stretches the verteces of an object.  You may need to FlipNormals if you
 * loaded a .VTX since this recalcs the normals.
 */
void m3d_StretchObj(obj *o, float x, float y, float z);

/* Calcs the normals for all faces in the objects */
void m3d_CalcNormals(obj *o);

/* Flips the normals for all the faces */
void m3d_FlipNormals(obj *o);

/* Starts the 8bpp palette code
 *
 * You use this code to convert all your textures to an 8bpp format, to
 * setup an 8bpp palette that could be used for the colors required and
 * also setup the rgb_map global pointer.  Note that this does change the
 * hardware palette.  Note that original bitmaps that were remapped are
 * removed from memory as well.  Note that textures must be in truecolor 
 * format for the remapping to work.  (i.e 15/16/24/32 bpp color)
 */
void m3d_MakePalInit(void);

/* Add a 24bpp RGB color (and it's shades) to the color count table */
void m3d_MakePalAddColor(unsigned long color, unsigned long shades);

/* Adds an object to the color count table with the specified number
 * of shades for each color.  The more shades the smoother objects appear
 * (the # of shades can be from 1 to 32)
 */
void m3d_MakePalObj(obj *o, unsigned long shades);

/* Adds the pixels of a truecolor bitmap to the color count tables.  This
 * is for adding colors that are not part of objects yet.
 */
void m3d_map_bitmap(BITMAP *src, unsigned long shades);

/* Remaps a truecolor bitmap to the palette setup by m3d_MakePal().  You
 * must call this after your palette is setup.
 */
void m3d_remap_bitmap(BITMAP *src, BITMAP *dest);

/* Like the above except it creates a new bitmap and returns it.  The new
 * bitmap is a 8bpp bitmap using the correct palette
 */
BITMAP *m3d_remap_bitmap_ex(BITMAP *src);

/* Makes a 8bpp palette, rgp_map and color_map from the given color counts,
 * it also frees the memory allocated from m3d_MakePalInit();
 */
void m3d_MakePal(void);

/* Loads bitmaps such that they are always in a truecolor format.  This is
 * for use with the texture mapping code.  If you plan on supporting 8bpp
 * modes or use 8bpp textures you need to use this instead of load_bitmap()
 */
BITMAP *load_truecolor(char *name);

/* this is the palette setup by m3d_MakePal() */
extern PALETTE m3dpal;

/* Clipping related stuff */
extern float sframes,           /* number of frames drawn */
             sfaces,            /* number of faces drawn */
             clipdist;          /* distance (+Z) before clipping */
#endif
