//////////////////////////////////////////////////////////////////////////////
// Quake 1 Model implementation
//////////////////////////////////////////////////////////////////////////////
#include <stdio.h>
#include <math.h>
#include "mdlib.h"





// load a Q1 model file
q1mdl_t Q1LoadFile(char *filename)
{
    PACKFILE *mdl_file;
    long skinwidth, skinheight;
    long i,j;
    long frame, trig;
    V3D_f v1, v2, v3;
    q1mdl_t model;
    BITMAP *tmp;


    mdl_file = pack_fopen(filename, F_READ);
    if(mdl_file == NULL)	// return an error and exit
    {
        printf("Q1LoadFile(): ");
        perror(filename);
        exit(1);
    }

    // read the header
    pack_fread(&model.header, sizeof(q1mdlheader_t), mdl_file);

   // in case of extra skin.pcx
//   if(load_skin_pcx)
//   {
//	   model.header.skinwidth = skinpcxwidth;
//   	model.header.skinheight = skinpcxheight;
//   }

    skinwidth = model.header.skinwidth;
    skinheight = model.header.skinheight;

    // allocate memory for the skins struct
    model.oneskin = (q1skin_t *)malloc( model.header.numskins*sizeof(q1skin_t) );

    // it's not working for multiskin !!!
    for(i=0; i<model.header.numskins; i++)
    {
        model.oneskin[i].group = pack_igetl(mdl_file);
        if(model.oneskin[i].group != 0) { printf("Q1LoadFile(): Multiskins not supported!\n"); exit(1); }

        // allocate memory for every picture
        model.oneskin[i].skin = (u_char *)malloc( skinwidth*skinheight*sizeof(u_char) );
        // read the pic
	pack_fread(model.oneskin[i].skin, skinwidth*skinheight*sizeof(u_char), mdl_file);
    }

    // allocate memory for the vertices
    model.vertices = (q1stvert_t *)malloc( model.header.numverts*sizeof(q1stvert_t) );
    // read the vertices
    pack_fread(model.vertices, model.header.numverts*sizeof(q1stvert_t), mdl_file);

    // allocate memory for the triangles
    model.triangles = (q1itriangle_t *)malloc( model.header.numtris*sizeof(q1itriangle_t) );
    // read the triangles
    pack_fread(model.triangles, model.header.numtris*sizeof(q1itriangle_t), mdl_file);

    // allocate memory for the frames
    model.frames = (q1simpleframe_t *)malloc( model.header.numframes*sizeof(q1simpleframe_t) );

    // read the frames
    for(i=0; i<model.header.numframes; i++)
    {
        // allocate memory for the frame vertices
        model.frames[i].frame = (q1trivertx_t *)malloc( model.header.numverts*sizeof(q1trivertx_t) );

        model.frames[i].type = pack_igetl(mdl_file);
        if(model.frames[i].type != 0) { perror("Q1LoadFile(): Multiframes not supported!\n"); exit(1); }

        pack_fread(&model.frames[i].min, sizeof(q1trivertx_t), mdl_file);
        pack_fread(&model.frames[i].max, sizeof(q1trivertx_t), mdl_file);
        pack_fread(model.frames[i].name, 16*sizeof(char), mdl_file);
        pack_fread(model.frames[i].frame, model.header.numverts*sizeof(q1trivertx_t), mdl_file);
    }

    // close file
    pack_fclose(mdl_file);

    // allocate memory for the 3D points
    for(frame=0; frame<model.header.numframes; frame++)
        model.frames[frame].pos = (V3D_f *)malloc( model.header.numverts*sizeof(V3D_f) );

    model.rpos = (V3D_f *)malloc( model.header.numverts*sizeof(V3D_f) );

    // compute scaling factors for texture because Allegro needs
    // textures that are power of 2  =>  512x256
    model.xscale = (float)XTEX_CONV/(float)model.header.skinwidth;
    model.yscale = (float)YTEX_CONV/(float)model.header.skinheight;

    // store the points for every frame
    for(frame=0; frame<model.header.numframes; frame++)
        for(i=0; i<model.header.numverts; i++)
	{
	    // WARNING: Carefull with this axis swap (xx <=> -zz && yy <=> xx) below
            model.frames[frame].pos[i].x = (model.header.scale.x * model.frames[frame].frame[i].packedposition[0] + model.header.origin.x);
            model.frames[frame].pos[i].z = (model.header.scale.y * model.frames[frame].frame[i].packedposition[1] + model.header.origin.y);
            model.frames[frame].pos[i].y = -(model.header.scale.z * model.frames[frame].frame[i].packedposition[2] + model.header.origin.z);
            model.frames[frame].pos[i].u = model.vertices[i].s * model.xscale;
            model.frames[frame].pos[i].v = model.vertices[i].t * model.yscale;
            model.frames[frame].pos[i].c = 10;
            model.rpos[i].u = model.frames[0].pos[i].u;
            model.rpos[i].v = model.frames[0].pos[i].v;
            model.rpos[i].c = i;
        }


    // allocate memory for distance to midpoints list and for the order array
    model.dist2 = (float *)malloc( model.header.numtris*sizeof(float) );
    model.trigorder = (long *)malloc( model.header.numtris*sizeof(long) );
    model.drawtrig = (char *)malloc( model.header.numtris*sizeof(char) );

    // allocate memory for normal vectors
    for(frame=0; frame<model.header.numframes; frame++)
        model.frames[frame].normvect = (V3D_f *)malloc( model.header.numtris*sizeof(V3D_f) );

    // this shouldn't be necessary
    model.numvistris = model.header.numtris;

    for(frame=0; frame<model.header.numframes; frame++)
        for(trig=0; trig<model.header.numtris; trig++)
	{
            v1 = model.frames[frame].pos[ model.triangles[trig].vertices[0] ];
            v2 = model.frames[frame].pos[ model.triangles[trig].vertices[1] ];
//            v3 = model.frames[frame].pos[ model.triangles[trig].vertices[2] ];
            model.frames[frame].normvect[trig].x = v1.y*v2.z - v1.z*v2.y;
            model.frames[frame].normvect[trig].y = v1.z*v2.x - v1.x*v2.z;
            model.frames[frame].normvect[trig].z = v1.x*v2.y - v1.y*v2.x;
        }

    model.tex = create_bitmap(XTEX_CONV, YTEX_CONV);

    tmp = create_bitmap(model.header.skinwidth, model.header.skinheight);

    // copy the skin to a BITMAP
    for(j=0; j<model.header.skinheight; j++)
        for(i=0; i<model.header.skinwidth; i++)
            putpixel(tmp, i, j, model.oneskin[0].skin[i+j*model.header.skinwidth]);

    // adjust to 512x256
    stretch_blit(tmp, model.tex, 0, 0, model.header.skinwidth, model.header.skinheight, 0, 0, XTEX_CONV, YTEX_CONV);
    destroy_bitmap( tmp );


    return( model );
}


// free memory ocuppied by a model
void Q1Unload(mdl_t *model)
{



}

// show info on the model
void Q1Debug(mdl_t *model)
{
    long i,j;
    long frame;

    printf("id = %X\n", model->q1mdl->header.id);
    printf("version = %i\n", model->q1mdl->header.version);
    printf("scale = %g %g %g\n", model->q1mdl->header.scale.x, model->q1mdl->header.scale.y, model->q1mdl->header.scale.z);
    printf("origin = %g %g %g\n", model->q1mdl->header.origin.x, model->q1mdl->header.origin.y, model->q1mdl->header.origin.z);
    printf("radius = %g\n", model->q1mdl->header.radius);
    printf("offsets = %g %g %g\n", model->q1mdl->header.offsets.x, model->q1mdl->header.offsets.y, model->q1mdl->header.offsets.z);
    printf("numskins = %i\n", model->q1mdl->header.numskins);
    printf("skinwidth = %i\n", model->q1mdl->header.skinwidth);
    printf("skinheight = %i\n", model->q1mdl->header.skinheight);
    printf("numverts = %i\n", model->q1mdl->header.numverts);
    printf("numtris = %i\n", model->q1mdl->header.numtris);
    printf("numframes = %i\n", model->q1mdl->header.numframes);
    printf("synctype = %i\n", model->q1mdl->header.synctype);
    printf("flags = %i\n", model->q1mdl->header.flags);
    printf("size = %g\n", model->q1mdl->header.size);

    printf("\n");
    for(i=0; i<model->q1mdl->header.numskins; i++)
    {
        printf("Picture %i\n", i);
        printf("group = %i\n", model->q1mdl->oneskin[i].group);
    }

    printf("\n");
    for(i=0; i<model->q1mdl->header.numverts; i++)
    {
        printf("Vertice %i\n", i);
        printf("onseam = %i\n", model->q1mdl->vertices[i].onseam);
        printf("s = %i\n", model->q1mdl->vertices[i].s);
        printf("t = %i\n", model->q1mdl->vertices[i].t);
    }

    printf("\n");
    for(i=0; i<model->q1mdl->header.numtris; i++)
    {
        printf("Triangle %i\n", i);
        printf("facesfront = %i\n", model->q1mdl->triangles[i].facesfront);
        printf("vertices = %i %i %i\n", model->q1mdl->triangles[i].vertices[0], model->q1mdl->triangles[i].vertices[1], model->q1mdl->triangles[i].vertices[2]);
    }

    printf("\n");
    for(i=0; i<model->q1mdl->header.numframes; i++)
    {
        printf("Frame %i\n", i);
        printf("type = %i\n", model->q1mdl->frames[i].type);
        printf("min = %i %i %i %i\n", model->q1mdl->frames[i].min.packedposition[0],
            model->q1mdl->frames[i].min.packedposition[1], model->q1mdl->frames[i].min.packedposition[2],
            model->q1mdl->frames[i].min.lightnormalindex);
        printf("max = %i %i %i %i\n", model->q1mdl->frames[i].max.packedposition[0],
            model->q1mdl->frames[i].max.packedposition[1], model->q1mdl->frames[i].max.packedposition[2],
            model->q1mdl->frames[i].max.lightnormalindex);
        printf("name = %s\n", model->q1mdl->frames[i].name);

        for(j=0; j<model->q1mdl->header.numverts; j++)
     	{
            printf("Vertice %i\n", j);
            printf("pos = %i %i %i %i\n", model->q1mdl->frames[i].frame[j].packedposition[0],
                model->q1mdl->frames[i].frame[j].packedposition[1], model->q1mdl->frames[i].frame[j].packedposition[2],
                model->q1mdl->frames[i].frame[j].lightnormalindex);
        }
    }

   printf("\n");
   for(frame=0; frame<model->q1mdl->header.numframes; frame++)
       for(i=0; i<model->q1mdl->header.numverts; i++)
       {
           printf("V3D point %i\n", i);
           printf("x = %g\n", model->q1mdl->frames[frame].pos[i].x);
           printf("y = %g\n", model->q1mdl->frames[frame].pos[i].y);
           printf("z = %g\n", model->q1mdl->frames[frame].pos[i].z);
       }

}

// draw only the points
void Q1DrawPoints(BITMAP *dbuf, mdl_t *model)
{
   long i;

   // draw dots
   for(i=0; i<model->q1mdl->header.numverts; i++)
		putpixel(dbuf, (int)model->q1mdl->rpos[i].x, (int)model->q1mdl->rpos[i].y, 10);
}

// draw wireframe triangles
void Q1DrawWireframe(BITMAP *dbuf, mdl_t *model)
{
    V3D_f v1, v2, v3;
    long i, trig;

    // draw every triangle
    for(i=model->q1mdl->numvistris-1; i>=0; i--)
    {
        trig = model->q1mdl->trigorder[i];
        v1 = model->q1mdl->rpos[model->q1mdl->triangles[trig].vertices[0]];
        v2 = model->q1mdl->rpos[model->q1mdl->triangles[trig].vertices[1]];
        v3 = model->q1mdl->rpos[model->q1mdl->triangles[trig].vertices[2]];

//        if(model->q1mdl->drawtrig[i] && v1.z > Z_CLIP && v2.z > Z_CLIP && v3.z > Z_CLIP)
        {
            line(dbuf, v1.x, v1.y, v2.x, v2.y, 10);
            line(dbuf, v1.x, v1.y, v3.x, v3.y, 10);
            line(dbuf, v3.x, v3.y, v2.x, v2.y, 10);
        }
    }
}


// Draw flat shaded triangles
void Q1DrawFlatShade(BITMAP *dbuf, mdl_t *model)
{
   long i, j;
   long trig;
   V3D_f v1, v2, v3;        // tmp variables

   // draw every triangle
   for(i=model->q1mdl->numvistris-1; i>=0; i--)
   {
      trig = model->q1mdl->trigorder[i];
      v1 = model->q1mdl->rpos[model->q1mdl->triangles[trig].vertices[0]];
      v2 = model->q1mdl->rpos[model->q1mdl->triangles[trig].vertices[1]];
      v3 = model->q1mdl->rpos[model->q1mdl->triangles[trig].vertices[2]];

//      if(model->q1mdl->drawtrig[i] && v1.z > Z_CLIP && v2.z > Z_CLIP && v3.z > Z_CLIP)
      {
/*
			if( (model->q1mdl->vertices[model->q1mdl->triangles[trig].vertices[0]].onseam)
					&& (!model->q1mdl->triangles[trig].facesfront)) v1.u += model->q1mdl->header.skinwidth * model->q1mdl->xscale / 2;
			if( (model->q1mdl->vertices[model->q1mdl->triangles[trig].vertices[1]].onseam)
					&& (!model->q1mdl->triangles[trig].facesfront)) v2.u += model->q1mdl->header.skinwidth * model->q1mdl->xscale / 2;
			if( (model->q1mdl->vertices[model->q1mdl->triangles[trig].vertices[2]].onseam)
					&& (!model->q1mdl->triangles[trig].facesfront)) v3.u += model->q1mdl->header.skinwidth * model->q1mdl->xscale / 2;
*/
   	   triangle3d_f(dbuf, POLYTYPE_FLAT, model->q1mdl->tex, &v1, &v2, &v3);
      }
   }

}

// Draw textured triangles
void Q1DrawTexture(BITMAP *dbuf, mdl_t *model)
{
   long i, j;
   long trig;
   V3D_f v1, v2, v3;        // tmp variables

   // draw every triangle
   for(i=model->q1mdl->numvistris-1; i>=0; i--)
   {
      trig = model->q1mdl->trigorder[i];
      v1 = model->q1mdl->rpos[model->q1mdl->triangles[trig].vertices[0]];
      v2 = model->q1mdl->rpos[model->q1mdl->triangles[trig].vertices[1]];
      v3 = model->q1mdl->rpos[model->q1mdl->triangles[trig].vertices[2]];

//      if(model->q1mdl->drawtrig[i] && v1.z > Z_CLIP && v2.z > Z_CLIP && v3.z > Z_CLIP)
      {

			if( (model->q1mdl->vertices[model->q1mdl->triangles[trig].vertices[0]].onseam)
					&& (!model->q1mdl->triangles[trig].facesfront)) v1.u += model->q1mdl->header.skinwidth * model->q1mdl->xscale / 2;
			if( (model->q1mdl->vertices[model->q1mdl->triangles[trig].vertices[1]].onseam)
					&& (!model->q1mdl->triangles[trig].facesfront)) v2.u += model->q1mdl->header.skinwidth * model->q1mdl->xscale / 2;
			if( (model->q1mdl->vertices[model->q1mdl->triangles[trig].vertices[2]].onseam)
					&& (!model->q1mdl->triangles[trig].facesfront)) v3.u += model->q1mdl->header.skinwidth * model->q1mdl->xscale / 2;

   	   triangle3d_f(dbuf, POLYTYPE_ATEX, model->q1mdl->tex, &v1, &v2, &v3);
      }
   }

}


// rotate and move the points and sort the traingles
void Q1UpdateView(camera_t cam, mdl_t *model)
{
    float xfront, yfront, zfront;
    float xup, yup, zup;
    float dot;
    long i, j;
//    long frameidx = 0;       // current frame
    MATRIX_f roller, camview;
    V3D_f v1, v2, v3, v4, vc, v4r, vn, vnr;   // tmp variables

    // calculate the in-front vector
    xfront = sin(cam.head) * cos(cam.pitch);
    yfront = sin(cam.pitch);
    zfront = cos(cam.head) * cos(cam.pitch);

    // rotate the up vector around the in-front vector by the roll angle
//    get_vector_rotation_matrix_f(&roller, xfront, yfront, zfront, cam.roll*128.0/M_PI);
//    apply_matrix_f(&roller, 0, -1, 0, &xup, &yup, &zup);
    get_transformation_matrix_f(&camview, 1., cam.head, cam.roll, cam.pitch, cam.x, cam.y, cam.z);
/*
    // build the camera matrix
    get_camera_matrix_f(&camview,
                        cam.x, cam.y, cam.z,     // camera position
                        xfront, yfront, zfront,  // in-front vector
                        xup, yup, zup,           // up vector
                        48,                      // field of view
                        1.);                     // aspect ratio
*/
    // rotate the points and
    for(i=0; i<model->q1mdl->header.numverts; i++)
        apply_matrix_f(&camview, model->q1mdl->frames[model->frameidx].pos[i].x, model->q1mdl->frames[model->frameidx].pos[i].y, model->q1mdl->frames[model->frameidx].pos[i].z,
                       &model->q1mdl->rpos[i].x, &model->q1mdl->rpos[i].y, &model->q1mdl->rpos[i].z);

    // this might do the work for now!
    for(i=0; i<model->q1mdl->header.numtris; i++)
    {
        v1 = model->q1mdl->rpos[model->q1mdl->triangles[i].vertices[0]];
        v2 = model->q1mdl->rpos[model->q1mdl->triangles[i].vertices[2]];
        v3 = model->q1mdl->rpos[model->q1mdl->triangles[i].vertices[1]];

        // average of the triangle
        v4.x = (v3.x + v2.x + v1.x)/3.;
        v4.y = (v3.y + v2.y + v1.y)/3.;
        v4.z = (v3.z + v2.z + v1.z)/3.;

        // calculate distance from view to average triangle point
//         model->q1mdl->dist2[i] = (v4.x-xpos)*(v4.x-xpos) + (v4.y-ypos)*(v4.y-ypos) + (v4.z-zpos)*(v4.z-zpos);
        model->q1mdl->dist2[i] = (v4.x)*(v4.x) + (v4.y)*(v4.y) + (v4.z)*(v4.z);
        model->q1mdl->trigorder[i] = i;

        cross_product_f(v2.x-v1.x, v2.y-v1.y, v2.z-v1.z, v3.x-v1.x, v3.y-v1.y, v3.z-v1.z, &vc.x, &vc.y, &vc.z);
        dot = dot_product_f(vc.x, vc.y, vc.z, v4.x, v4.y, v4.z);

//        if(dot > 0) model->q1mdl->drawtrig[i] = 1;
//        else model->q1mdl->drawtrig[i] = 0;
        model->q1mdl->drawtrig[i] = 1;
    }

    // calculate perspective
    for(i=0; i<model->q1mdl->header.numverts; i++)
    persp_project_f(model->q1mdl->rpos[i].x, model->q1mdl->rpos[i].y, model->q1mdl->rpos[i].z, &model->q1mdl->rpos[i].x, &model->q1mdl->rpos[i].y);

    model->q1mdl->numvistris = model->q1mdl->header.numtris;

    // sort the triangles
    Mysort(model->q1mdl->dist2, model->q1mdl->trigorder, 0, model->q1mdl->numvistris-1);

}





