/*
 *    Chipset-specific SVGA video drivers for the Allegro library.
 *
 *    See vdrv.txt for copyright information.
 *
 *    Helper functions for use by the video drivers.
 */


#include "vintern.h"



/* in case you want to report version numbers */
char vdrv_id[] = "Allegro SVGA drivers version " VDRV_VERSION_STR ", " VDRV_DATE_STR;



/* linear framebuffer address */
static unsigned long lb_linear = 0; 
static int lb_segment = 0;



/* _gfx_mode_set_helper:
 *  Helper function for native SVGA drivers, does most of the work of setting
 *  up an SVGA mode, using the detect() and get_mode_number() functions in
 *  the driver structure.
 */
BITMAP *_gfx_mode_set_helper(int w, int h, int v_w, int v_h, int color_depth, GFX_DRIVER *driver, int (*detect)(), _GFX_MODE_INFO *mode_list, void (*set_width)(int w, int bpp), int align)
{
   BITMAP *b = NULL;
   int mode;
   int height;
   int width;
   int bpp = BYTES_PER_PIXEL(color_depth);

   __dpmi_regs r;

   if (!(*detect)()) {
      sprintf(allegro_error, get_config_text("%s not found"), driver->name);
      return NULL;
   }

   mode = 0;
   while ((mode_list[mode].w != w) || (mode_list[mode].h != h) || (mode_list[mode].bpp != color_depth)) {
      if (mode_list[mode].w == 0) {
	 strcpy(allegro_error, get_config_text("Resolution not supported"));
	 return NULL;
      }
      mode++;
   }

   width = MAX(w, v_w) * bpp;
   _sort_out_virtual_width(&width, driver);

   if (align <= 0)
      align = 16;

   width += align - 1;
   width -= width % align;

   height = driver->vid_mem / width;

   if ((width > 4096) || (h > height) || (v_w > width/bpp) || (v_h > height)) {
      strcpy(allegro_error, get_config_text("Virtual screen size too large"));
      return NULL;
   }

   if (driver->linear) {
      if (_create_physical_mapping(&lb_linear, &lb_segment,
				   driver->vid_phys_base, 
				   driver->vid_mem) == 0) {

	 b = _make_bitmap(width/bpp, height, 0, driver, color_depth, width);

	 if (b)
	    b->seg = lb_segment;
	 else
	    _remove_physical_mapping(&lb_linear, &lb_segment);
      }

      if (!b)
	 strcpy(allegro_error, get_config_text("Can't make linear screen bitmap"));
   } 
   else
      b = _make_bitmap(width/bpp, height, 0xA0000, driver, color_depth, width);

   if (!b)
      return NULL;

   if (mode_list[mode].setter) {
      if (mode_list[mode].setter(w, h, color_depth) != 0) {
	 strcpy(allegro_error, get_config_text("Error setting video mode"));
	 destroy_bitmap(b);
	 return NULL;
      }
   }
   else {
      if (mode_list[mode].bios_int) { 
	 r.x.ax = mode_list[mode].bios_int;
	 r.x.bx = mode_list[mode].bios_num;
      }
      else
	 r.x.ax = mode_list[mode].bios_num;

      __dpmi_int(0x10, &r);            /* set gfx mode */

      if ((mode_list[mode].bios_int == 0x4F02) && (r.h.ah)) {
	 strcpy(allegro_error, get_config_text("VBE mode not available"));
	 destroy_bitmap(b);
	 return NULL;
      }
   }

   if (set_width)
      set_width(width, color_depth);
   else
      _set_vga_virtual_width(w, width);

   driver->w = b->cr = w;
   driver->h = b->cb = h;

   return b;
}



/* _vdrv_exit:
 *  Shutdown routine.
 */
void _vdrv_exit(BITMAP *b)
{
      if (lb_linear)
	 _remove_physical_mapping(&lb_linear, &lb_segment);
}



/* _test_vga_register:
 *  Tests whether specific bits of a VGA register can be changed.
 */
int _test_vga_register(int port, int index, int mask)
{
   int old, nw1, nw2;

   old = _read_vga_register(port, index);
   _write_vga_register(port, index, old & (~mask));
   nw1 = _read_vga_register(port, index) & mask;
   _write_vga_register(port, index, old | mask);
   nw2 = _read_vga_register(port, index) & mask;
   _write_vga_register(port, index, old);

   return ((nw1==0) && (nw2==mask));
}



/* _test_register:
 *  Tests whether specific bits of a register can be changed.
 */
int _test_register(int port, int mask)
{
   int old, nw1, nw2;

   old = inportb(port);
   outportb(port, old & (~mask));
   nw1 = inportb(port) & mask;
   outportb(port, old | mask);
   nw2 = inportb(port) & mask;
   outportb(port, old);

   return ((nw1==0) && (nw2==mask));
}

