/*
	This file is part of CrocoLib.

	CrocoLib is free software: you can redistribute it and/or modify
	it under the terms of the GNU Lesser General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	CrocoLib is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU Lesser General Public License for more details.

	You should have received a copy of the GNU Lesser General Public License
	along with Ceremony.  If not, see <http://www.gnu.org/licenses/>.
*/

// ----------------------------------------------------------------------------
#include "crLIB.h"
#include <lib3d.h>

#include <string.h> // memset

#ifdef COMPILEPART1
#include "Loader_Link1.h"
#endif // COMPILEPART1
#ifdef COMPILEPART2
#include "Loader_Link2.h"
#endif // COMPILEPART2
#ifdef COMPILEPART3
#include "Loader_Link3.h"
#endif // COMPILEPART3
#ifdef COMPILEPART4
#include "Loader_Link4.h"
#endif // COMPILEPART4
#ifdef COMPILEPART5
#include "Loader_Link5.h"
#endif // COMPILEPART5
#ifdef COMPILELASTPART
#include "Loader_LinkLastPart.h"
#endif // COMPILELASTPART

#include "Nuages_Link.h"
#include "Solar_Link.h"
#include "Tunnel_Link.h"
#include "PreBozo_Link.h"
#include "Bozo_Link.h"
#include "RotoZoom_Link.h"
#include "Interference_Link.h"
#include "Tunnel2_Link.h"
#include "Nuages2_Link.h"

#include "Demo.h"

// ----------------------------------------------------------------------------
unsigned char *s_dataPtr = (unsigned char *) 0x1500;
#define GETDATAPTR( offset ) ( (unsigned char *) ( s_dataPtr + offset ) )

// ----------------------------------------------------------------------------
#define MOVE_PTR ( 0xA000 )
#define MOVE_SIZE ( 0x0300 )

// ----------------------------------------------------------------------------
static unsigned char *partCount = 0x808;
static unsigned char *bank = 0x809;
static unsigned int *destPtr = 0x810;
struct SPart
{
	unsigned char bank;
	unsigned int ptrInBank;
	unsigned int size1;
	unsigned int size2;
	unsigned int z80size;
	unsigned int dataSize;
};
static struct SPart *parts = 0x816;

// 0x814 secondtime nuages
// 0x812 secondtime solar

// ----------------------------------------------------------------------------
#ifdef COMPILELASTPART
static void unpackPartFromBank( unsigned char part )
{
	switch ( parts[ part ].bank )
	{
		case 0xC4:	crBANKSetC4( ); break;
		case 0xC5:	crBANKSetC5( ); break;
		case 0xC6:	crBANKSetC6( ); break;
		case 0xC7:	crBANKSetC7( ); break;
	}

	if ( parts[ part ].size2 != 0 )
	{
		memcpy( 0x8000, parts[ part ].ptrInBank, parts[ part ].size1 );

		switch ( parts[ part ].bank + 1 )
		{
		case 0xC4:	crBANKSetC4( ); break;
		case 0xC5:	crBANKSetC5( ); break;
		case 0xC6:	crBANKSetC6( ); break;
		case 0xC7:	crBANKSetC7( ); break;
		}

		memcpy( 0x8000 + parts[ part ].size1, 0x4000, parts[ part ].size2 );
	}
	else
	{
		memcpy( 0x8000, parts[ part ].ptrInBank, parts[ part ].size1 );
	}

	crBANKSetC0( );

	crPACKERUnpack( 0x1500, 0x8000 );
	memcpy( 0x3600, 0x8000 + parts[ part ].z80size, parts[ part ].dataSize );
}
#endif // COMPILELASTPART

// ----------------------------------------------------------------------------
static void movePartIntoBank( unsigned char *partSrc, unsigned int partSize, unsigned int z80size, unsigned int dataSize )
{
	unsigned int blockSize;
	unsigned int size1;
	unsigned int size2;

	parts[ *partCount ].bank = *bank;
	parts[ *partCount ].z80size = z80size;
	parts[ *partCount ].dataSize = dataSize;
	parts[ *partCount ].ptrInBank = *destPtr;

	if ( ( *destPtr + partSize ) >= 0x8000 )
	{
		size1 = 0x8000 - *destPtr;
		size2 = partSize - size1;

		parts[ *partCount ].size1 = size1;
		parts[ *partCount ].size2 = size2;
	}
	else
	{
		parts[ *partCount ].size1 = partSize;
		parts[ *partCount ].size2 = 0;
	}

	( *partCount )++;

	do
	{
		blockSize = partSize;
		if ( blockSize > MOVE_SIZE )
		{
			blockSize = MOVE_SIZE;
		}

		memcpy( MOVE_PTR, partSrc, blockSize );

		switch ( *bank )
		{
		case 0xC4:	crBANKSetC4( ); break;
		case 0xC5:	crBANKSetC5( ); break;
		case 0xC6:	crBANKSetC6( ); break;
		case 0xC7:	crBANKSetC7( ); break;
		}

		if ( ( *destPtr + blockSize ) >= 0x8000 )
		{
			size1 = 0x8000 - *destPtr;
			size2 = blockSize - size1;

			memcpy( *destPtr, MOVE_PTR, size1 );
			*destPtr = 0x4000;

			( *bank )++;
			switch ( *bank )
			{
			case 0xC4:	crBANKSetC4( ); break;
			case 0xC5:	crBANKSetC5( ); break;
			case 0xC6:	crBANKSetC6( ); break;
			case 0xC7:	crBANKSetC7( ); break;
			}

			memcpy( *destPtr, MOVE_PTR + size1, size2 );
			( *destPtr ) += size2;
		}
		else
		{
			memcpy( *destPtr, MOVE_PTR, blockSize );
			( *destPtr ) += blockSize;
		}

		crBANKSetC0( );

		partSrc += blockSize;
		partSize -= blockSize;
	}
	while ( partSize != 0 );
}

// ----------------------------------------------------------------------------
static void movePartsIntoBanks( )
{
#ifdef COMPILEPART1
	movePartIntoBank( GETDATAPTR( DATA_NUAGES_PART ), DATA_NUAGES_PART_SIZE, NUAGES_Z80_SIZE, NUAGES_DATA_SIZE );
	movePartIntoBank( GETDATAPTR( DATA_SOLAR_PART ), DATA_SOLAR_PART_SIZE, SOLAR_Z80_SIZE, SOLAR_DATA_SIZE );
#endif // COMPILEPART1

#ifdef COMPILEPART2
	movePartIntoBank( GETDATAPTR( DATA_TUNNEL_PART ), DATA_TUNNEL_PART_SIZE, TUNNEL_Z80_SIZE, TUNNEL_DATA_SIZE );
	movePartIntoBank( GETDATAPTR( DATA_INTER_PART ), DATA_INTER_PART_SIZE, INTER_Z80_SIZE, INTER_DATA_SIZE );
#endif // COMPILEPART2

#ifdef COMPILEPART3
	movePartIntoBank( GETDATAPTR( DATA_ROTOZOOM_PART ), DATA_ROTOZOOM_PART_SIZE, ROTOZOOM_Z80_SIZE, ROTOZOOM_DATA_SIZE );
	movePartIntoBank( GETDATAPTR( DATA_TUNNEL2_PART ), DATA_TUNNEL2_PART_SIZE, TUNNEL2_Z80_SIZE, TUNNEL2_DATA_SIZE );
#endif // COMPILEPART3

#ifdef COMPILEPART4
	movePartIntoBank( GETDATAPTR( DATA_NUAGES2_PART ), DATA_NUAGES2_PART_SIZE, NUAGES2_Z80_SIZE, NUAGES2_DATA_SIZE );
	movePartIntoBank( GETDATAPTR( DATA_PREBOZO_PART ), DATA_PREBOZO_PART_SIZE, PREBOZO_Z80_SIZE, PREBOZO_DATA_SIZE );
#endif // COMPILEPART4

#ifdef COMPILEPART5
	movePartIntoBank( GETDATAPTR( DATA_BOZO_PART ), DATA_BOZO_PART_SIZE, BOZO_Z80_SIZE, BOZO_DATA_SIZE );
#endif // COMPILEPART5

#ifdef COMPILELASTPART
#endif // COMPILELASTPART
}

// ----------------------------------------------------------------------------
#ifdef COMPILELASTPART
static void executePart( )
{
#asm
	ld ( jump + 1 ), sp;
	ld hl, jump;
	ld ( $1500 - 2 ), hl;
	jp $1500;
.jump
	ld sp, 0;
#endasm	
}
#endif // COMPILELASTPART

// ----------------------------------------------------------------------------
#ifdef COMPILELASTPART
static void setupAYPlayer( )
{
	crPACKERUnpack( 0x0100, GETDATAPTR( DATA_DMALIB ) );
	
	crBANKSetC7( );
	
#asm
	ld hl, $6c00;
	call $6300;

	ld hl, 312-15;
	ld de, $220;
	ld bc, $6303;
	ld a, 37;
	call $100;

	ld b, $F5;
.syncVBL
	in a, (c);
	rra;
	jr nc, syncVBL;
#endasm

	crBANKSetC0( );
}
#endif // COMPILELASTPART

// ----------------------------------------------------------------------------
void main( )
{	
	unsigned char iPart;
	
	asm( "ld sp, $8E0" );


#asm
	call $103;
#endasm

#ifdef COMPILEPART1
	*partCount = 0;
	*bank = 0xC4;
	*destPtr = (unsigned int) 0x4000;
#endif // COMPILEPART1

	crBANKGetBank( );
	
	movePartsIntoBanks( );

#ifndef COMPILELASTPART
#asm
	ld hl, ( $1500 - 2 );
	jp ( hl );
#endasm
#endif // COMPILELASTPART

#ifdef COMPILELASTPART
	// first time for solar
#asm
	ld hl, $812;
	ld ( hl ), 255;
	ld hl, $814;
	ld ( hl ), 255;
#endasm

	crLIBInit( 0x4000 );

	setupAYPlayer( );

	{
		unsigned char iVBL;
		unsigned int s_palette[ 17 ];

		s_palette[ 1 ] = 0x666;
		s_palette[ 2 ] = 0x666;
		s_palette[ 3 ] = 0xCCC;

		for ( iVBL = 0; iVBL <= 5; iVBL++ )
		{
			s_palette[ 0 ] = 0;
			s_palette[ 1 ] -= 0x111;
			s_palette[ 2 ] -= 0x111;
			s_palette[ 3 ] -= 0x222;

			crSYNCWaitVBL( );
			crASICSetPalette( s_palette );
		}
	}

	memcpy( 0x6500, GETDATAPTR( DATA_ALONE ), DATA_ALONE_SIZE );		
	
	for ( iPart = 0; iPart < *partCount; iPart++ )
	{
		unpackPartFromBank( iPart );

		if ( iPart == 0 )
		{
			crBANKSetC0();
			crPACKERUnpack( 0x8000, 0x6500 );
		}
		
		executePart( );		
	}

	unpackPartFromBank( 1 );
	executePart( );

	unpackPartFromBank( 0 );
	executePart( );

	unpackPartFromBank( 2 );
	executePart( );
#endif // COMPILELASTPART
}
