/* Copyright Stefan Hlln
 * May not be used without
 * the authors specific
 * authorization
 * el98shn@ing.umu.se       */

#include <stdio.h>
#include <math.h>

#include "isVoice.h"
#include "isDistortion.h"
#include "isFilter.h"
#include "isEnvelope.h"
#include "isLFO.h"
#include "isController.h"

isVoice::isVoice(isSamplePool &samplePool) : samplePool(samplePool)
{
//	this->samplePool=samplePool;
	OutputDebugString("isVoice\n");

	bufferLength=1;

	for(int polyNr=0;polyNr<MAXPOLY;polyNr++)
	{
		active[polyNr]=false; // make this voice TOTALLY inactive :)
		noteOn[polyNr]=false;
		pitch[polyNr]=-1;
		cVol[polyNr]=0.0f;
	}

	nrOfOscillators=nrOfModulators=nrOfInserts=0;
	voicePoly=8;

	noteRangeStart=0;
	noteRangeEnd=255;
	noteIndex=0;
	for(int bah=0;bah<256;bah++)
		modulator[bah]=NULL;

	// ok, this is how you add a modulator
	// this first one is used as the volume
	// envelope, therefor removing it is
	// strictly forbinnen. but using it as
	// modulator for other things is ok 
	AddModulator(is_ENVELOPE);
	//AddModulator(is_CONTROLLER);
	//AddModulator(is_LFO);
	
	// the first oscillator has to be added without AddOscillator
	nrOfOscillators=1;
	oscillator[0]=new isOscillator(samplePool,modulator);
	//AddOscillator();

	nrOfInserts=0;
	AddInsert(is_FILTER);

	modNr[0]=2;
	modNr[1]=3;
	modAmount[0]=0.0f;
	modAmount[1]=0.0f;
}

isVoice::~isVoice()
{
}
void isVoice::Load(isFile *f)
{
	unsigned int nr;
	
	noteRangeStart=f->ReadInt(1);
	noteRangeEnd=f->ReadInt(1);

	// load modulators
	nr=f->ReadInt(1);
	// delete all modulators but volume envelope
	while(nrOfModulators>1) 
	{
		DelModulator();
	}
	// load the volume envelope
	LoadType(f);
	modulator[nrOfModulators-1]->Load(f);

	// then load the new ones from f
	while(nrOfModulators<nr) 
	{
		isTYPE loadedType=LoadType(f);
		AddModulator(loadedType);
		modulator[nrOfModulators-1]->Load(f);
	}

	// load oscillators
	nr=f->ReadInt(1);
	// remove all current oscillators
	while(nrOfOscillators>nr) 
	{
		DelOscillator();
	}
	// then load the new ones from f
	while(nrOfOscillators<nr) 
	{
		AddOscillator();
	}
	for(int o=0;o<nrOfOscillators;o++)
		oscillator[o]->Load(f);
		
	// load inserts
	nr=f->ReadInt(1);
	// remove all current modulators
	while(nrOfInserts>0) 
	{
		DelInsert();
	}
	// then load the new ones from f
	while(nrOfInserts<nr) 
	{
		isTYPE loadedType=LoadType(f);
		AddInsert(loadedType);
		insert[nrOfInserts-1]->Load(f);
	}
	
// load modulation
	modNr[0]=f->ReadInt(1);
	modNr[1]=f->ReadInt(1);
	modAmount[0]=f->ReadFloat();
	modAmount[1]=f->ReadFloat();

}

void isVoice::Save(isFile *f)
{
	f->WriteInt(noteRangeStart,1);
	f->WriteInt(noteRangeEnd,1);
	// save modulators
	f->WriteInt(nrOfModulators,1);
	for(int m=0;m<nrOfModulators;m++)
	{
		modulator[m]->Save(f);
	}
	// save oscillators
	f->WriteInt(nrOfOscillators,1);
	for(int o=0;o<nrOfOscillators;o++)
	{
		oscillator[o]->Save(f);
	}
	// save inserts
	f->WriteInt(nrOfInserts,1);
	for(int i=0;i<nrOfInserts;i++)
	{
		insert[i]->Save(f);
	}
	// save modulation
	f->WriteInt(modNr[0],1);
	f->WriteInt(modNr[1],1);
	f->WriteFloat(modAmount[0]);
	f->WriteFloat(modAmount[1]);
}

void isVoice::Play(float *buffer)
{
	for(int polyNr=0;polyNr<voicePoly;polyNr++)
	{
		if(!active[polyNr]) continue;
		// first render the modulators so the oscillators can use them
		for(int mNr=0;mNr<nrOfModulators;mNr++)
		{
			if(modulator[mNr]!=NULL)
				modulator[mNr]->Render(polyNr);
		}
/*		// is volume envelope done and note is off?
		if(!noteOn[polyNr] && modulator[0]->StartValue(polyNr)==0.0f)
		{
			active[polyNr]=false;
			continue;
		}*/

		// then render the oscillators
		// we ALWAYS must have atleast one oscillator, so we render that first.
		oscillator[0]->Play(bufferLength,tempBuff[0],polyNr);
		// then we render the rest
		for(int o=1;o<nrOfOscillators;o++)
		{
			oscillator[o]->Play(bufferLength,tempBuff[1],polyNr);
			float fromMix,toMix;
			fromMix=oscillator[o]->mix;
			toMix=1.0f - fromMix;
			switch(oscillator[o]->mode)
			{
				int i;
			case ADD:
				for(i=0;i<bufferLength;i++)
				{
					tempBuff[0][i]=tempBuff[0][i]*toMix+tempBuff[1][i]*fromMix;
				}
				break;
			case MUL:
				for(i=0;i<bufferLength;i++)
				{
					tempBuff[0][i]*=toMix+tempBuff[1][i]*fromMix;
				}
				break;
			}
		}
		for(int ins=0;ins<nrOfInserts;ins++)
			insert[ins]->Play(tempBuff[0],polyNr);
		// volume envelope
		float sVol,eVol,dVol;
		sVol=modulator[0]->StartValue(polyNr);
		eVol=modulator[0]->EndValue(polyNr);
		for(int v=0;v<2;v++)
		{
			if(modulator[modNr[v]]!=NULL)
			{
				float s,e;
				s=modulator[modNr[v]]->StartValue(polyNr);
				e=modulator[modNr[v]]->EndValue(polyNr);
				sVol=sVol*s*modAmount[v]+sVol*(1-modAmount[v]);
				eVol=eVol*e*modAmount[v]+eVol*(1-modAmount[v]);
			}
		}
		sVol*=60; // -60dB..0dB
		eVol*=60;
		sVol-=48;
		eVol-=48;

		sVol=powf(10,sVol*0.05f); // 10^(dB/20)
		eVol=powf(10,eVol*0.05f); // 10^(dB/20)
		dVol=(eVol-sVol)/bufferLength;

		for(int i=0;i<bufferLength;i++)
		{
			cVol[polyNr] = cVol[polyNr]*0.99f + sVol*0.01f;
			buffer[i]+=tempBuff[0][i]*cVol[polyNr];
			sVol+=dVol;
			if(!noteOn[polyNr] && cVol[polyNr]<=0.005f)
			{
				active[polyNr]=false;
				break;
			}
		}

/*		// Optimering(?) av Dan. 2002-01-09.
		float *tb = tempBuff[0];
		for(int i=0;i<bufferLength;i++)
		{
			float tmp = sVol * 0.05f;
			cVol[polyNr] *= 0.95f;		
			tmp += cVol[polyNr];
			tmp *= *tb++;
			buffer[i] += tmp;
			sVol += dVol;
		}
*/

	}
}

void isVoice::Event(isEVENT event)
{
	// within note-range?
	if(event.type==NOTE_ON || event.type==NOTE_OFF)
	{
		if(event.param1<noteRangeStart) return;
		if(event.param1>noteRangeEnd)	return;
	}
	if(event.type==NOTE_ON)
	{
		// check for free poly, otherwise overwrite
		for(int polyNr=0;polyNr<voicePoly;polyNr++)
		{
			if(!active[noteIndex])
			{
				noteIndex=(noteIndex-1)%voicePoly;
				break;				
			}
			noteIndex=(noteIndex+1)%voicePoly;
		}
		noteIndex=(noteIndex+1)%voicePoly;
		active[noteIndex]=true;
		noteOn[noteIndex]=true;
		pitch[noteIndex]=event.param1;

		for(int modNr=0;modNr<nrOfModulators;modNr++)
			modulator[modNr]->Event(event,noteIndex);

		for(int o=0;o<nrOfOscillators;o++)
			oscillator[o]->Event(event,noteIndex);

		for(int ins=0;ins<nrOfInserts;ins++)
			insert[ins]->Event(event,noteIndex);
	}

	else
	{
		for(int polyNr=0;polyNr<voicePoly;polyNr++)
		{
			if(!active[polyNr])
				continue;
			if(event.type==NOTE_OFF)
			{
				if(event.param1==pitch[polyNr])
				{
					noteOn[polyNr]=false;
					for(int modNr=0;modNr<nrOfModulators;modNr++)
						modulator[modNr]->Event(event,polyNr);

					for(int o=0;o<nrOfOscillators;o++)
						oscillator[o]->Event(event,polyNr);

					for(int ins=0;ins<nrOfInserts;ins++)
						insert[ins]->Event(event,polyNr);
				}
				continue;
			}
			else
			{
				for(int modNr=0;modNr<nrOfModulators;modNr++)
					modulator[modNr]->Event(event,polyNr);

				for(int o=0;o<nrOfOscillators;o++)
					oscillator[o]->Event(event,polyNr);

				for(int ins=0;ins<nrOfInserts;ins++)
					insert[ins]->Event(event,polyNr);
			}

		}
	}
}

void isVoice::SetBufferSize(int length)
{
	bufferLength=length;
	tempBuff[0]=isBase::tempBuffer[1];
	tempBuff[1]=isBase::tempBuffer[2];
	for(int o=0;o<nrOfOscillators;o++)
		oscillator[o]->SetBufferSize(length);
	for(int i=0;i<nrOfInserts;i++)
		insert[i]->SetBufferSize(length);
}

isModulator *isVoice::GetModulator(int index)
{
	if(index>=nrOfModulators)
		return NULL;
	else
		return modulator[index];
}

isOscillator *isVoice::GetOscillator(int index)
{
	if(index>=nrOfOscillators)
		return NULL;
	else
		return oscillator[index];
}

isInsert *isVoice::GetInsert(int index)
{
	if(index>=nrOfInserts)
		return NULL;
	else
		return insert[index];
}

// for adding stuff in the voice. remember to supply the correct "type"
void isVoice::AddModulator(isTYPE type)
{
	nrOfModulators++;
	switch(type) {
	case is_ENVELOPE:
		modulator[nrOfModulators-1]=new isEnvelope;
		break;
	case is_LFO:
		modulator[nrOfModulators-1]=new isLFO;
		break;
	case is_CONTROLLER:
		modulator[nrOfModulators-1]=new isController;
		break;
	default:
		break;
	}
}

void isVoice::AddOscillator()
{
	nrOfOscillators++;
	oscillator[nrOfOscillators-1]=new isOscillator(samplePool,modulator,oscillator[nrOfOscillators-2]);
}

void isVoice::AddInsert(isTYPE type)
{
	nrOfInserts++;
	switch(type) {
	case is_DISTORTION:
		insert[nrOfInserts-1]=new isDistortion;
		break;
	case is_FILTER:
		insert[nrOfInserts-1]=new isFilter(modulator);
		break;
	default:
		break;
	}
	SetBufferSize(bufferLength);
	
}

void isVoice::DelModulator()
{
	if(nrOfModulators==1) return; // we can not delete the volume envelope
	delete modulator[nrOfModulators-1];
	nrOfModulators--;
}

void isVoice::DelOscillator()
{
	if(nrOfOscillators==1) return; // we got to have tha motha oscillator left
	delete oscillator[nrOfOscillators-1];
	nrOfOscillators--;
}

void isVoice::DelInsert()
{
	if(nrOfInserts==0) return; // nothing to delete
	delete insert[nrOfInserts-1];
	nrOfInserts--;
}