/*
   Reverie
   delaycube.cpp
   Copyright (C)2003 Dan Potter
*/

#include "global.h"
#include "delaycube.h"

/*

The third demo effect in my series of Parallax demo effects. This one
might actually be of practical interest to people still. This effect
generates a cube out of a 3D grid of dots and then does something that
looks somewhat like motion blurring on it. The trick here is that we're
using alpha blended copies of the object instead of doing raster effects,
so it becomes feasible on the PVR.

This works by saving the last N modelview matrices and then restoring
each one for each "delay" cube. Each "delay" cube has a successively
lower alpha value as well.

This is very rendering stage intensive due to the alpha blending. It
might be made faster by switching entirely to opaque polys, but I'll
leave the working out of that as an exercise to the reader ;)

Another exercise for the reader... it is possible (and not too difficult)
to modify this code so that you have many geometric objects and morph
between them. The key is to have the same number of points in each object
and then have a "delta" array to tell you how much to move each point
each frame.

*/

DelayCube::DelayCube() {
	memset(&m_oldmats, 0, sizeof(m_oldmats));
	makeCube(m_points);
	m_theta = 0.0f;

	m_hsv = 0.0f;

	m_needTrans = false;
}

DelayCube::~DelayCube() {
}

static float scols[8][3] = {
	{ 1.0f, 0.0f, 0.0f },	// Red
	{ 1.0f, 0.0f, 1.0f },	// Magenta
	{ 0.0f, 0.0f, 1.0f },	// Blue
	{ 0.0f, 1.0f, 1.0f },	// Cyan
	{ 0.0f, 1.0f, 0.0f },	// Green
	{ 1.0f, 1.0f, 0.0f },	// Yellow 
	{ 1.0f, 0.0f, 0.0f },	// Red
	{ 1.0f, 0.0f, 1.0f }	// Magenta
};
void hsvSpectrum(float w, float & r, float & g, float & b) {
	// Figure out what our two endpoints are
	while (w >= 1.0f)
		w -= 1.0f;
	int w1 = (int)(w * 6);
	float i = 1.0f - ((w * 6) - w1);

	// Interpolate between
	r = scols[w1][0] * i + scols[w1+1][0] * (1.0f - i);
	g = scols[w1][1] * i + scols[w1+1][1] * (1.0f - i);
	b = scols[w1][2] * i + scols[w1+1][2] * (1.0f - i);
}

void DelayCube::switchToBox() {
	makeBox();
	m_needTrans = true;
}

void DelayCube::switchToPyramid() {
	makePyramid();
	m_needTrans = true;
}

void DelayCube::switchToCube() {
	makeCube(m_pointsTo);
	m_needTrans = true;
}

void DelayCube::switchToSphere() {
	makeSphere();
	m_needTrans = true;
}

// Generate a cube object for our usage. This could really be anything, I
// just like cubes. :)
void DelayCube::makeCube(Vector * t) {
	int x, y, z;

	for (x=0; x<6; x++) {
		for (y=0; y<6; y++) {
			for (z=0; z<6; z++) {
				t[x*6*6+y*6+z].x = x - 3;
				t[x*6*6+y*6+z].y = y - 3;
				t[x*6*6+y*6+z].z = z - 3;
			}
		}
	}
}

void DelayCube::makeBox() {
	int x, y, z, o;

	o = 0;
	for (x=0; x<18; x++) {
		m_pointsTo[o].x = (x*5.0f/18.0f) - 3.0f;
		m_pointsTo[o].y = -3.0f;
		m_pointsTo[o].z = -3.0f;
		o++;

		m_pointsTo[o].x = (x*5.0f/18.0f) - 3.0f;
		m_pointsTo[o].y = 2.0f;
		m_pointsTo[o].z = -3.0f;
		o++;

		m_pointsTo[o].x = (x*5.0f/18.0f) - 3.0f;
		m_pointsTo[o].y = -3.0f;
		m_pointsTo[o].z = 2.0f;
		o++;

		m_pointsTo[o].x = (x*5.0f/18.0f) - 3.0f;
		m_pointsTo[o].y = 2.0f;
		m_pointsTo[o].z = 2.0f;
		o++;
	}
	for (y=0; y<18; y++) {
		m_pointsTo[o].x = -3.0f;
		m_pointsTo[o].y = (y*5.0f/18.0f) - 3.0f;
		m_pointsTo[o].z = -3.0f;
		o++;

		m_pointsTo[o].x = 2.0f;
		m_pointsTo[o].y = (y*5.0f/18.0f) - 3.0f;
		m_pointsTo[o].z = -3.0f;
		o++;

		m_pointsTo[o].x = -3.0f;
		m_pointsTo[o].y = (y*5.0f/18.0f) - 3.0f;
		m_pointsTo[o].z = 2.0f;
		o++;

		m_pointsTo[o].x = 2.0f;
		m_pointsTo[o].y = (y*5.0f/18.0f) - 3.0f;
		m_pointsTo[o].z = 2.0f;
		o++;
	}
	for (z=0; z<18; z++) {
		m_pointsTo[o].x = -3.0f;
		m_pointsTo[o].y = -3.0f;
		m_pointsTo[o].z = (z*5.0f/18.0f) - 3.0f;
		o++;

		m_pointsTo[o].x = 2.0f;
		m_pointsTo[o].y = -3.0f;
		m_pointsTo[o].z = (z*5.0f/18.0f) - 3.0f;
		o++;

		m_pointsTo[o].x = -3.0f;
		m_pointsTo[o].y = 2.0f;
		m_pointsTo[o].z = (z*5.0f/18.0f) - 3.0f;
		o++;

		m_pointsTo[o].x = 2.0f;
		m_pointsTo[o].y = 2.0f;
		m_pointsTo[o].z = (z*5.0f/18.0f) - 3.0f;
		o++;
	}

	assert( o == 6*6*6 );
}

void DelayCube::makePyramid() {
	int x, y, z, o;

	o = 0;
	for (y=0; y<27; y++) {
		float f = (y*6.0f / 27.0f) - 3.0f, f2 = (f+3.0f) / 2.0f;
		m_pointsTo[o].x = -f2;
		m_pointsTo[o].y = f;
		m_pointsTo[o].z = -f2;
		o++;

		m_pointsTo[o].x = -f2;
		m_pointsTo[o].y = f;
		m_pointsTo[o].z = f2;
		o++;

		m_pointsTo[o].x = f2;
		m_pointsTo[o].y = f;
		m_pointsTo[o].z = -f2;
		o++;

		m_pointsTo[o].x = f2;
		m_pointsTo[o].y = f;
		m_pointsTo[o].z = f2;
		o++;
	}

	for (z=0; z<27; z++) {
		m_pointsTo[o].x = -3.0f;
		m_pointsTo[o].y = 3.0f;
		m_pointsTo[o].z = (z*6.0f/27.0f) - 3.0f;
		o++;

		m_pointsTo[o].x = 3.0f;
		m_pointsTo[o].y = 3.0f;
		m_pointsTo[o].z = (z*6.0f/27.0f) - 3.0f;
		o++;
	}

	for (x=0; x<27; x++) {
		m_pointsTo[o].x = (x*6.0f/27.0f) - 3.0f;
		m_pointsTo[o].y = 3.0f;
		m_pointsTo[o].z = -3.0f;
		o++;

		m_pointsTo[o].x = (x*6.0f/27.0f) - 3.0f;
		m_pointsTo[o].y = 3.0f;
		m_pointsTo[o].z = 3.0f;
		o++;
	}

	assert( o == 6*6*6 );
}

void DelayCube::makeSphere() {
	int x, y, o;

	o = 0;
	for (y=0; y<12; y++) {
		float wx = fsin(y * 2*M_PI / 24.0f) * 6.0f;
		float wy = fcos(y * 2*M_PI / 24.0f) * 6.0f;
		for (x=0; x<18; x++) {
			m_pointsTo[o].x = fcos(x * 2*M_PI / 18.0f) * wx;
			m_pointsTo[o].y = wy;
			m_pointsTo[o].z = fsin(x * 2*M_PI / 18.0f) * wx;
			o++;
		}
	}

	assert( o == 6*6*6 );
}

// Draw a single point in 3D space. Uses the currently loaded matrix.
static inline void drawpnt(plx_dr_state_t * dr, float x, float y, float z, uint32 col) {
	// Transform the point, clip the Z plane to avoid artifacts.
	plx_mat_tfip_3d(x, y, z);
	if (z <= 0.00001f) z = 0.00001f;

	// Draw it.
	plx_vert_ind(dr, PLX_VERT, x, y+1.0f, z, col);
	plx_vert_ind(dr, PLX_VERT, x, y, z, col);
	plx_vert_ind(dr, PLX_VERT, x+1.0f, y+1.0f, z, col);
	plx_vert_ind(dr, PLX_VERT_EOS, x+1.0f, y, z, col);
}

// Draws the full cube at the current position.
void DelayCube::drawCube(float b) {
	int i;
	plx_dr_state_t dr;

	plx_dr_init(&dr);

	Color col;
	hsvSpectrum(m_hsv, col.r, col.g, col.b);
	for (i=0; i<6*6*6; i++) {
		drawpnt(&dr, m_points[i].x, m_points[i].y, m_points[i].z, (uint32)(col * b));
	}
}

// Temporary matrix so we don't have to keep recalculating our
// screenview and projection matrices. This has to be declared out
// here or GCC will ignore your aligned(32) request (potentially
// causing crashage).
static matrix_t tmpmat __attribute__((aligned(32)));

#define DELAYS 6
void DelayCube::draw(int list) {
	if (list != PLX_LIST_OP_POLY)
		return;

	// Submit the context
	plx_cxt_texture(NULL);
	plx_cxt_culling(PLX_CULL_NONE);
	plx_cxt_send(PLX_LIST_OP_POLY);

	// Setup some rotations/translations. These are "executed" from
	// bottom to top. So we rotate the cube in place on all three
	// axes a bit, then translate it to the left by 10, and rotate
	// the whole deal (so it zooms around the screen a bit). Finally
	// we translate the whole mess back by 30 so it's visible.
	plx_mat3d_identity();
	plx_mat3d_translate(0.0f, 0.0f, -30.0f);
	plx_mat3d_rotate(m_theta*0.3f, 1.0f, 0.0f, 0.0f);
	plx_mat3d_rotate(m_theta*1.5f, 0.0f, 1.0f, 0.0f);
	plx_mat3d_translate(10.0f, 0.0f, 0.0f);
	plx_mat3d_rotate(m_theta, 1.0f, 0.0f, 0.0f);
	plx_mat3d_rotate(m_theta*0.5f, 0.0f, 0.1f, 0.0f);
	plx_mat3d_rotate(m_theta*0.3f, 0.0f, 0.0f, 0.1f);

	// Save this latest modelview matrix
	for (int i=0; i<(DELAYS-1); i++)
		memcpy(&m_oldmats[i], &m_oldmats[i+1], sizeof(matrix_t));
	plx_mat3d_store(&m_oldmats[DELAYS-1]);

	Color col = getColor();

	// Draw a cube at our current position/rotation. We make a snapshot
	// of the full translation matrix after projection so that we can
	// just apply new modelview matrices for the delay cubes.
	plx_mat_identity();
	plx_mat3d_apply(PLX_MAT_SCREENVIEW);
	plx_mat3d_apply(PLX_MAT_PROJECTION);
	plx_mat_store(&tmpmat);
	plx_mat3d_apply(PLX_MAT_MODELVIEW);
	drawCube(col.r * 1.5f);

	// Now go back and draw the delay cubes
	for (int i=0; i<DELAYS-1; i++) {
		// Load the old matrix
		plx_mat3d_load(&m_oldmats[(DELAYS-2)-i]);

		// Apply it and draw the cube
		plx_mat_load(&tmpmat);
		plx_mat3d_apply(PLX_MAT_MODELVIEW);
		drawCube(col.r * (1.0f - (i+2)*(0.8f / DELAYS)));
	}
}

void DelayCube::nextFrame() {
	m_theta++;
	m_hsv += 1.0f/240.0f;

	// Are we morphing from one to another?
	if (m_needTrans) {
		// Move every point towards its new destination.
		bool any = false;
		for (int i=0; i<6*6*6; i++) {
			Vector dv = m_pointsTo[i] - m_points[i];
			if (dv.length() > 0.1f) {
				any = true;
				m_points[i] += dv * (1.0f/16.0f);
			} else
				m_points[i] = m_pointsTo[i];
		}

		// If we didn't move anything, we're done.
		if (!any)
			m_needTrans = false;
	}

	Drawable::nextFrame();
}
