//	vectors.cpp

#include <math.h>
#include "vectors.h"

// sin and cos tables
double sint[1440], cost[1440];


// fast float to int conversion
#ifndef DTOI_MAGIK
#define DTOI_MAGIK ((((65536.0 * 65536.0 * 16) + (65536.0 * 0.5)) * 65536.0))

int dtoi(double n) {
	double temp = DTOI_MAGIK + n;
	return ((*(int *)&temp) - 0x80000000);
}
#endif


// vector class implementation

void TVector::Normalize() {
	double magn = Magnitude();
	if (magn != 0.0) {
		x /= magn;
		y /= magn;
		z /= magn;
	}
}

double TVector::Magnitude() {
	return sqrt(x * x + y * y + z * z);
}


void TVector::Rotate(int ax, int ay, int az) {
	double nx = x;
	double ny = y;
	double nz = z;
	
	if (ax < 0) ax += 1440;
	if (ay < 0) ay += 1440;
	if (az < 0) az += 1440;
	
	if (ax) {
		ny = y * cost[ax] - z * sint[ax];
		nz = y * sint[ax] + z * cost[ax];
		y = ny;
		z = nz;
	}
	
	if (ay) {
		nx = x * cost[ay] - z * sint[ay];
		nz = x * sint[ay] + z * cost[ay];
		x = nx;
		z = nz;
	}
	
	if (az) {
		nx = x * cost[az] - y * sint[az];
		ny = x * sint[az] + y * cost[az];
		x = nx;
		y = ny;
	}
}


// common vector operations

double DotProduct(TVector a, TVector b) {
	return a.x * b.x + a.y * b.y + a.z * b.z;
}


TVector CrossProduct(TVector a, TVector b) {
	TVector r;
	r.x = a.y * b.z - a.z * b.y;
	r.y = b.x * a.z - a.x * b.z;
	r.z = a.x * b.y - a.y * b.x;
	return r;
}

TVector operator *(TVector a, TVector b) {
	TVector r;
	r.x = a.y * b.z - a.z * b.y;
	r.y = b.x * a.z - a.x * b.z;
	r.z = a.x * b.y - a.y * b.x;
	return r;
}


TVector AddVectors(TVector a, TVector b) {
	TVector r;
	r.x = a.x + b.x;
	r.y = a.y + b.y;
	r.z = a.z + b.z;
	return r;
}

TVector operator +(TVector a, TVector b) {
	TVector r;
	r.x = a.x + b.x;
	r.y = a.y + b.y;
	r.z = a.z + b.z;
	return r;
}

TVector operator +=(TVector &a, TVector b) {
	a.x += b.x;
	a.y += b.y;
	a.z += b.z;
	return a;
}


TVector SubVectors(TVector a, TVector b) {
	TVector r;
	r.x = a.x - b.x;
	r.y = a.y - b.y;
	r.z = a.z - b.z;
	return r;
}

TVector operator -(TVector a, TVector b) {
	TVector r;
	r.x = a.x - b.x;
	r.y = a.y - b.y;
	r.z = a.z - b.z;
	return r;
}

TVector operator -=(TVector &a, TVector b) {
	a.x -= b.x;
	a.y -= b.y;
	a.z -= b.z;
	return a;
}


TVector MulVector(TVector v, double k) {
	TVector r;
	r.x = k * v.x;
	r.y = k * v.y;
	r.z = k * v.z;
	return r;
}

TVector operator *(TVector v, double k) {
	TVector r;
	r.x = k * v.x;
	r.y = k * v.y;
	r.z = k * v.z;
	return r;
}

TVector operator *(double k, TVector v) {
	TVector r;
	r.x = k * v.x;
	r.y = k * v.y;
	r.z = k * v.z;
	return r;
}


TVector DivVector(TVector v, double k) {
	TVector r;
	r.x = v.x / k;
	r.y = v.y / k;
	r.z = v.z / k;
	return r;
}

TVector operator /(TVector v, double k) {
	TVector r;
	r.x = v.x / k;
	r.y = v.y / k;
	r.z = v.z / k;
	return r;
}


TVector CopyVector(TVector source) {
	TVector r;
	r.x = source.x;
	r.y = source.y;
	r.z = source.z;
	return r;
}


// TVertex class implementation

TVertex::TVertex() {
	SetP(0.0, 0.0, 0.0);
	normal.SetP(0.0, 0.0, 0.0);
	x2d = 0;
	y2d = 0;
	u = 0.0;
	v = 0.0;
	c = 0;
}

TVertex::TVertex(double nx, double ny, double nz) {
	SetP(nx, ny, nz);
	normal.SetP(0.0, 0.0, 0.0);
	x2d = 0;
	y2d = 0;
	u = 0.0;
	v = 0.0;
	c = 0;
}


void TVertex::CalcEnvUV() {
	u = (normal.x + 1.0) / 2.0;
	v = (normal.y + 1.0) / 2.0;
}

void TVertex::Calc2D(int xc, int yc) {
	double proy = 512.0 / (1024.0 - z);
	x2d = dtoi(x * proy) + xc;
	y2d = dtoi(-y * proy) + yc;
}

void TVertex::VRotate(int ax, int ay, int az) {
	Rotate(ax, ay, az);
	normal.Rotate(ax, ay, az);
}

void TVertex::FakeNormal() {
	normal.x = x;
	normal.y = y;
	normal.z = z;
	normal.Normalize();
}


// setup trig tables
void CalcTrigTables() {
	for (int ang = 0; ang < 1440; ang++) {
		sint[ang] = sin(ang * Pi / 720.0);
		cost[ang] = cos(ang * Pi / 720.0);
	}
}

