//	triangle.cpp

#include <stdio.h>
#include <malloc.h>
#include <i86.h>
#include "mode13.h"
#include "lightsrc.h"
#include "vectors.h"
#include "textures.h"
#include "triangle.h"


// fast float to int conversion
#ifndef DTOI_MAGIK
#define DTOI_MAGIK ((((65536.0 * 65536.0 * 16) + (65536.0 * 0.5)) * 65536.0))

int dtoi(double n) {
	double temp = DTOI_MAGIK + n;
	return ((*(int *)&temp) - 0x80000000);
}
#endif


// globals
int StartCol, WidthCol;
unsigned int TextureOffset;
int GlobalStyle;


// global vars for trifillers
TVertex *larray[3], *rarray[3];
int lsection, rsection, lheight, rheight;
double leftx, leftdx, rightx, rightdx;
double leftc, leftdc;					// for gouraud triangles
double leftu, leftdu, leftv, leftdv;	// for textured triangles



void DDALine(int x1, int y1, int x2, int y2, unsigned int where) {
	int width = x2 - x1;
	int height = y2 - y1;
	if (width < 0) width = -width;
	if (height < 0) height = - height;
	if (width > height) {
		if (height == 0) height = 1;
		if (x1 > x2) {
			int temp = x1; x1 = x2; x2 = temp;
			temp = y1; y1 = y2; y2 = temp;
		}
		int dy = (height << 16) / width;
		int yinc = (y1 > y2) ? (-320) : (320);
		unsigned int off = where + (y1 << 8) + (y1 << 6) + x1;
		_asm {
			mov edi, [off]
			mov esi, [yinc]
			mov eax, [StartCol]
			mov edx, [dy]
			xor ebx, ebx
			mov ecx, [width]
			lloop:  mov [edi], al
					inc edi
					add bx, dx
					jnc cont
					add edi, esi
			cont:   dec ecx
					jnz lloop
		}
		return;
	}
	if (width < height) {
		if (width == 0) width = 1;
		if (y1 > y2) {
			int temp = y1; y1 = y2; y2 = temp;
			temp = x1; x1 = x2; x2 = temp;
		}
		int ddx = (width << 16) / height;
		int xinc = (x1 > x2) ? (-1) : (1);
		unsigned int off = where + (y1 << 8) + (y1 << 6) + x1;
		_asm {
			mov edi, [off]
			mov esi, [xinc]
			mov eax, [StartCol]
			mov edx, [ddx]
			xor ebx, ebx
			mov ecx, [height]
			lloop:  mov [edi], al
					add edi, 320
					add bx, dx
					jnc cont
					add edi, esi
			cont:   dec ecx
					jnz lloop
		}
		return;
	}
	if (width == 0) width = 1;
	int dx = (x1 > x2) ? (-1) : (1);
	int dy = (y1 > y2) ? (-320 + dx) : (320 + dx);
	unsigned int off = where + (y1 << 8) + (y1 << 6) + x1;
	_asm {
		mov edi, [off]
		mov eax, [StartCol]
		mov ebx, [dy]
		mov ecx, [width]
		lloop:  mov [edi], al
				add edi, ebx
				dec ecx
				jnz lloop
	}
}

void ClipDDALine(int x1, int y1, int x2, int y2, unsigned int where) {
	//	line clipping
	double m = (double)(y2 - y1) / (double)(x2 - x1);
	double im = 1 / m;
	if (x1 <= x2) {
		if ((x1 > 319) || (x2 < 0)) return;
		if (x1 < 0) { y1 += dtoi(-x1 * m); x1 = 0; }
		if (x2 > 319) { y2 -= dtoi((x2 - 319) * m); x2 = 319; }
	}
	else {
		if ((x1 < 0) || (x2 > 319)) return;
		if (x2 < 0) { y2 += dtoi(-x2 * m); x2 = 0; }
		if (x1 > 319) { y1 -= dtoi((x1 - 319) * m); x1 = 319; }
	}
	if (y1 <= y2) {
		if ((y1 > 199) || (y2 < 0)) return;
		if (y1 < 0) { x1 += dtoi(-y1 * im); y1 = 0; }
		if (y2 > 199) { x2 -= dtoi((y2 - 199) * im); y2 = 199; }
	}
	else {
		if ((y1 < 0) || (y2 > 199)) return;
		if (y2 < 0) { x2 += dtoi(-y2 * im); y2 = 0; }
		if (y1 > 199) { x1 -= dtoi((y1 - 199) * im); y1 = 199; }
	}
	
	int width = x2 - x1;
	int height = y2 - y1;
	if (width < 0) width = -width;
	if (height < 0) height = - height;
	if (width > height) {
		if (height == 0) height = 1;
		if (x1 > x2) {
			int temp = x1; x1 = x2; x2 = temp;
			temp = y1; y1 = y2; y2 = temp;
		}
		int dy = (height << 16) / width;
		int yinc = (y1 > y2) ? (-320) : (320);
		unsigned int off = where + (y1 << 8) + (y1 << 6) + x1;
		_asm {
			mov edi, [off]
			mov esi, [yinc]
			mov eax, [StartCol]
			mov edx, [dy]
			xor ebx, ebx
			mov ecx, [width]
			lloop:  mov [edi], al
					inc edi
					add bx, dx
					jnc cont
					add edi, esi
			cont:   dec ecx
					jnz lloop
		}
		return;
	}
	if (width < height) {
		if (width == 0) width = 1;
		if (y1 > y2) {
			int temp = y1; y1 = y2; y2 = temp;
			temp = x1; x1 = x2; x2 = temp;
		}
		int ddx = (width << 16) / height;
		int xinc = (x1 > x2) ? (-1) : (1);
		unsigned int off = where + (y1 << 8) + (y1 << 6) + x1;
		_asm {
			mov edi, [off]
			mov esi, [xinc]
			mov eax, [StartCol]
			mov edx, [ddx]
			xor ebx, ebx
			mov ecx, [height]
			lloop:  mov [edi], al
					add edi, 320
					add bx, dx
					jnc cont
					add edi, esi
			cont:   dec ecx
					jnz lloop
		}
		return;
	}
	if (width == 0) width = 1;
	int dx = (x1 > x2) ? (-1) : (1);
	int dy = (y1 > y2) ? (-320 + dx) : (320 + dx);
	unsigned int off = where + (y1 << 8) + (y1 << 6) + x1;
	_asm {
		mov edi, [off]
		mov eax, [StartCol]
		mov ebx, [dy]
		mov ecx, [width]
		lloop:  mov [edi], al
				add edi, ebx
				dec ecx
				jnz lloop
	}
}


void ClipAddDDALine(int x1, int y1, int x2, int y2, unsigned int where) {
	//	line clipping
	double m = (double)(y2 - y1) / (double)(x2 - x1);
	double im = 1 / m;
	if (x1 <= x2) {
		if ((x1 > 319) || (x2 < 0)) return;
		if (x1 < 0) { y1 += dtoi(-x1 * m); x1 = 0; }
		if (x2 > 319) { y2 -= dtoi((x2 - 319) * m); x2 = 319; }
	}
	else {
		if ((x1 < 0) || (x2 > 319)) return;
		if (x2 < 0) { y2 += dtoi(-x2 * m); x2 = 0; }
		if (x1 > 319) { y1 -= dtoi((x1 - 319) * m); x1 = 319; }
	}
	if (y1 <= y2) {
		if ((y1 > 199) || (y2 < 0)) return;
		if (y1 < 0) { x1 += dtoi(-y1 * im); y1 = 0; }
		if (y2 > 199) { x2 -= dtoi((y2 - 199) * im); y2 = 199; }
	}
	else {
		if ((y1 < 0) || (y2 > 199)) return;
		if (y2 < 0) { x2 += dtoi(-y2 * im); y2 = 0; }
		if (y1 > 199) { x1 -= dtoi((y1 - 199) * im); y1 = 199; }
	}
	
	int width = x2 - x1;
	int height = y2 - y1;
	if (width < 0) width = -width;
	if (height < 0) height = - height;
	if (width > height) {
		if (height == 0) height = 1;
		if (x1 > x2) {
			int temp = x1; x1 = x2; x2 = temp;
			temp = y1; y1 = y2; y2 = temp;
		}
		int dy = (height << 16) / width;
		int yinc = (y1 > y2) ? (-320) : (320);
		unsigned int off = where + (y1 << 8) + (y1 << 6) + x1;
		_asm {
			mov edi, [off]
			mov esi, [yinc]
			mov eax, [StartCol]
			mov edx, [dy]
			xor ebx, ebx
			mov ecx, [width]
                        lloop:  add byte ptr [edi], al
                                jnc nomax
                                mov al, 0FFh
                                mov [edi], al
                        nomax:  inc edi
                                add bx, dx
                                jnc cont
                                add edi, esi
			cont:   dec ecx
                                jnz lloop
		}
		return;
	}
	if (width < height) {
		if (width == 0) width = 1;
		if (y1 > y2) {
			int temp = y1; y1 = y2; y2 = temp;
			temp = x1; x1 = x2; x2 = temp;
		}
		int ddx = (width << 16) / height;
		int xinc = (x1 > x2) ? (-1) : (1);
		unsigned int off = where + (y1 << 8) + (y1 << 6) + x1;
		_asm {
			mov edi, [off]
			mov esi, [xinc]
			mov eax, [StartCol]
			mov edx, [ddx]
			xor ebx, ebx
			mov ecx, [height]
                        lloop:  add byte ptr [edi], al
                                jnc nomax
                                mov al, 0FFh
                                mov [edi], al
                        nomax:  add edi, 320
                                add bx, dx
                                jnc cont
                                add edi, esi
                        cont:   dec ecx
                                jnz lloop
		}
		return;
	}
	if (width == 0) width = 1;
	int dx = (x1 > x2) ? (-1) : (1);
	int dy = (y1 > y2) ? (-320 + dx) : (320 + dx);
	unsigned int off = where + (y1 << 8) + (y1 << 6) + x1;
	_asm {
		mov edi, [off]
		mov eax, [StartCol]
		mov ebx, [dy]
		mov ecx, [width]
		lloop:  mov [edi], al
                        jnc nomax
                        mov al, 0FFh
                        mov [edi], al
                nomax:  add edi, ebx
                        dec ecx
                        jnz lloop
	}
}


// all triangles use the same DoRightSide function
int DoRightSide() {
	TVertex *tv1 = rarray[rsection];
	TVertex *tv2 = rarray[rsection - 1];
	int height = tv2->y2d - tv1->y2d;
	if (height == 0) return 0;
	rightdx = (tv2->x2d - tv1->x2d) / (double)height;
	rightx = tv1->x2d;
	rheight = height;
	return height;
}


// DoLeftSide function for flat filled triangles
int FlatDoLeftSide() {
	TVertex *tv1 = larray[lsection];
	TVertex *tv2 = larray[lsection - 1];
	int height = tv2->y2d - tv1->y2d;
	if (height == 0) return 0;
	leftdx = (tv2->x2d - tv1->x2d) / (double)height;
	leftx = tv1->x2d;
	lheight = height;
	return height;
}



// flat triangle filler
void FlatTriangle(TVertex *vt1, TVertex *vt2, TVertex *vt3, unsigned int where) {
	TVertex *v1 = vt1;
	TVertex *v2 = vt2;
	TVertex *v3 = vt3;
	if (v1->y2d > v2->y2d) { TVertex *v = v1; v1 = v2; v2 = v; }
	if (v1->y2d > v3->y2d) { TVertex *v = v1; v1 = v3; v3 = v; }
	if (v2->y2d > v3->y2d) { TVertex *v = v2; v2 = v3; v3 = v; }
	int height = v3->y2d - v1->y2d;
	if (height == 0) return;
	double temp = (v2->y2d - v1->y2d) / (double)height;
	double longest = temp * (v3->x2d - v1->x2d) + v1->x2d - v2->x2d;
	if (longest == 0.0) return;
	if (longest < 0.0) {
		rarray[0] = v3; rarray[1] = v2; rarray[2] = v1;
		rsection = 2;
		larray[0] = v3; larray[1] = v1;
		lsection = 1;
		if (FlatDoLeftSide() <= 0) return;
		if (DoRightSide() <= 0) {
			rsection--;
			if (DoRightSide() <= 0) return;
		}
		if (longest > -1.0) longest = -1.0;
	}
	else {
		larray[0] = v3; larray[1] = v2; larray[2] = v1;
		lsection = 2;
		rarray[0] = v3; rarray[1] = v1;
		rsection = 1;
		if (DoRightSide() <= 0) return;
		if (FlatDoLeftSide() <= 0) {
			lsection--;
			if (FlatDoLeftSide() <= 0) return;
		}
		if (longest < 1.0) longest = 1.0;
	}
	unsigned int off = where + v1->y2d * 320;
	while (1) {
		int x1 = dtoi(leftx);
		int width = dtoi(rightx) - x1;
		if (width > 0)
			_asm {
				mov edi, [x1]
				mov eax, [StartCol]
				add edi, [off]
				mov ecx, [width]
				loop1:  mov [edi], al
						inc edi
						dec ecx
						jnz loop1
			}
		off += 320;
		if (--lheight <= 0) {
			if (--lsection <= 0) return;
			if (FlatDoLeftSide() <= 0) return;
		}
		else leftx += leftdx;
		if (--rheight <= 0) {
			if (--rsection <= 0) return;
			if (DoRightSide() <= 0) return;
		}
		else rightx += rightdx;
	}
}


void ClipFlatTriangle(TVertex *vt1, TVertex *vt2, TVertex *vt3, unsigned int where) {
	TVertex *v1 = vt1;
	TVertex *v2 = vt2;
	TVertex *v3 = vt3;
	if (v1->y2d > v2->y2d) { TVertex *v = v1; v1 = v2; v2 = v; }
	if (v1->y2d > v3->y2d) { TVertex *v = v1; v1 = v3; v3 = v; }
	if (v2->y2d > v3->y2d) { TVertex *v = v2; v2 = v3; v3 = v; }
	int height = v3->y2d - v1->y2d;
	if (height == 0) return;
	double temp = (v2->y2d - v1->y2d) / (double)height;
	double longest = temp * (v3->x2d - v1->x2d) + v1->x2d - v2->x2d;
	if (longest == 0.0) return;
	if (longest < 0.0) {
		rarray[0] = v3; rarray[1] = v2; rarray[2] = v1;
		rsection = 2;
		larray[0] = v3; larray[1] = v1;
		lsection = 1;
		if (FlatDoLeftSide() <= 0) return;
		if (DoRightSide() <= 0) {
			rsection--;
			if (DoRightSide() <= 0) return;
		}
		if (longest > -1.0) longest = -1.0;
	}
	else {
		larray[0] = v3; larray[1] = v2; larray[2] = v1;
		lsection = 2;
		rarray[0] = v3; rarray[1] = v1;
		rsection = 1;
		if (DoRightSide() <= 0) return;
		if (FlatDoLeftSide() <= 0) {
			lsection--;
			if (FlatDoLeftSide() <= 0) return;
		}
		if (longest < 1.0) longest = 1.0;
	}
	unsigned int clipymax = where + 63680;
	unsigned int off = where + v1->y2d * 320;
	while (1) {
		if ((off >= where) && (off <= clipymax)) {
			int x1 = dtoi(leftx);
			int x2 = dtoi(rightx);
			int width = x2 - x1;
			if (x1 < 0) { width = x2; x1 = 0; }
			if (x2 > 319) width = 319 - x1;
			if (width > 0)
				_asm {
					mov edi, [x1]
					mov eax, [StartCol]
					add edi, [off]
					mov ecx, [width]
					loop1:  mov [edi], al
							inc edi
							dec ecx
							jnz loop1
				}
		}
		off += 320;
		if (--lheight <= 0) {
			if (--lsection <= 0) return;
			if (FlatDoLeftSide() <= 0) return;
		}
		else leftx += leftdx;
		if (--rheight <= 0) {
			if (--rsection <= 0) return;
			if (DoRightSide() <= 0) return;
		}
		else rightx += rightdx;
	}
}




// LeftSide function for Gouraud shaded triangles
int GouraudDoLeftSide() {
	TVertex *tv1 = larray[lsection];
	TVertex *tv2 = larray[lsection - 1];
	int height = tv2->y2d - tv1->y2d;
	if (height == 0) return 0;
	leftdx = (tv2->x2d - tv1->x2d) / (double)height;
	leftdc = (double)(tv2->c - tv1->c) / (double)height;
	leftx = tv1->x2d;
	leftc = tv1->c;
	lheight = height;
	return height;
}


// Gouraud trifiller

void GouraudTriangle(TVertex *vt1, TVertex *vt2, TVertex *vt3, unsigned int where) {
	int c, dc;
	TVertex *v1 = vt1;
	TVertex *v2 = vt2;
	TVertex *v3 = vt3;
	if (v1->y2d > v2->y2d) { TVertex *v = v1; v1 = v2; v2 = v; }
	if (v1->y2d > v3->y2d) { TVertex *v = v1; v1 = v3; v3 = v; }
	if (v2->y2d > v3->y2d) { TVertex *v = v2; v2 = v3; v3 = v; }
	int height = v3->y2d - v1->y2d;
	if (height == 0) return;
	double temp = (v2->y2d - v1->y2d) / (double)height;
	double longest = temp * (v3->x2d - v1->x2d) + v1->x2d - v2->x2d;
	if (longest == 0.0) return;
//	v1->c = ((DotProduct(v1->normal, Light) + 1) * WidthCol) + StartCol;
//	v2->c = ((DotProduct(v2->normal, Light) + 1) * WidthCol) + StartCol;
//	v3->c = ((DotProduct(v3->normal, Light) + 1) * WidthCol) + StartCol;
	if (longest < 0.0) {
		rarray[0] = v3; rarray[1] = v2; rarray[2] = v1;
		rsection = 2;
		larray[0] = v3; larray[1] = v1;
		lsection = 1;
		if (GouraudDoLeftSide() <= 0) return;
		if (DoRightSide() <= 0) {
			rsection--;
			if (DoRightSide() <= 0) return;
		}
		if (longest > -1.0) longest = -1.0;
	}
	else {
		larray[0] = v3; larray[1] = v2; larray[2] = v1;
		lsection = 2;
		rarray[0] = v3; rarray[1] = v1;
		rsection = 1;
		if (DoRightSide() <= 0) return;
		if (GouraudDoLeftSide() <= 0) {
			lsection--;
			if (GouraudDoLeftSide() <= 0) return;
		}
		if (longest < 1.0) longest = 1.0;
	}
	double dcdx = (temp * (v3->c - v1->c) + v1->c - v2->c) / longest;
	unsigned int off = where + v1->y2d * 320;
	while (1) {
		int x1 = dtoi(leftx);
		int width = dtoi(rightx) - x1;
		if (width > 0) {
			c = dtoi(leftc * 256);
			dc = dtoi(dcdx * 256);
			_asm {
				mov edi, [x1]
				mov ebx, [dc]
				mov eax, [c]
				add edi, [off]
				mov ecx, [width]
				loop1:  mov [edi], ah
						add eax, ebx
						inc edi
						dec ecx
						jnz loop1
			}
		}
		off += 320;
		if (--lheight <= 0) {
			if (--lsection <= 0) return;
			if (GouraudDoLeftSide() <= 0) return;
		}
		else {
			leftx += leftdx;
			leftc += leftdc;
		}
		if (--rheight <= 0) {
			if (--rsection <= 0) return;
			if (DoRightSide() <= 0) return;
		}
		else rightx += rightdx;
	}
}


void ClipGouraudTriangle(TVertex *vt1, TVertex *vt2, TVertex *vt3, unsigned int where) {
	int c, dc;
	TVertex *v1 = vt1;
	TVertex *v2 = vt2;
	TVertex *v3 = vt3;
	if (v1->y2d > v2->y2d) { TVertex *v = v1; v1 = v2; v2 = v; }
	if (v1->y2d > v3->y2d) { TVertex *v = v1; v1 = v3; v3 = v; }
	if (v2->y2d > v3->y2d) { TVertex *v = v2; v2 = v3; v3 = v; }
	int height = v3->y2d - v1->y2d;
	if (height == 0) return;
	double temp = (v2->y2d - v1->y2d) / (double)height;
	double longest = temp * (v3->x2d - v1->x2d) + v1->x2d - v2->x2d;
	if (longest == 0.0) return;
//	v1->c = ((DotProduct(v1->normal, Light) + 1) * WidthCol) + StartCol;
//	v2->c = ((DotProduct(v2->normal, Light) + 1) * WidthCol) + StartCol;
//	v3->c = ((DotProduct(v3->normal, Light) + 1) * WidthCol) + StartCol;
	if (longest < 0.0) {
		rarray[0] = v3; rarray[1] = v2; rarray[2] = v1;
		rsection = 2;
		larray[0] = v3; larray[1] = v1;
		lsection = 1;
		if (GouraudDoLeftSide() <= 0) return;
		if (DoRightSide() <= 0) {
			rsection--;
			if (DoRightSide() <= 0) return;
		}
		if (longest > -1.0) longest = -1.0;
	}
	else {
		larray[0] = v3; larray[1] = v2; larray[2] = v1;
		lsection = 2;
		rarray[0] = v3; rarray[1] = v1;
		rsection = 1;
		if (DoRightSide() <= 0) return;
		if (GouraudDoLeftSide() <= 0) {
			lsection--;
			if (GouraudDoLeftSide() <= 0) return;
		}
		if (longest < 1.0) longest = 1.0;
	}
	double dcdx = (temp * (v3->c - v1->c) + v1->c - v2->c) / longest;
	unsigned int clipymax = where + 63680;
	unsigned int off = where + v1->y2d * 320;
	while (1) {
		if ((off >= where) && (off <= clipymax)) {
			int x1 = dtoi(leftx);
			int x2 = dtoi(rightx);
			int width = x2 - x1;
			int clipflag = 0;
			if (x1 < 0) {
				width = x2;
				clipflag = -x1;
				x1 = 0;
			}
			if (x2 > 319) width = 319 - x1;
			if (width > 0) {
				c = dtoi(leftc * 256);
				dc = dtoi(dcdx * 256);
				if (clipflag) c += dc * clipflag;
				_asm {
					mov edi, [x1]
					mov ebx, [dc]
					mov eax, [c]
					add edi, [off]
					mov ecx, [width]
					loop1:  mov [edi], ah
							add eax, ebx
							inc edi
							dec ecx
							jnz loop1
				}
			}
		}
		off += 320;
		if (--lheight <= 0) {
			if (--lsection <= 0) return;
			if (GouraudDoLeftSide() <= 0) return;
		}
		else {
			leftx += leftdx;
			leftc += leftdc;
		}
		if (--rheight <= 0) {
			if (--rsection <= 0) return;
			if (DoRightSide() <= 0) return;
		}
		else rightx += rightdx;
	}
}



// DoLeftSide function for texture mapped triangles
int TextureDoLeftSide() {
	TVertex *tv1 = larray[lsection];
	TVertex *tv2 = larray[lsection - 1];
	int height = tv2->y2d - tv1->y2d;
	if (height == 0) return 0;
	leftdx = (tv2->x2d - tv1->x2d) / (double)height;
	leftdu = (tv2->u - tv1->u) / (double)height;
	leftdv = (tv2->v - tv1->v) / (double)height;
	leftx = tv1->x2d;
	leftu = tv1->u;
	leftv = tv1->v;
	lheight = height;
	return height;
}



// texture mapped triangle filler
void TextureTriangle(TVertex *vt1, TVertex *vt2, TVertex *vt3, unsigned int where) {
	int u, v, du, dv;
	TVertex *v1 = vt1;
	TVertex *v2 = vt2;
	TVertex *v3 = vt3;
	if (v1->y2d > v2->y2d) { TVertex *v = v1; v1 = v2; v2 = v; }
	if (v1->y2d > v3->y2d) { TVertex *v = v1; v1 = v3; v3 = v; }
	if (v2->y2d > v3->y2d) { TVertex *v = v2; v2 = v3; v3 = v; }
	int height = v3->y2d - v1->y2d;
	if (height == 0) return;
	double temp = (v2->y2d - v1->y2d) / (double)height;
	double longest = temp * (v3->x2d - v1->x2d) + v1->x2d - v2->x2d;
	if (longest == 0.0) return;
	if (longest < 0.0) {
		rarray[0] = v3; rarray[1] = v2; rarray[2] = v1;
		rsection = 2;
		larray[0] = v3; larray[1] = v1;
		lsection = 1;
		if (TextureDoLeftSide() <= 0) return;
		if (DoRightSide() <= 0) {
			rsection--;
			if (DoRightSide() <= 0) return;
		}
		if (longest > -1.0) longest = -1.0;
	}
	else {
		larray[0] = v3; larray[1] = v2; larray[2] = v1;
		lsection = 2;
		rarray[0] = v3; rarray[1] = v1;
		rsection = 1;
		if (DoRightSide() <= 0) return;
		if (TextureDoLeftSide() <= 0) {
			lsection--;
			if (TextureDoLeftSide() <= 0) return;
		}
		if (longest < 1.0) longest = 1.0;
	}
	double dudx = (temp * (v3->u - v1->u) + v1->u - v2->u) / longest;
	double dvdx = (temp * (v3->v - v1->v) + v1->v - v2->v) / longest;
	unsigned int off = where + v1->y2d * 320;
	while (1) {
		int x1 = dtoi(leftx);
		int width = dtoi(rightx) - x1;
		if (width > 0) {
			u = dtoi(leftu * 65536);
			v = dtoi(leftv * 65536);
			du = dtoi(dudx * 65536);
			dv = dtoi(dvdx * 65536);
			_asm {
				mov edi, [x1]
				mov esi, [v]
				mov edx, [u]
				add edi, [off]
				mov ebx, [TextureOffset]
				mov ecx, [width]
				dec edi
				loop1:  mov bx, si
						add esi, [dv]
						mov bl, dh
						inc edi
						mov al, [ebx]
						add edx, [du]
						mov [edi], al
						dec ecx
						jnz loop1
			}
		}
		off += 320;
		if (--lheight <= 0) {
			if (--lsection <= 0) return;
			if (TextureDoLeftSide() <= 0) return;
		}
		else {
			leftx += leftdx;
			leftu += leftdu;
			leftv += leftdv;
		}
		if (--rheight <= 0) {
			if (--rsection <= 0) return;
			if (DoRightSide() <= 0) return;
		}
		else rightx += rightdx;
	}
}


// clipped texture mapped triangle
void ClipTextureTriangle(TVertex *vt1, TVertex *vt2, TVertex *vt3, unsigned int where) {
	int u, v, du, dv;
	TVertex *v1 = vt1;
	TVertex *v2 = vt2;
	TVertex *v3 = vt3;
	if (v1->y2d > v2->y2d) { TVertex *v = v1; v1 = v2; v2 = v; }
	if (v1->y2d > v3->y2d) { TVertex *v = v1; v1 = v3; v3 = v; }
	if (v2->y2d > v3->y2d) { TVertex *v = v2; v2 = v3; v3 = v; }
	int height = v3->y2d - v1->y2d;
	if (height == 0) return;
	double temp = (v2->y2d - v1->y2d) / (double)height;
	double longest = temp * (v3->x2d - v1->x2d) + v1->x2d - v2->x2d;
	if (longest == 0.0) return;
	if (longest < 0.0) {
		rarray[0] = v3; rarray[1] = v2; rarray[2] = v1;
		rsection = 2;
		larray[0] = v3; larray[1] = v1;
		lsection = 1;
		if (TextureDoLeftSide() <= 0) return;
		if (DoRightSide() <= 0) {
			rsection--;
			if (DoRightSide() <= 0) return;
		}
		if (longest > -1.0) longest = -1.0;
	}
	else {
		larray[0] = v3; larray[1] = v2; larray[2] = v1;
		lsection = 2;
		rarray[0] = v3; rarray[1] = v1;
		rsection = 1;
		if (DoRightSide() <= 0) return;
		if (TextureDoLeftSide() <= 0) {
			lsection--;
			if (TextureDoLeftSide() <= 0) return;
		}
		if (longest < 1.0) longest = 1.0;
	}
	double dudx = (temp * (v3->u - v1->u) + v1->u - v2->u) / longest;
	double dvdx = (temp * (v3->v - v1->v) + v1->v - v2->v) / longest;
	unsigned int clipymax = where + 63680;
	unsigned int off = where + v1->y2d * 320;
	while (1) {
		if ((off >= where) && (off <= clipymax)) {
			int x1 = dtoi(leftx);
			int x2 = dtoi(rightx);
			int width = x2 - x1;
			int clipflag = 0;
			if (x1 < 0) {
				width = x2;
				clipflag = -x1;
				x1 = 0;
			}
			if (x2 > 319) width = 319 - x1;
			if (width > 0) {
				u = dtoi(leftu * 65536);
				v = dtoi(leftv * 65536);
				du = dtoi(dudx * 65536);
				dv = dtoi(dvdx * 65536);
				if (clipflag) {
					u += du * clipflag;
					v += dv * clipflag;
				}
				_asm {
					mov edi, [x1]
					mov esi, [v]
					mov edx, [u]
					add edi, [off]
					mov ecx, [width]
					dec edi
					loop1:  movzx ebx, si
							add esi, [dv]
							mov bl, dh
							inc edi
							add ebx, [TextureOffset]
							mov al, [ebx]
							add edx, [du]
							mov [edi], al
							dec ecx
							jnz loop1
				}
			}
		}
		off += 320;
		if (--lheight <= 0) {
			if (--lsection <= 0) return;
			if (TextureDoLeftSide() <= 0) return;
		}
		else {
			leftx += leftdx;
			leftu += leftdu;
			leftv += leftdv;
		}
		if (--rheight <= 0) {
			if (--rsection <= 0) return;
			if (DoRightSide() <= 0) return;
		}
		else rightx += rightdx;
	}
}


void ClipAddTextureTriangle(TVertex *vt1, TVertex *vt2, TVertex *vt3, unsigned int where) {
	int u, v, du, dv;
	TVertex *v1 = vt1;
	TVertex *v2 = vt2;
	TVertex *v3 = vt3;
	if (v1->y2d > v2->y2d) { TVertex *v = v1; v1 = v2; v2 = v; }
	if (v1->y2d > v3->y2d) { TVertex *v = v1; v1 = v3; v3 = v; }
	if (v2->y2d > v3->y2d) { TVertex *v = v2; v2 = v3; v3 = v; }
	int height = v3->y2d - v1->y2d;
	if (height == 0) return;
	double temp = (v2->y2d - v1->y2d) / (double)height;
	double longest = temp * (v3->x2d - v1->x2d) + v1->x2d - v2->x2d;
	if (longest == 0.0) return;
	if (longest < 0.0) {
		rarray[0] = v3; rarray[1] = v2; rarray[2] = v1;
		rsection = 2;
		larray[0] = v3; larray[1] = v1;
		lsection = 1;
		if (TextureDoLeftSide() <= 0) return;
		if (DoRightSide() <= 0) {
			rsection--;
			if (DoRightSide() <= 0) return;
		}
		if (longest > -1.0) longest = -1.0;
	}
	else {
		larray[0] = v3; larray[1] = v2; larray[2] = v1;
		lsection = 2;
		rarray[0] = v3; rarray[1] = v1;
		rsection = 1;
		if (DoRightSide() <= 0) return;
		if (TextureDoLeftSide() <= 0) {
			lsection--;
			if (TextureDoLeftSide() <= 0) return;
		}
		if (longest < 1.0) longest = 1.0;
	}
	double dudx = (temp * (v3->u - v1->u) + v1->u - v2->u) / longest;
	double dvdx = (temp * (v3->v - v1->v) + v1->v - v2->v) / longest;
	unsigned int clipymax = where + 63680;
	unsigned int off = where + v1->y2d * 320;
	while (1) {
		if ((off >= where) && (off <= clipymax)) {
			int x1 = dtoi(leftx);
			int x2 = dtoi(rightx);
			int width = x2 - x1;
			int clipflag = 0;
			if (x1 < 0) {
				width = x2;
				clipflag = -x1;
				x1 = 0;
			}
			if (x2 > 319) width = 319 - x1;
			if (width > 0) {
				u = dtoi(leftu * 65536);
				v = dtoi(leftv * 65536);
				du = dtoi(dudx * 65536);
				dv = dtoi(dvdx * 65536);
				if (clipflag) {
					u += du * clipflag;
					v += dv * clipflag;
				}
				_asm {
					mov edi, [x1]
					mov esi, [v]
					mov edx, [u]
					add edi, [off]
					mov ebx, [TextureOffset]
					mov ecx, [width]
					dec edi
					loop1:  movzx ebx, si
							add esi, [dv]
							mov bl, dh
							inc edi
							add ebx, [TextureOffset]
							mov al, [ebx]
							shl al, 1
							add edx, [du]
							add al, [edi]
							jnc cont
							mov al, 0FFh
					cont:   mov [edi], al
							dec ecx
							jnz loop1
				}
			}
		}
		off += 320;
		if (--lheight <= 0) {
			if (--lsection <= 0) return;
			if (TextureDoLeftSide() <= 0) return;
		}
		else {
			leftx += leftdx;
			leftu += leftdu;
			leftv += leftdv;
		}
		if (--rheight <= 0) {
			if (--rsection <= 0) return;
			if (DoRightSide() <= 0) return;
		}
		else rightx += rightdx;
	}
}


void ClipAddTextureTriangle2(TVertex *vt1, TVertex *vt2, TVertex *vt3, unsigned int where) {
	int u, v, du, dv;
	TVertex *v1 = vt1;
	TVertex *v2 = vt2;
	TVertex *v3 = vt3;
	if (v1->y2d > v2->y2d) { TVertex *v = v1; v1 = v2; v2 = v; }
	if (v1->y2d > v3->y2d) { TVertex *v = v1; v1 = v3; v3 = v; }
	if (v2->y2d > v3->y2d) { TVertex *v = v2; v2 = v3; v3 = v; }
	int height = v3->y2d - v1->y2d;
	if (height == 0) return;
	double temp = (v2->y2d - v1->y2d) / (double)height;
	double longest = temp * (v3->x2d - v1->x2d) + v1->x2d - v2->x2d;
	if (longest == 0.0) return;
	if (longest < 0.0) {
		rarray[0] = v3; rarray[1] = v2; rarray[2] = v1;
		rsection = 2;
		larray[0] = v3; larray[1] = v1;
		lsection = 1;
		if (TextureDoLeftSide() <= 0) return;
		if (DoRightSide() <= 0) {
			rsection--;
			if (DoRightSide() <= 0) return;
		}
		if (longest > -1.0) longest = -1.0;
	}
	else {
		larray[0] = v3; larray[1] = v2; larray[2] = v1;
		lsection = 2;
		rarray[0] = v3; rarray[1] = v1;
		rsection = 1;
		if (DoRightSide() <= 0) return;
		if (TextureDoLeftSide() <= 0) {
			lsection--;
			if (TextureDoLeftSide() <= 0) return;
		}
		if (longest < 1.0) longest = 1.0;
	}
	double dudx = (temp * (v3->u - v1->u) + v1->u - v2->u) / longest;
	double dvdx = (temp * (v3->v - v1->v) + v1->v - v2->v) / longest;
	unsigned int clipymax = where + 63680;
	unsigned int off = where + v1->y2d * 320;
	while (1) {
		if ((off >= where) && (off <= clipymax)) {
			int x1 = dtoi(leftx);
			int x2 = dtoi(rightx);
			int width = x2 - x1;
			int clipflag = 0;
			if (x1 < 0) {
				width = x2;
				clipflag = -x1;
				x1 = 0;
			}
			if (x2 > 319) width = 319 - x1;
			if (width > 0) {
				u = dtoi(leftu * 65536);
				v = dtoi(leftv * 65536);
				du = dtoi(dudx * 65536);
				dv = dtoi(dvdx * 65536);
				if (clipflag) {
					u += du * clipflag;
					v += dv * clipflag;
				}
				_asm {
					mov edi, [x1]
					mov esi, [v]
					mov edx, [u]
					add edi, [off]
					mov ebx, [TextureOffset]
					mov ecx, [width]
					dec edi
					loop1:  movzx ebx, si
							add esi, [dv]
							mov bl, dh
							inc edi
							add ebx, [TextureOffset]
							mov al, [ebx]
                                                        sub al, 240
                                                        shl al, 3
							add edx, [du]
							add al, [edi]
							jnc cont
							mov al, 0FFh
					cont:   mov [edi], al
							dec ecx
							jnz loop1
				}
			}
		}
		off += 320;
		if (--lheight <= 0) {
			if (--lsection <= 0) return;
			if (TextureDoLeftSide() <= 0) return;
		}
		else {
			leftx += leftdx;
			leftu += leftdu;
			leftv += leftdv;
		}
		if (--rheight <= 0) {
			if (--rsection <= 0) return;
			if (DoRightSide() <= 0) return;
		}
		else rightx += rightdx;
	}
}



//	particle triangle
void AddParticle(int x, int y, int color, unsigned int where);
#pragma aux AddParticle = 		\
"mov ecx, ebx"				 \
"shl ebx, 8"				   \
"shl ecx, 6"				   \
"add ebx, ecx"				 \
"add ebx, edx"				 \
"add ebx, edi"				 \
"add al, [ebx]"				\
"jnc @fin"					 \
"mov al, 0xFF"				 \
"@fin: mov ah, al"			\
"mov [ebx], ax"				\
"mov [ebx+320], ax"			\
parm [edx] [ebx] [eax] [edi]   \
modify [ecx];

void ClipParticleTriangle(TVertex *v1, TVertex *v2, TVertex *v3, unsigned int where) {
	if ((v1->x2d >= 0) && (v1->x2d < 319) && (v1->y2d >= 0) && (v1->y2d < 199))
		AddParticle(v1->x2d, v1->y2d, v1->c, where);
	if ((v2->x2d >= 0) && (v2->x2d < 319) && (v2->y2d >= 0) && (v2->y2d < 199))
		AddParticle(v2->x2d, v2->y2d, v2->c, where);
	if ((v3->x2d >= 0) && (v3->x2d < 319) && (v3->y2d >= 0) && (v3->y2d < 199))
		AddParticle(v3->x2d, v3->y2d, v3->c, where);
}

// general drawing function
void DrawTriangle(TVertex *v1, TVertex *v2, TVertex *v3, unsigned int where) {
	switch (GlobalStyle) {
		case nc_wireframe:
			DDALine(v1->x2d, v1->y2d, v2->x2d, v2->y2d, where);
			DDALine(v2->x2d, v2->y2d, v3->x2d, v3->y2d, where);
			DDALine(v3->x2d, v3->y2d, v1->x2d, v1->y2d, where); break;
		case nc_flatshaded: FlatTriangle(v1, v2, v3, where); break;
		case nc_gouraudshaded: GouraudTriangle(v1, v2, v3, where); break;
		case nc_texturemapped: TextureTriangle(v1, v2, v3, where); break;
		case c_wireframe:
			ClipDDALine(v1->x2d, v1->y2d, v2->x2d, v2->y2d, where);
			ClipDDALine(v2->x2d, v2->y2d, v3->x2d, v3->y2d, where);
			ClipDDALine(v3->x2d, v3->y2d, v1->x2d, v1->y2d, where); break;
		case c_flatshaded: ClipFlatTriangle(v1, v2, v3, where); break;
		case c_gouraudshaded: ClipGouraudTriangle(v1, v2, v3, where); break;
		case c_texturemapped: ClipTextureTriangle(v1, v2, v3, where); break;
		case c_particles: ClipParticleTriangle(v1, v2, v3, where); break;
                case ca_wireframe:
                        ClipAddDDALine(v1->x2d, v1->y2d, v2->x2d, v2->y2d, where);
                        ClipAddDDALine(v2->x2d, v2->y2d, v3->x2d, v3->y2d, where);
                        ClipAddDDALine(v3->x2d, v3->y2d, v1->x2d, v1->y2d, where); break;
		case ca_texturemapped: ClipAddTextureTriangle(v1, v2, v3, where); break;
                case ca_texturemapped2: ClipAddTextureTriangle2(v1, v2, v3, where); break;
	}
}

