/*
  pic.c - part of intro
  Copyright (C) 2001, Bekir Osman Keskin <spectre@portent.net>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "intro.h"

picture *load_pcx (char *fn)
{
  FILE *f;
  uchar ch, cn;
  int i, p = 0, size;
  ushort sx, sy;
  picture *pic;

  f = fopen (fn, "rb");
  if (f == 0)
  {
    printf ("can't load pcx file [%s]\n", fn);
    exit (1);
  }

  fseek (f, 0x8, SEEK_SET);
  fread (&sx, 2, 1, f);
  fread (&sy, 2, 1, f);
  sx++; sy++;

  pic = (picture *) malloc (sizeof (picture));
  memset (pic, 0, sizeof (picture));
  pic->buffer = (uchar *) malloc (sx * sy + 1);
  pic->w = sx;
  pic->h = sy;
  fseek (f, 128, SEEK_SET);

  size = sx * sy + 1;

  while (p < size)
  {
    ch = fgetc (f);
    cn = 1;

    if (ch >= 192)        // if ch's bits: 11......
    {
      cn = ch & 0x3F;     // and with 00111111 to get count
      ch = fgetc (f);
    }

    for (i = 0; i < cn; i++)
    {
      if (p >= size) break;
      pic->buffer[p++] = ch;
    }
  }

  for (i = 0; i < 256; i++)
  {
    video.palette[i].r = fgetc (f) << 8;
    video.palette[i].g = fgetc (f) << 8;
    video.palette[i].b = fgetc (f) << 8;
  }

  fclose (f);

  /* shadow tint */
  for (i = 0; i < 32; i++)
  {
    video.palette[i | 32].r = video.palette[i].r / 2;
    video.palette[i | 32].g = video.palette[i].g / 2;
    video.palette[i | 32].b = video.palette[i].b / 2;
  }

  ggiSetPalette (video.visual, 0, 256, video.palette);

  return pic;
}

picture *make_buffer (picture *source, int x, int y, int w, int h, int multi)
{
  picture *p;
  int i;

  p = (picture *) malloc (sizeof (picture));
  memset (p, 0, sizeof (picture));
  p->w = w;
  p->h = h;

  for (i = 0; i < 4; i++)
  {
    if (!multi && i > 0) break;
    p->lbuf[i] = (ulong *) malloc (w * h + 4);
    memset (p->lbuf[i], 0, w * h + 4);
  }

  for (i = 0; i < h; i++)
  {
    memcpy (p->lbuf[0] + ((i * p->w) / 4), source->buffer + ((i + y) * source->w + x), w);
  }

  if (multi)
  {
    for (i = 1; i < 4; i++)
    {
      memcpy (((uchar *) p->lbuf[i]) + i, ((uchar *) p->lbuf[0]), w * h);
    }
  }
  else
  {
    p->buffer = (uchar *) p->lbuf[0];
  }

  return p;
}

void copy_pic (picture *pic, int x, int y)
{
  int i, j, vs, pofs = 0;
  int xs, ys, xe, ye;
  int pao; /* pic align offset */
  ulong *dest, *src;

  xs = x;
  if (xs < 0) xs = 0;
  xe = x + pic->w;
  if (xe > 320) xe = 320;
  
  ys = y;
  if (ys < 0) ys = 0;
  ye = y + pic->h;
  if (ye > 200) ye = 200;

  vs = ys * 320 + xs;
  pao = vs % 4;
  vs >>= 2;

  pofs -= (y - ys) * pic->w;
  pofs /= 4;

  src = (ulong *) &pic->lbuf[pao][pofs];
  dest = (ulong *) &video.lvscr[vs];

  for (j = ys; j < ye; j++)
  {
    src -= (x - xs) / 4;

    for (i = xs; i < xe; i += 4)
    {
      *dest++ |= *src++;
    }

    dest += (320 - (xe - xs)) >> 2;
    src += (x + pic->w - xe) >> 2;
  }
}

void copy_box (int x, int y, int sx, int sy, int sw, int sh, picture *pic)
{
  int i, j;
  int vs, pofs;
  int xs = 0, ys = 0, xe = sw, ye = sh;
  int pao; /* pic align offset */
  ulong *dest, *src;
  uchar col;
 
  xs = x;
  if (xs < 0) xs = 0;
  xe = x + sw;
  if (xe > 320) xe = 320;

  ys = y;
  if (ys < 0) ys = 0;
  ye = y + sh;
  if (ye > 200) ye = 200;

  vs = ys * 320 + xs;
  pao = vs % 4;
  vs >>= 2;

  pofs = sy * pic->w + sx;
  pofs -= (y - ys) * pic->w;
  pofs /= 4;

  src = (ulong *) &pic->lbuf[pao][pofs];
  dest = (ulong *) &video.lvscr[vs];

  for (j = ys; j < ye; j++)
  {
    src -= (x - xs) / 4;

    for (i = xs; i < xe; i += 4)
    {
      *dest++ |= *src++; // & 0xEFEFEFEF;
    }

    dest += (320 - sw) >> 2;
    src += ((pic->w - sw) + (x + sw - xe)) >> 2;
  }
}

void copy_shadow_box (int x, int y, int sx, int sy, int sw, int sh,
                      picture *pic)
{
  int i, j;
  int vs, pofs;
  int xs = 0, ys = 0, xe = sw, ye = sh;
  int pao; /* pic align offset */
  ulong *dest, *src;
  uchar col;
 
  xs = x;
  if (xs < 0) xs = 0;
  xe = x + sw;
  if (xe > 320) xe = 320;

  ys = y;
  if (ys < 0) ys = 0;
  ye = y + sh;
  if (ye > 200) ye = 200;

  vs = ys * 320 + xs;
  pao = vs % 4;
  vs >>= 2;

  pofs = sy * pic->w + sx;
  pofs -= (y - ys) * pic->w;
  pofs /= 4;

  src = (ulong *) &pic->lbuf[pao][pofs];
  dest = (ulong *) &video.lvscr[vs];

  for (j = ys; j < ye; j++)
  {
    src -= (x - xs) / 4;

    for (i = xs; i < xe; i += 4)
    {
      *dest++ |= (*src++ >> 1) & 0x20202020;
    }

    dest += (320 - sw) >> 2;
    src += ((pic->w - sw) + (x + sw - xe)) >> 2;
  }
}
