/*
 * $Id: scroller_state.cpp 150 2007-11-08 23:44:59Z ehaase $
 *
 * What We Are
 *
 * Copyright (C) 1994 - 2007 Enver Haase
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */



#include "scroller_state.hpp"

#include "main.hpp"
#include "audio/audio.h"
#include "bitmap/bitmap.hpp"
#include "timer.hpp"
#include "exceptions.hpp"
#include "keyboard.hpp"


// FUNCTIONS //////////////////////////////////////////////



scroller_state::scroller_state(void) : state() /* like Java's "super()" */
{
    this->isWhatWeAre = NULL;
    this->members = NULL;


    /* Load the OVERKILL logo bitmaps */
    char buf[3]; // temp buffer for itoa conversion
    for (int i=0; i<NUM_LOGO_FRAMES; i++){
        std::string name = "gfx_anim";
        int frame = i+1;
        if (frame<10)
            name += "0";
        _itoa_s(frame, buf, sizeof(buf), 10);
        name += buf;

        bitmaps[i] = new Bitmap(lpdd, name);
    }

    /* Load the IS_WHAT_WE_ARE bitmap */
    isWhatWeAre = new Bitmap(lpdd, std::string("gfx_iswhatweare"));

    /* Load the giant members bitmap */
    members = new Bitmap(lpdd, std::string("gfx_members"));

    // Create the colors used for color blinking
    int size = sizeof(members_blinker)/sizeof(members_blinker[0]);
    for (int i=0; i<size; i++)
    {
        members_blinker[i].peRed = 0;
        members_blinker[i].peGreen = i* (255/size); // 255 is max of 8-bit
        members_blinker[i].peBlue = 0;
        members_blinker[i].peFlags = PC_RESERVED;
    }



    // Create the rectangles used for blitting the secondary onto the primary
    // surface (which will be created below).
    members_source_rect.left    = members_dest_rect.left    = 0;
    members_source_rect.right   = members_dest_rect.right   = members->getWidth();
    //members_source_rect.top     = 0;
    //members_source_rect.bottom  = 0+MEMBERS_NUMLINES;
    members_dest_rect.top       = MEMBERS_FROMLINE;
    members_dest_rect.bottom    = MEMBERS_FROMLINE+MEMBERS_NUMLINES;
	
}

scroller_state::~scroller_state(void)
{
    if (members != NULL)
    {
        delete members;
    }
    if (isWhatWeAre != NULL)
    {
        delete isWhatWeAre;
    }
    for (int i=0; i<NUM_LOGO_FRAMES; i++)
    {
        if (bitmaps[i] != NULL)
        {
            delete bitmaps[i];
        }
    }

}







/**
 * Calculates what frame of the "Overkill" logo needs to be displayed
 * at a given time.
 */
unsigned int
scroller_state::overkillFrame(DWORD millisElapsed) throw (GameException)
{
    const unsigned int startingPause = 9600;    //msec
    const unsigned int animEvery = 40;          //msec
    const unsigned int animPause = 4000;        //msec
    unsigned int retVal = 0;

    if (millisElapsed <= startingPause)
    {
        retVal = 0;
    }
    else
    {
        millisElapsed = (millisElapsed-startingPause) % (2*((NUM_LOGO_FRAMES*animEvery)+animPause));

        unsigned int boundary;
        if (millisElapsed >= 0 && millisElapsed < (boundary= NUM_LOGO_FRAMES*animEvery))
        {
            retVal = millisElapsed / animEvery;
        }
        else if (millisElapsed >= boundary && millisElapsed < (boundary += animPause))
        {
            retVal = NUM_LOGO_FRAMES-1;
        }
        else if (millisElapsed >= boundary && millisElapsed < (boundary += NUM_LOGO_FRAMES*animEvery))
        {
            retVal = NUM_LOGO_FRAMES-1-((millisElapsed-(NUM_LOGO_FRAMES*animEvery+animPause))/animEvery);
        }
        else if (millisElapsed >= boundary && millisElapsed < (boundary += animPause))
        {
            retVal = 0;
        }
    }

    if (retVal >= NUM_LOGO_FRAMES)
    {
        char buf1[16];
        _itoa_s(retVal, buf1, sizeof(buf1), 10);
        char buf2[16];
        _itoa_s(millisElapsed, buf2, sizeof(buf2), 10);
        throwException((((std::string("Logo frame calculation internal error: frame ")+buf1)+", elapsed ")+buf2));
    }
    return retVal;
}


/**
 * -1 if nothing to display!
 * Else, "frame" to display, depending on current elapsed time.
 */
int
scroller_state::membersFrame(DWORD millisElapsed) throw (GameException)
{
    const unsigned int height = members->getHeight() - 150; // don't completely scroll out of the visible area
    const unsigned int startingPause = 8000;    //msec
    const unsigned int animEvery = 20;          //msec
    unsigned int retVal = 0;

    if (millisElapsed < startingPause)
    {
        return -1;
    }
    else
    {
        millisElapsed -= startingPause;
        millisElapsed = millisElapsed % (animEvery * height * 2);
        retVal = millisElapsed / animEvery; // now 0 <= retVal < height*2 
        if (retVal >= height)
        {
            retVal = height - (retVal-height) - 1;
        }
    }

    if (retVal >= height)
    {
        char buf1[16];
        _itoa_s(retVal,         buf1, sizeof(buf1), 10);
        char buf2[16];
        _itoa_s(millisElapsed,  buf2, sizeof(buf2), 10);
        char buf3[16];
        _itoa_s(height,         buf3, sizeof(buf3), 10);
        throwException(((((std::string("Members frame calculation internal error: frame ")+buf1)+", elapsed ")+buf2)+", height ")+buf3);
    }
    return retVal;
}

unsigned int
scroller_state::colorFrame(DWORD millisElapsed) throw (GameException)
{
    const unsigned int animEvery = 40;          //msec
    const unsigned int numColors = sizeof(members_blinker)/sizeof(members_blinker[0]);
    unsigned int retVal;

    retVal = millisElapsed % (animEvery * 2 * numColors);

    millisElapsed = millisElapsed % (animEvery * numColors * 2);
    retVal = millisElapsed / animEvery; // now 0 <= retVal < height*2 
    if (retVal >= numColors)
    {
        retVal = numColors - (retVal-numColors) - 1;
    }

    if (retVal >= numColors)
    {
        char buf1[16];
        _itoa_s(retVal,         buf1, sizeof(buf1), 10);
        char buf2[16];
        _itoa_s(millisElapsed,  buf2, sizeof(buf2), 10);
        char buf3[16];
        _itoa_s(numColors,      buf3, sizeof(buf3), 10);
        throwException(((((std::string("Colors frame calculation internal error: frame ")+buf1)+", elapsed ")+buf2)+", number of colors ")+buf3);
    }
    return retVal;

}

void
scroller_state::enterState(void) throw (GameException)
{	    
    // create and attach palette
    const int num_col = 1<<SCREEN_DEPTH;

    // create palette data
    PALETTEENTRY            palette[num_col];         // color palette
    
    // now create the palette object
    HRESULT createPal = lpdd->CreatePalette(DDPCAPS_8BIT | DDPCAPS_INITIALIZE | DDPCAPS_ALLOW256, palette, &lpddpal, NULL);
    if (createPal != DD_OK)
    {
        throwException("Could not create palette.", createPal);
    }

    // now load the palette entries
    HRESULT palsetent = lpddpal->SetEntries(0, 0, 256, bitmaps[0]->getPaletteEntries());
    if ( palsetent != DD_OK )
    {
        throwException("Cannot set palette entries.", palsetent);
    }
    if ( lpddpal->GetEntries(0, 0, 256, palette) != DD_OK)
    {
        throwException("Cannot get palette entries.");
    }
    for (int index=0; index<num_col; index++)
    {
        // set flags
        palette[index].peFlags = PC_NOCOLLAPSE;
    } // end for index
    palette[BLINK_COL_IDX].peFlags = PC_RESERVED;
    /* Funny: this color seems to be wrong in the bitmap
       file, or the bitmap reader is defective?! */
    palette[MEMBERS_BG_IDX].peRed = 0;
    palette[MEMBERS_BG_IDX].peGreen = 0;
    palette[MEMBERS_BG_IDX].peBlue = 0;
    lpddpal->SetEntries(0, 0, 256, palette);
    // attach the palette to the primary surface
    HRESULT setPal = lpddsprimary->SetPalette(lpddpal);
    if (setPal == DDERR_SURFACELOST)
    {
        lpddsprimary->Restore();
        setPal = lpddsprimary->SetPalette(lpddpal);
    }
    if (setPal != DD_OK)
    {
        throwException("Could not set palette.", setPal);
    }
    
    clearSurfaces();


    AResumeModule();
    this->overkill_timer.init(lastElapsed);    
}

void
scroller_state::exitState(void)
{
    APauseModule();	

    lastElapsed = overkill_timer.elapsed();
}


void
scroller_state::update(void) throw (GameException)
{	    
    // check if user is trying to exit
    if (keyPressed(VK_ESCAPE) || keyPressed(VK_SPACE) || keyPressed(VK_RETURN))
    {        
        PostMessage(main_window_handle, WM_APP, APPMSG_ADVANCE_STATE, APPMSG_ADVANCE_STATE);        
    }
    
	// Audio updates.
    AUpdateAudio();    

    // Video updates.
    unsigned int order = 0;
    unsigned int row = 0;
    static boolean firstTime = true;
    if (AGetModulePosition(&order, &row) == AUDIO_ERROR_NONE)
    {
        if ((order == 0) && (row < 110) && firstTime)
        {
            overkill_timer.init(8600);
        }
        else
        {
            firstTime = false;
        }
    }

    RECT ovkRect;
    ovkRect.left = 0;
    ovkRect.top  = 10;
    ovkRect.right = bitmaps[0]->getWidth();
    ovkRect.bottom = ovkRect.top + bitmaps[0]->getHeight();
    // copy image data
    bitmaps[overkillFrame(overkill_timer.elapsed())]->Blit(lpddsback, ovkRect);

    // "is what we are" in the middle.
    RECT whatweareRect;
    whatweareRect.left   = 0;
    whatweareRect.top    = ovkRect.bottom + 10;
    whatweareRect.right  = isWhatWeAre->getWidth();
    whatweareRect.bottom = isWhatWeAre->getHeight() + whatweareRect.top;
    isWhatWeAre->Blit(lpddsback, whatweareRect);
    
   
    int scrollOffset = membersFrame(overkill_timer.elapsed());
    // Prepare for Member-List blitting
    if (scrollOffset >= 0)
    {
        members_source_rect.top     = scrollOffset;
        members_source_rect.bottom  = scrollOffset+MEMBERS_NUMLINES;

        members->Blit(lpddsback, members_dest_rect, members_source_rect);
    }

    /* Colour blinking */
    lpddpal -> SetEntries(0, BLINK_COL_IDX, 1, &(members_blinker[colorFrame(overkill_timer.elapsed())]));

    /* Finally, flip backbuffer and primary buffer to see the newly generated frame */
    HRESULT flipResult = lpddsprimary->Flip(NULL, DDFLIP_WAIT);
    if (flipResult != DD_OK)
    {
        throwException("Flip problem?", flipResult);
    }
}
