/*  FreeJ
 *  (c) Copyright 2001 Denis Roio aka jaromil <jaromil@dyne.org>
 *
 * This source code is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Public License as published 
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * This source code is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * Please refer to the GNU Public License for more details.
 *
 * You should have received a copy of the GNU Public License along with
 * this source code; if not, write to:
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * "$Id: layer.h,v 1.28 2004/04/26 16:11:36 jaromil Exp $"
 *
 */

/**
   @file layer.h
   @brief FreeJ generic Layer interface
*/

#ifndef __LAYER_H__
#define __LAYER_H__

#include <inttypes.h>
#include <SDL.h>

#include <filter.h>
#include <jsync.h>

class Context;

/**
   This class describes methods and properties common to all Layers in
   FreeJ: it is the main interface for functionalities like blit
   changes, filter management and position changes.

   The public methods hereby described are matching the javascript API
   which is made available by the internal parser.

   Methods implemented to create and destroy a layer:
   - Layer::open
   - Layer::init
   - Layer::close
   
   Miscellaneus operations made available for the layer:
   - Layer::set_position
   - Layer::set_blit
   - Layer::get_blit
   - Layer::set_alpha
   - Layer::get_name

   LinkList of filters used on the layer:
   - Layer::filters

   Pointer to the initialized Context where the layer is used:
   - Layer::freej

   Geometrical informations about the layer:
   - Layer::geo
   
   @brief Layer parent abstract class
*/
class Layer: public Entry, public JSyncThread {
  friend class Context;
  friend class JSyncThread;

 public:

  Layer(); ///< Layer constructor
  ~Layer(); ///< Layer destructor
  
  /* these must be defined in layer implementations */
  virtual bool open(char *file) =0;
  virtual bool init(Context *scr) =0;
  virtual void close() =0;

  char *get_name();
  ///< Get Layer's descriptive name (3 letters)
    
  char *get_filename() { return filename; };
  ///< Get Layer's filename

  void set_position(int x, int y);
  ///< Set Layer's position on screen

  /* BLIT */
  void set_blit(int b) { blit = b; }; ///< Set Layer's blit algorithm
  uint8_t blit; ///< Layer's blit algorithm integer identifier
  char *get_blit(); ///< Get a short string describing Layer's blit algorithm


  /* ALPHA */
  void set_alpha(int opaq); ///< Set Layer's alpha opacity
  uint8_t alpha; ///< Layer's alpha opacity value
  uint8_t get_alpha() { return alpha; }; ///< Get Layer's alpha value

  uint8_t fade_in, fade_out;

  virtual bool keypress(SDL_keysym *keysym) =0;
  ///< pass to the Layer a keypress

  Linklist filters;
  ///< Filter list of effects applied on the Layer

  ScreenGeometry geo;
  ///< Geometrical information about the Layer

  bool active; ///< is active? (read-only)
  bool quit; ///< should it quit? (read-write)
  bool running; ///< is running? (read-only)
  bool hidden; ///< is hidden (read-only by the blit)
  int bgcolor; ///< matte background color

  SDL_Rect rect; ///< SDL rectangle for blit crop

  void *offset; ///< pointere to where all goes after processing


 protected:

  void _init(Context *freej, int wdt, int hgt, int bpp=0);
  ///< Layer abstract initialization

  Context *freej;

  void set_filename(char *f);
  char filename[256];

  void setname(char *s);

  void *buffer;


 private:
  char _name[5];
  char alphastr[5];

  void run(); ///< Main Layer thread loop

  virtual void *feed() = 0; ///< feeds in the image source

  bool cafudda(); ///< cafudda is called by the Context



  void *bgmatte;

};

/* function for type detection of implemented layers */

extern Layer *create_layer(char *file);
///< create the propriate Layer type from a file

extern const char *layers_description;

#endif
