#include <math.h>
#include "GL/gl.h"

#include "fadeeffect.h"
#include "traileffect.h"
#include "textures.h"
#include "textview.h"

#define TIMEOUT_NEXT 1500

#define SMOKE_PAUSE 2

#define DEG( x ) ( ( x ) * 3.1415926f / 180.f )

// Updates one smoke trail list
static void updateSmoke( TrailEffect::TrailList &list );

// Renders one smoke trail list
static void renderSmoke( TrailEffect::TrailList &list );

////////

TrailEffect::TrailEffect():
	myTimeout( 0 ), mySmokePause( 0 ), myEnding( false )
{
	glClearColor( 0.05f, 0.15f, 0.15f, 0.0f );

	textviewClear();

	// Add the trails
	for ( int i = 0; i < 8; ++i )
	{
		Trail trail;

		trail.texId = i % 4;
		trail.ang = i * 35;
		trail.angAdd = 0.1f + ( ( float ) i / 20.f );
		trail.x = trail.y = 0;
		trail.z = 10.f;

		myTrails.push_back( trail );
	}
}

TrailEffect::~TrailEffect()
{
}

// Updates the effect for one frame (returns the next effect; 0 (quit), this (continue) or a new effect)
Effect *TrailEffect::update()
{
	// Add smoke?
	bool addSmoke = false;
	--mySmokePause;
	if ( mySmokePause < 0 )
	{
		addSmoke = true;
		mySmokePause = SMOKE_PAUSE;
	}

	// Update the trails
	{
		TrailList::iterator pos = myTrails.begin();
		TrailList::iterator end = myTrails.end();

		for ( ; pos != end; ++pos )
		{
			Trail &trail = *pos;

			trail.ang += trail.angAdd;
			float a1 = DEG( trail.ang );
			float a2 = a1 * 2;
			float a3 = a2 * 5;

			trail.x = cosf( a1 ) * 0.2f + cosf( a2 ) * 0.3f + cosf( a3 ) * 0.5f;
			trail.y = sinf( a3 ) * 0.2f + sinf( a1 ) * 0.3f + sinf( a2 ) * 0.5f;

			// Add smoke?
			if ( addSmoke && !myEnding )
				mySmokes.push_back( trail );
		}
	}

	// Update the smokes
	updateSmoke( mySmokes );

	// Timeout?
	switch ( myTimeout++ )
	{
	case TIMEOUT_NEXT:
		myEnding = true;
		break;
	}

	// Ended?
	if ( myEnding &&
	     mySmokes.size() <= 0 )
	{
		return new FadeEffect( 0.05f, 0.15f, 0.15f,
		                       FadeEffect::Effect_Greet,
		                       0.3f, 0.4f, 0.5f,
		                       140 );
	}

	return this;
}

// Renders the effects current frame
void TrailEffect::render()
{
	glDisable( GL_LIGHTING );
	glEnable( GL_BLEND );
	glBlendFunc( GL_SRC_ALPHA, GL_ONE /*GL_ONE_MINUS_SRC_ALPHA*/ );

	// Render the smokes
	setTexture( textureTrail );
	renderSmoke( mySmokes );
}

////////

// Updates one smoke trail list
static void updateSmoke( TrailEffect::TrailList &list )
{
	TrailEffect::TrailList::iterator removePos = list.end();
	TrailEffect::TrailList::iterator pos = list.begin();
	TrailEffect::TrailList::iterator end = list.end();

	for ( ; pos != end; ++pos )
	{
		pos->z -= 0.1f;
		if ( pos->z <= 0.f )
			removePos = pos;
	}

	// Something to remove
	while ( removePos != end )
	{
		list.erase( removePos );

		// Find next to erase
		pos = list.begin();
		removePos = end = list.end();

		for ( ; pos != end; ++pos )
		{
			if ( pos->z <= 0.f )
			{
				removePos = pos;
				break;
			}
		}		
	}
}

// Renders one smoke trail list
static void renderSmoke( TrailEffect::TrailList &list )
{
	TrailEffect::TrailList::reverse_iterator pos = list.rbegin();
	TrailEffect::TrailList::reverse_iterator end = list.rend();

	const float size = 0.3f;

	if ( pos == end )
		return;

	glBegin( GL_QUADS );

	for ( ; pos != end; ++pos )
	{
		float x = pos->x;
		float y = pos->y;
		float z = -pos->z;

		float u0 = ( pos->texId < 2 ) ? 0.f : 0.5f;
		float v0 = ( pos->texId == 1 || pos->texId == 2 ) ? 0.f : 0.5f;
		float u1 = u0 + 0.5f;
		float v1 = v0 + 0.5f;

		glTexCoord2f( u0, v1 );
		glVertex3f( x - size, y + size, z );

		glTexCoord2f( u1, v1 );
		glVertex3f( x + size, y + size, z );

		glTexCoord2f( u1, v0 );
		glVertex3f( x + size, y - size, z );

		glTexCoord2f( u0, v0 );
		glVertex3f( x - size, y - size, z );
	}

	glEnd();
}
