/*
   tAAt 2001 graphical demonstration program
   Copyright 2000,2001 Jetro Lauha

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
  
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

/*
   The effect runs pretty slowly since the code isn't optimized.
   And since this code is made for this production and not for general
   usage, some parts are of course a bit kludgy in a reasonable way.
   If you use this source, extend it, optimize it, fix it or have any
   comments about it, please drop me an E-mail to jetro@iki.fi.
*/

#include <stdlib.h>
#include <math.h>
#include <limits.h>

#include "SDL.h"
#include "fmod.h"
#include "fmod_errors.h"

#include "fixfont.xpm"


#define FULLSCREEN

#define MUSICFILE	"t2001.mp3"

#define BANNERTEXT	"tAAt 2000  2001"

#define PI		3.1415926535897932

#define CHAR_XPMWIDTH	18
#define CHAR_XPMHEIGHT	font_height
#define CHARS		(font_width / CHAR_XPMWIDTH)
#define CHAR_WIDTH	(CHAR_XPMWIDTH + 2)
#define CHAR_HEIGHT	(CHAR_XPMHEIGHT + 2)
#define CHAR_FOREGROUND	0xff
#define CHAR_BACKGROUND	0
#define CHAR_OUTLINE	0x20

#define WIDTH		640
#define HEIGHT		480

#define FPS		30

#define EFFROWHEIGHT	35

#define MAXWRITERCHARS	100
#define CHARLIFETIME	7000
#define CHARMOVEAREA	100
#define CHARBIRTHDIST	1500
#define CHARROTSPEED	1500
#define CHARDROPSPEED	40

#define WRITERCHARDELAY	120
#define WRITERROWDELAY	1700

#define CLIP_TOP	0
#define CLIP_LEFT	0
#define CLIP_RIGHT	(WIDTH - 1)
#define CLIP_BOTTOM	(HEIGHT - 1)

#ifdef FULLSCREEN
#define FULLSCREENSETTING SDL_FULLSCREEN
#else
#define FULLSCREENSETTING 0
#endif

enum {
	DEAD,
	ALIVE,
	DYING,
} WRITERCHARSTATE;

typedef struct {
	int y;
	char *text;
} WRITERROW;

typedef struct {
	char state;
	int x, y, age;
	int tx, ty, r;
	char ch;
} WRITERCHAR;


WRITERROW writerrow[] = {
	{ 0, " " },
	{ 100, "New millennium has begun.." },
	{ 200, "..so as you probably expected," },
	{ 300, "tAAt 2000 changed it's name to" },
	{ 400, "tAAt 2001" },

	{ 100, "And this is the official first" },
	{ 200, "production of the year 2001." },
	{ 300, "Release timestamp:" },
	{ 400, "2001-01-01 00:00:00.0000" },

	{ 0, " " },
	{ 100, "So as you may have already" },
	{ 180, "noticed, this production is" },
        { 260, "released for both Linux and" },
	{ 340, "Windows. With source code." },
	{ 420, "(using GPL license)" },

	{ 0, "     " },
	{ 200, "Remember to visit tAAt web" },
	{ 300, "site regularly! Current URL is" },
	{ 400, "http://taat.cjb.net" },

	{ 0, " " },
	{ 100, "Also contact us if you are" },
	{ 200, "interested in joining tAAt." },
	{ 300, "We are still taking new active" },
	{ 400, "and talented members ;-D .." },

	{ 0, " " },
	{ 100, "That's it. We hope you had fun" },
	{ 200, "with this somewhat oldschool" },
	{ 300, "styled production.     " },

	{ 0, "             " },
	{ 440, "looping ..." },
};

#define WRITERROWS	(sizeof(writerrow) / sizeof(WRITERROW))


static WRITERCHAR *wrchars;

static SDL_Surface *screen;
static char quit = 0;
static Uint8 *chars[CHARS];

static FSOUND_STREAM *musicstream;




/* sets up the beautiful colors
*/
void setPal()
{
	SDL_Color c[256];
	int a, i;
	
	for (a = 0; a < 64; ++a) {
		i = a;
		c[i].r = a * 2;
		c[i].g = 0;
		c[i].b = a;
		i += 64;
		c[i].r = 128 + a;
		c[i].g = a;
		c[i].b = 64;
		i += 64;
		c[i].r = 192 + a;
		c[i].g = 64 + a;
		c[i].b = 64 + a * 2;
		i += 64;
		c[i].r = 255;
		c[i].g = 128 + a * 2;
		c[i].b = 192 + a;
	}
	SDL_SetColors(screen, c, 0, 256);
}


/* kludge font xpm conversion routine, supposed to work
*  only with the font provided with this production
*/
void buildChars()
{
	const int xa = (CHAR_WIDTH - CHAR_XPMWIDTH) / 2;
	const int ya = (CHAR_HEIGHT - CHAR_XPMHEIGHT) / 2;
	int a, ca, oc;
	int x, y;
	
	/* grab the chars */
	ca = 0;
	for (a = 0; a < CHARS; ++a) {
		chars[a] = (Uint8 *)calloc(CHAR_WIDTH * CHAR_HEIGHT, 1);
		for (y = 0; y < CHAR_XPMHEIGHT; ++y) {
			for (x = 0; x < CHAR_XPMWIDTH; ++x)
				chars[a][(y + ya) * CHAR_WIDTH + x + xa] =
				(font_pixels[y][x + ca] == 'a' ? CHAR_BACKGROUND : CHAR_FOREGROUND);
		}
		ca += CHAR_XPMWIDTH;
	}
	
#define CHAR_CHECK_OUTLINE_PIXEL(cnt, x, y) \
	if (x >= 0 && x < CHAR_WIDTH && \
	y >= 0 && y < CHAR_HEIGHT && \
	chars[a][(y) * CHAR_WIDTH + x] == CHAR_FOREGROUND) \
	++(*cnt);
	
	/* create dark outline */
	for (a = 0; a < CHARS; ++a) {
		for (y = 0; y < CHAR_HEIGHT; ++y) {
			for (x = 0; x < CHAR_WIDTH; ++x) {
				oc = 0;
				CHAR_CHECK_OUTLINE_PIXEL(&oc, x - 1, y - 1);
				CHAR_CHECK_OUTLINE_PIXEL(&oc, x, y - 1);
				CHAR_CHECK_OUTLINE_PIXEL(&oc, x + 1, y - 1);
				CHAR_CHECK_OUTLINE_PIXEL(&oc, x - 1, y);
				CHAR_CHECK_OUTLINE_PIXEL(&oc, x + 1, y);
				CHAR_CHECK_OUTLINE_PIXEL(&oc, x - 1, y + 1);
				CHAR_CHECK_OUTLINE_PIXEL(&oc, x, y + 1);
				CHAR_CHECK_OUTLINE_PIXEL(&oc, x - 1, y + 1);
				if (oc && chars[a][y * CHAR_WIDTH + x] != CHAR_FOREGROUND)
					chars[a][y * CHAR_WIDTH + x] = CHAR_OUTLINE;
			}
		}
	}
	
#undef CHAR_CHECK_OUTLINE_PIXEL
	
}


/* frees the font character memory
*/
void freeChars()
{
	int a;
	for (a = 0; a < CHARS; ++a)
		free(chars[a]);
}


/* draws one character (clipped against the screen boundaries)
*/
void drawChar(Uint8 *dst, int ch, int x, int y)
{
	Uint8 *src;
	int a, b, w, h;
	
	if (ch < 32 || ch >= 32 + CHARS ||
		x >= WIDTH || y >= HEIGHT ||
		x <= -CHAR_WIDTH || y <= -CHAR_HEIGHT)
		return;
	
	src = chars[ch - 32];
	
	w = CHAR_WIDTH;
	h = CHAR_HEIGHT;
	
	if (x < CLIP_LEFT) {
		w -= CLIP_LEFT - x;
		src += CLIP_LEFT - x;
		x = CLIP_LEFT;
	}
	if (y < CLIP_TOP) {
		h -= CLIP_TOP - y;
		src += (CLIP_TOP - y) * CHAR_WIDTH;
		y = CLIP_TOP;
	}
	if (x + w - 1 > CLIP_RIGHT) {
		w -= x + w - CLIP_RIGHT - 1;
	}
	if (y + h - 1 > CLIP_BOTTOM) {
		h -= y + h - CLIP_BOTTOM - 1;
	}
	
	dst += y * WIDTH + x;
	
	for (b = 0; b < h; ++b) {
		for (a = 0; a < w; ++a) {
			if (*src)
				*dst = *src;
			++dst;
			++src;
		}
		dst += WIDTH - w;
		src += CHAR_WIDTH - w;
	}
}


/* draws a centered string
*/
void drawString(Uint8 *dst, Uint8 *str, int mx, int my)
{
	int l = strlen(str);
	int x = mx - l * CHAR_WIDTH / 2;
	int y = my - CHAR_HEIGHT / 2;
	
	for (; *str; ++str, x += CHAR_WIDTH)
		drawChar(dst, *str, x, y);
}


/* SDL event filter
*/
int filterEvents(const SDL_Event *event)
{
	if (event->type == SDL_KEYDOWN) {
		Uint8 *keys = SDL_GetKeyState(NULL);
		
		if (keys[SDLK_ESCAPE] == SDL_PRESSED) {
			quit = 1;
			return 0;
		}
	}
	return 1;
}


/* initialize music system
*/
void initMusic()
{
	if (FSOUND_GetVersion() < FMOD_VERSION) {
		fprintf(stderr, "Error : You are using the wrong DLL version!  You should be using FMOD %.02f\n", FMOD_VERSION);
		exit(EXIT_FAILURE);
	}
	
	FSOUND_SetBufferSize(100);
	
	if (!FSOUND_Init(44100, 2, 0)) {
		fprintf(stderr, "%s\n", FMOD_ErrorString(FSOUND_GetError()));
		exit(EXIT_FAILURE);
	}
	
	musicstream = FSOUND_Stream_OpenFile(MUSICFILE, FSOUND_NORMAL | FSOUND_LOOP_NORMAL, 0);
	if (!musicstream) {
		fprintf(stderr, "Error!\n%s\n", FMOD_ErrorString(FSOUND_GetError()));
		FSOUND_Close();
		exit(EXIT_FAILURE);
	}
}


/* begin to play the music
*/
int playMusic()
{
	int channel;
	if ((channel = FSOUND_Stream_Play(FSOUND_FREE, musicstream)) == -1) {
		fprintf(stderr, "Error!\n%s\n", FMOD_ErrorString(FSOUND_GetError()));
		FSOUND_Close();
		return 0;
	}
	FSOUND_SetPan(channel, FSOUND_STEREOPAN);
	return 1;
}


/* deinitialize music system
*/
void deinitMusic()
{
	FSOUND_Stream_Close(musicstream);
	FSOUND_Close();
}


/* initialize writer data
*/
void initWriter()
{
	wrchars = (WRITERCHAR *)calloc(MAXWRITERCHARS, sizeof(WRITERCHAR));
}


/* deinitialize writer
*/
void deinitWriter()
{
	free(wrchars);
}


/* draw the sine effect (simple and slow)
*/
void drawSineEffect(int time, Uint8 *buf)
{
	int q;
	float r, cx, cy;
	int x, y, o, w;

	q = (int)(time / (1000 / (float)FPS));
	r = (float)(
		sin(q * 0.050) * 0.1 + 0.1 +
		cos(q * 0.075) * 0.1 + 0.1 +
		sin(q * 0.035) * 0.1 + 0.1 +
		cos(q * 0.056) * 0.1 * 
		sin(q * 0.096)
		+ 0.1
		)/4.0f;
	cx = (float)sin(q * 0.005) * (WIDTH / 3) - WIDTH / 2;
	cy = (float)sin(q * 0.05) * (EFFROWHEIGHT / 2) - EFFROWHEIGHT / 2;
	
	o = WIDTH * HEIGHT;
	w = (time - 100) / 10;
	if (w < 0) w = 0;
	if (w > WIDTH) w = WIDTH;
	for (y = 0 - EFFROWHEIGHT / 2; y < EFFROWHEIGHT - EFFROWHEIGHT / 2; ++y) {
		for (x = 0; x < w; ++x) {
			buf[o++] = (int)((sin((x + cx) * r) +
				cos((y + cy) * r) ) * 63 + 64 );
		}
		o += WIDTH - w;
	}
}


/* draw moving top banner text
*/
void drawBanner(int time, Uint8 *buf)
{
	if (time > 15000)
		drawString(buf, BANNERTEXT,
			WIDTH / 2 + 500 - (time / 10) % 1000 + (int)(cos(time * PI / 900) * 15),
			50 + (int)(sin(time * PI / 800) * 10));
}

/* draw the text writer
*/
void drawWriter(int time, Uint8 *buf)
{
	static int lasttime = 0;
	static int currentrow = 0, currentcol = 0;
	static int nextchartime = 300, nextpos = 0;
	int a, x, y, m;
	float f;

	if (time >= nextchartime) {

		/* spawn new character */
		wrchars[nextpos].age = 0;
		wrchars[nextpos].state = ALIVE;
		wrchars[nextpos].r = currentrow;
		wrchars[nextpos].y = writerrow[currentrow].y;
		wrchars[nextpos].x = currentcol * CHAR_WIDTH + 
			(WIDTH - strlen(writerrow[currentrow].text) * CHAR_WIDTH) / 2;
		wrchars[nextpos].ch = writerrow[currentrow].text[currentcol];

		nextpos = (nextpos + 1) % MAXWRITERCHARS;

		/* determine next character */
		nextchartime += WRITERCHARDELAY;
		++currentcol;
		if (currentcol >= (int)strlen(writerrow[currentrow].text)) {
			nextchartime += WRITERROWDELAY;
			currentcol = 0;
			currentrow = (currentrow + 1) % WRITERROWS;
		}
	}

	for (a = 0; a < MAXWRITERCHARS; ++a) {
		if (wrchars[a].state != DEAD) {
			wrchars[a].age += time - lasttime;
			if (wrchars[a].age < CHARLIFETIME) {
				/* move character */
				m = wrchars[a].age < CHARBIRTHDIST ?
					CHARMOVEAREA + CHARBIRTHDIST - wrchars[a].age :
					CHARMOVEAREA;
				m /= 10;
				f = (time + a * (30 + wrchars[a].r * 10)) * (float)PI;
				x = wrchars[a].x + (int)(cos(f / CHARROTSPEED) * m);
				y = wrchars[a].y + (int)(sin(f / CHARROTSPEED) * m + m);
				wrchars[a].tx = x;
				wrchars[a].ty = y;
			} else {
				if (wrchars[a].state != DYING) {
					/* drop character */
					wrchars[a].x = wrchars[a].tx;
					wrchars[a].y = wrchars[a].ty;
					wrchars[a].ty = 0;
					wrchars[a].state = DYING;
				}
				/* drop effect */
				wrchars[a].ty += time - lasttime;
				wrchars[a].y += wrchars[a].ty / CHARDROPSPEED;
				if (wrchars[a].y >= HEIGHT)
					wrchars[a].state = DEAD;
				x = wrchars[a].x;
				y = wrchars[a].y;
			}
			drawChar(buf, wrchars[a].ch, x, y);
		}
	}

	lasttime = time;
}


/* perform a simple smooth operation
*/
void drawSmooth(int time, Uint8 *buf)
{
	int x, y;

	buf += WIDTH * 2;

	for (y = 2; y < HEIGHT + EFFROWHEIGHT; ++y) {
		for (x = 0; x < WIDTH / 4; ++x) {
			*(Uint32 *)buf =
				((*(Uint32 *)buf & 0xfcfcfcfc) >> 2) +
				((*(Uint32 *)(buf - 639) & 0xfcfcfcfc) >> 2) +
				((*(Uint32 *)(buf - 641) & 0xfcfcfcfc) >> 2) +
				((*(Uint32 *)(buf + EFFROWHEIGHT * WIDTH) & 0xfcfcfcfc) >> 2);
			buf += 4;
		}
	}
}


/* update screen from offscreen graphics buffer
*/
int updateScreen(Uint8 *buf)
{
	if (SDL_MUSTLOCK(screen)) {
		if (SDL_LockSurface(screen) < 0)
			return EXIT_FAILURE;
	}
	
	memcpy(screen->pixels, buf, WIDTH * HEIGHT);
	
	if (SDL_MUSTLOCK(screen))
		SDL_UnlockSurface(screen);
	
	SDL_UpdateRect(screen, 0, 0, WIDTH, HEIGHT);

	return EXIT_SUCCESS;
}


/* the main effect loop
*/
int mainEffect()
{
	int ret, time, starttime = SDL_GetTicks();
	Uint8 *buf = (Uint8 *)calloc(WIDTH * (HEIGHT + EFFROWHEIGHT * 2), sizeof(Uint8));

	ret = EXIT_SUCCESS;

	while (!SDL_QuitRequested() && !quit && ret == EXIT_SUCCESS) {
	
		time = SDL_GetTicks() - starttime;
		
		drawSineEffect(time, buf);
		drawBanner(time, buf);
		drawWriter(time, buf);
		drawSmooth(time, buf);

		ret = updateScreen(buf);
		
	}

	return ret;	
}


int main(int argc, char *argv[])
{
	int ret = EXIT_FAILURE;
	
	/* kludge to prevent a compiler warning message */
	font_colors[0] = font_colors[0];
	
	if (SDL_Init(SDL_INIT_AUDIO|SDL_INIT_VIDEO|SDL_INIT_TIMER) < 0) {
		fprintf(stderr, "Unable to init SDL: %s\n", SDL_GetError());
		exit(EXIT_FAILURE);
	}
	atexit(SDL_Quit);
	
	initMusic();
	
	screen = SDL_SetVideoMode(WIDTH, HEIGHT, 8, SDL_HWSURFACE | SDL_HWPALETTE | FULLSCREENSETTING);
	if (screen == NULL) {
		fprintf(stderr, "Unable to set %dx%d mode: %s\n",
			WIDTH, HEIGHT, SDL_GetError());
		exit(1);
	}
	
	setPal();
	buildChars();
	initWriter();
	
	SDL_SetEventFilter(filterEvents);
	
	if (playMusic())
		ret = mainEffect();
	
	deinitWriter();
	deinitMusic();
	
	freeChars();
	
	return ret;
}
