#include "groundplane.hpp"

Vector3 GroundPlane::fadeColor = Vector3(0,0,0);

GroundPlane::GroundPlane(int xsize, int ysize, float size, float texturemul, float ypos )
{
    int x = 0;
    int y = 0;
    this->xres = xsize;
    this->yres = ysize;

    this->hasFade = false;
    this->gridSize = size;

    int vertexcount = this->xres * this->yres;
    this->vertices = new Vector3[vertexcount];
    this->uv = new TTexCoord[vertexcount];

    Vector3 v1 = Vector3(-size, ypos, -size);
    Vector3 v2 = Vector3( size, ypos, -size);
    Vector3 v3 = Vector3( size, ypos,  size);
    Vector3 v4 = Vector3(-size, ypos,  size);

    for (y = 0; y < this->yres; y++)
    {
        float yt = y / (float)(this->yres - 1.0f);
        for (x = 0; x < this->xres; x++)
        {
            float xt = x / (float)(this->xres - 1.0f);
            int offset = x + y * this->xres;

            Vector3 vasen = v1 + (v4 - v1) * yt;
            Vector3 oikea = v2 + (v3 - v2) * yt;
            Vector3 piste = vasen + (oikea - vasen) * xt;

            this->vertices[offset] = piste;
            this->uv[offset].u = xt * texturemul;
            this->uv[offset].v = yt * texturemul;

            offset++;
        }
    }
}

void GroundPlane::makeCircularFade(float brightness, float maxdistmultiplier)
{
    int x, y;
    int vertexcount = this->xres * this->yres;
    
    this->fade = new float[vertexcount];

    int offset = 0;
    for (y = 0; y < this->yres; y++)
    {
        for (x = 0; x < this->xres; x++)
        {
            const float maxdist = this->gridSize * maxdistmultiplier;
            float bright = 0.0f;

            Vector3 piste = this->vertices[offset];
            float d = piste.length();

            if (d <= maxdist)
            {
                bright = 1.0f - d / maxdist;
            }
            else
            {
                bright = 0.0f;
            }

            bright = powf(bright, brightness);
            this->fade[offset] = bright;
            offset++;
        }
    }
    this->hasFade = true;
}


void GroundPlane::renderWithNormalMaps(Texture *texture, Texture *normalmap, Vector3 light, float alpha)
{
    int x, y;
	
	Shader *ss = shaders->getShader("ground");
	ss->bind(); 

	glActiveTextureARB(GL_TEXTURE0_ARB);
    glEnable(GL_TEXTURE_2D);
    glBindTexture(GL_TEXTURE_2D, texture->getID());

	glActiveTextureARB(GL_TEXTURE1_ARB);
    glEnable(GL_TEXTURE_2D);    
    glBindTexture(GL_TEXTURE_2D, normalmap->getID());
    
	ss->setUniform1i("texture0", 0);  
	ss->setUniform1i("normal0", 1);  
	ss->setUniform4f("minColor", fadeColor.x, fadeColor.y, fadeColor.z, alpha);  

	glDisable(GL_BLEND);

	float g_li[4] = {light.x, light.y, light.z,0};
	glPushAttrib(GL_LIGHTING_BIT);
    glLightfv(GL_LIGHT0, GL_POSITION, g_li); 

    if (this->hasFade)
    {
        for (y = 0; y < this->yres - 1; y++)
        {
            int offs = y * this->xres;

            glBegin(GL_QUAD_STRIP);
            for (x = 0; x < this->xres; x++)
            {
                float a1 = alpha * this->fade[offs];
                float a2 = alpha * this->fade[offs + this->xres];
                glColor3f(a1, a1, a1);
                glTexCoord2fv((float *)&this->uv[offs]);
                glVertex3fv((float *)&this->vertices[offs]);
                glColor3f(a2, a2, a2);
                glTexCoord2fv((float *)&this->uv[offs + this->xres]);
                glVertex3fv((float *)&this->vertices[offs + this->xres]);
                offs++;
            }
            glEnd();
        }
    }
    else
    {
        glEnable(GL_TEXTURE_2D);
        glDisable(GL_BLEND);
        glBindTexture(GL_TEXTURE_2D, texture->getID());

        glColor3f(alpha, alpha, alpha);

        for (y = 0; y < this->yres - 1; y++)
        {
            int offs = y * this->xres;

            glBegin(GL_QUAD_STRIP);
            for (x = 0; x < this->xres; x++)
            {
                glTexCoord2fv((float *)&this->uv[offs]);
                glVertex3fv((float *)&this->vertices[offs]);
                glTexCoord2fv((float *)&this->uv[offs + this->xres]);
                glVertex3fv((float *)&this->vertices[offs + this->xres]);
                offs++;
            }
            glEnd();

        }

    }

	glPopAttrib();

	ss->unbind();

	glDisable(GL_TEXTURE_2D);
	glActiveTextureARB(GL_TEXTURE1_ARB);
	glDisable(GL_TEXTURE_2D);

	glDisable(GL_TEXTURE_2D);
	glActiveTextureARB(GL_TEXTURE0_ARB);
	glDisable(GL_TEXTURE_2D);

}

void GroundPlane::render(Texture *texture, float alpha)
{
    int x, y;

    if (this->hasFade)
    {
        glEnable(GL_TEXTURE_2D);
        glDisable(GL_BLEND);
        glBindTexture(GL_TEXTURE_2D, texture->getID());

        for (y = 0; y < this->yres - 1; y++)
        {
            int offs = y * this->xres;

            glBegin(GL_QUAD_STRIP);
            for (x = 0; x < this->xres; x++)
            {
                float a1 = alpha * this->fade[offs];
                float a2 = alpha * this->fade[offs + this->xres];
                glColor3f(a1, a1, a1);
                glTexCoord2fv((float *)&this->uv[offs]);
                glVertex3fv((float *)&this->vertices[offs]);
                glColor3f(a2, a2, a2);
                glTexCoord2fv((float *)&this->uv[offs + this->xres]);
                glVertex3fv((float *)&this->vertices[offs + this->xres]);
                offs++;
            }
            glEnd();
        }
    }
    else
    {
        glEnable(GL_TEXTURE_2D);
        glDisable(GL_BLEND);
        glBindTexture(GL_TEXTURE_2D, texture->getID());

        glColor3f(alpha, alpha, alpha);

        for (y = 0; y < this->yres - 1; y++)
        {
            int offs = y * this->xres;

            glBegin(GL_QUAD_STRIP);
            for (x = 0; x < this->xres; x++)
            {
                glTexCoord2fv((float *)&this->uv[offs]);
                glVertex3fv((float *)&this->vertices[offs]);
                glTexCoord2fv((float *)&this->uv[offs + this->xres]);
                glVertex3fv((float *)&this->vertices[offs + this->xres]);
                offs++;
            }
            glEnd();

        }

    }

}

void GroundPlane::enableFade()
{
    this->hasFade = true;
}


float GroundPlane::safeGet(int x, int y)
{
    if (x >= 0 && x < this->xres && y >= 0 && y < this->yres)
    {
        return this->fade[x + y * this->xres];
    }
    else
    {
        return 0.0f;
    }

}
void GroundPlane::safePut(int x, int y, float value)
{
    if (x >= 0 && x < this->xres && y >= 0 && y < this->yres)
    {
        this->fade[x + y * this->xres] = value;
    }
}


void GroundPlane::setupFade()
{
    int count = this->xres * this->yres;
    this->fade = new float [count];
    for (int i = 0; i < count; i++)
    {
        this->fade[i] = 0.0f;
    }
}

float *GroundPlane::getFade()
{
    return this->fade;
}

Vector3 *GroundPlane::getVertices()
{
	return this->vertices;
}
TTexCoord *GroundPlane::getUVs()
{
	return this->uv;
}
int GroundPlane::getXres()
{
	return this->xres;
}
int GroundPlane::getYres()
{
	return this->yres;
}
