// Cubes.cpp: implementation of the CCubes class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "Cubes.h"

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

VOID SetupPod( PTLVERTEX pVertices, FLOAT fScale, FLOAT fRotation, DWORD dwAlpha )
{
	he3d_CVector			vUV[6];
	he3d_CMatrix			mtx = ZRotationMtx( fRotation )*ScaleMtx( fScale );
	he3d_CVector			v;
	DWORD					dwColor = (dwAlpha<<24);

	vUV[0] = he3d_CVector( -0.5f, -0.5f, 0.0f );
	vUV[1] = he3d_CVector( -0.5f,  0.5f, 0.0f );
	vUV[2] = he3d_CVector(  0.5f, -0.5f, 0.0f );

	vUV[3] = he3d_CVector(  0.5f, -0.5f, 0.0f );
	vUV[4] = he3d_CVector( -0.5f,  0.5f, 0.0f );
	vUV[5] = he3d_CVector(  0.5f,  0.5f, 0.0f );

	for( DWORD i = 0 ; i < 6 ; i++ )
	{
		v = vUV[i]*mtx;

		pVertices[i].m_fU = v.x;
		pVertices[i].m_fV = v.y;
		pVertices[i].m_dwColor = dwColor;
	}
}

VOID CreateDotCube( PSIMPLEVERTEX pVertices, he3d_CVector& vCenter, DWORD dwCount )
{
	for( DWORD i = 0 ; i < dwCount ; i++ )
		for( DWORD j = 0 ; j < dwCount ; j++ )
		{		
			*(pVertices++) = SIMPLEVERTEX( he3d_CVector( -0.5f + (FLOAT)i/(FLOAT)(dwCount-1), -0.5f + (FLOAT)j/(FLOAT)(dwCount-1), -0.5f ) + vCenter, 0x80505050 );
			*(pVertices++) = SIMPLEVERTEX( he3d_CVector( -0.5f + (FLOAT)i/(FLOAT)(dwCount-1), -0.5f + (FLOAT)j/(FLOAT)(dwCount-1), 0.5f ) + vCenter, 0x80505050 );
		}

	for( i = 0 ; i < dwCount ; i++ )
		for( DWORD j = 1 ; j < dwCount - 1 ; j++ )
		{		
			*(pVertices++) = SIMPLEVERTEX( he3d_CVector( -0.5f + (FLOAT)i/(FLOAT)(dwCount-1), 0.5f, -0.5f + (FLOAT)j/(FLOAT)(dwCount-1) ) + vCenter, 0x80505050 );
			*(pVertices++) = SIMPLEVERTEX( he3d_CVector( -0.5f + (FLOAT)i/(FLOAT)(dwCount-1), -0.5f, -0.5f + (FLOAT)j/(FLOAT)(dwCount-1) ) + vCenter, 0x80505050 );
		}

	for( i = 1 ; i < dwCount-1 ; i++ )
		for( DWORD j = 1 ; j < dwCount - 1 ; j++ )
		{		
			*(pVertices++) = SIMPLEVERTEX( he3d_CVector( 0.5f, -0.5f + (FLOAT)i/(FLOAT)(dwCount-1), -0.5f + (FLOAT)j/(FLOAT)(dwCount-1) ) + vCenter, 0x80505050 );
			*(pVertices++) = SIMPLEVERTEX( he3d_CVector( -0.5f, -0.5f + (FLOAT)i/(FLOAT)(dwCount-1), -0.5f + (FLOAT)j/(FLOAT)(dwCount-1) ) + vCenter, 0x80505050 );
		}
}

CCubes::CCubes( PDIRECT3DDEVICE8 pDevice ) : CEfx( pDevice ), 
											 m_pvbCube( NULL ), 
											 m_pvbCube2( NULL ),
											 m_pvbDotCube( NULL ),
											 m_pvbFxPod( NULL ), 
											 m_piqEfx( NULL ), 
											 m_piqDisplay( NULL ), 
											 m_piqNoise( NULL )
{
	try
	{
		LoadAlphaTexture( "border.bmp" );
		LoadAlphaTexture( "model.bmp" );
		LoadAlphaTexture( "model2.bmp" );
		LoadTexture( "metal.jpg" );
		LoadAlphaTexture( "fx.bmp" );
		LoadAlphaTexture( "display.bmp" );
		LoadAlphaTexture( "display2.bmp" );
	}
	catch( CTextureException )	
	{	
		throw CSystemException( "unable to load textures" );	
	}

	if( FAILED( m_pDevice->CreateVertexBuffer( 36*64*sizeof(LVERTEX), 0, FVF_LVERTEX, D3DPOOL_DEFAULT, &m_pvbCube ) ) )	
		throw CSystemException( "unable to create cube2 vertex buffer" );	

	if( FAILED( m_pDevice->CreateVertexBuffer( 36*64*sizeof(LVERTEX), 0, FVF_LVERTEX, D3DPOOL_DEFAULT, &m_pvbCube2 ) ) )	
		throw CSystemException( "unable to create cube2 vertex buffer" );	

	if( FAILED( m_pDevice->CreateVertexBuffer( 296*64*sizeof(SIMPLEVERTEX), 0, FVF_SIMPLEVERTEX, D3DPOOL_DEFAULT, &m_pvbDotCube ) ) )	
		throw CSystemException( "unable to create dot cube vertex buffer" );	

	if( FAILED( m_pDevice->CreateVertexBuffer( PODCOUNT*6*sizeof(TLVERTEX), 0, FVF_TLVERTEX, D3DPOOL_DEFAULT, &m_pvbFxPod ) ) )	
		throw CSystemException( "unable to creatre fxpod vertex buffer" );	
	
	if( FAILED( m_pDevice->CreateVertexBuffer( 4*6*sizeof(TLMULTITEXVERTEX), 0, FVF_TLMULTITEXVERTEX, D3DPOOL_DEFAULT, &m_pvbSyncPod ) ) )	
		throw CSystemException( "unable to creatre syncpod vertex buffer" );	

	m_piqEfx = new CImageQuad( m_pDevice );
	m_piqDisplay = new CImageQuad( m_pDevice );
	m_piqNoise = new CImageQuad( m_pDevice );
	m_piqFade = new CImageQuad( m_pDevice );

	PTLVERTEX				pV;
	D3DVIEWPORT8			d3dvp;

	m_pDevice->GetViewport( &d3dvp );

	m_pvbFxPod->Lock( 0, 0, (LPBYTE*)&pV, 0 );

	for( DWORD i = 0 ; i < PODCOUNT ; i++, pV += 6 )
	{			
		pV[0] = TLVERTEX( 0.0f, (FLOAT)d3dvp.Height, 0.0f, 0.5f, 0x00ffffff,  0.0f, 0.0f );
		pV[1] = TLVERTEX( 0.0f,   0.0f, 0.0f, 0.5f, 0x00ffffff,  0.0f, 0.0f );
		pV[2] = TLVERTEX( (FLOAT)d3dvp.Width, (FLOAT)d3dvp.Height, 0.0f, 0.5f, 0x00ffffff,  0.0f, 0.0f );

		pV[3] = TLVERTEX( (FLOAT)d3dvp.Width, (FLOAT)d3dvp.Height, 0.0f, 0.5f, 0x00ffffff,  0.0f, 0.0f );
		pV[4] = TLVERTEX( 0.0f, 0.0f, 0.0f, 0.5f, 0x00ffffff,  0.0f, 0.0f );
		pV[5] = TLVERTEX( (FLOAT)d3dvp.Width, 0.0f, 0.0f, 0.5f, 0x00ffffff,  0.0f, 0.0f );
				
	}

	m_pvbFxPod->Unlock();

	PSIMPLEVERTEX			pVerts;
	PLVERTEX				pVertices;
	DWORD					index = 0;

	for( i = 0 ; i < 4 ; i++ )
		for( DWORD j = 0 ; j < 4 ; j++ )
			for( DWORD k = 0 ; k < 4 ; k++ )
			{
				m_vPosition[index].x = ((FLOAT)k - 1.5f)*2.0f;
				m_vPosition[index].y = ((FLOAT)j - 1.5f)*2.0f;
				m_vPosition[index].z = ((FLOAT)i - 1.5f)*2.0f;				

				index++;
			}

	m_pvbCube->Lock( 0, 0, (LPBYTE*)&pVertices, 0 );

	for( i = 0 ; i < 64 ; i++, pVertices += 36 )	
		CreateCube( pVertices, 1.0f, m_vPosition[i], 0x00000000 );	

	m_pvbCube->Unlock();

	m_pvbCube2->Lock( 0, 0, (LPBYTE*)&pVertices, 0 );

	for( i = 0 ; i < 64 ; i++, pVertices += 36 )	
		CreateCube( pVertices, 1.0f, m_vPosition[i], 0xffffffff );	

	m_pvbCube2->Unlock();

	m_pvbDotCube->Lock( 0, 0, (LPBYTE*)&pVerts, 0 );

	for( i = 0 ; i < 64 ; i++, pVerts += 296 )	
		CreateDotCube( pVerts, m_vPosition[i], 8 );	

	m_pvbDotCube->Unlock();

	m_bRenderSolid = FALSE;
	m_fShift = -1.0f;		

	for( i = 0 ; i < PODCOUNT ; i++ )
	{	
		m_fRotationSpeed[i] = 0.5f + 0.5f*RAND();
		m_fOffset[i] = H_PI*RAND();

		m_fAlpha[i] = 1.0f;
		m_fScale[i] = (FLOAT)(PODCOUNT - i)/(FLOAT)PODCOUNT;
	}

	m_fAlpha[0] = 0.0f;
	m_dwCurrentPod = PODCOUNT - 1;
	m_fSwapTime = 0.0f;

	m_bRenderDisplay = FALSE;
	m_piqDisplay->SetColor( 0x0 );
	m_piqNoise->Resize( 0, 0, 0, 0 );

	m_bRenderFade = FALSE;

	// sync spline init

	m_fSync[0][ 0] = FLOATKEY(  0.00f, 1.00f );
	m_fSync[0][ 1] = FLOATKEY(  0.20f, 1.00f );
	m_fSync[0][ 2] = FLOATKEY(  1.46f, 1.00f );
	m_fSync[0][ 3] = FLOATKEY(  1.66f, 0.20f );
	m_fSync[0][ 4] = FLOATKEY(  1.85f, 1.00f );
	m_fSync[0][ 5] = FLOATKEY(  2.04f, 0.30f );
	m_fSync[0][ 6] = FLOATKEY(  2.24f, 1.00f );
	m_fSync[0][ 7] = FLOATKEY(  3.45f, 1.00f );
	m_fSync[0][ 8] = FLOATKEY(  3.65f, 0.20f );
	m_fSync[0][ 9] = FLOATKEY(  3.76f, 0.70f );
	m_fSync[0][10] = FLOATKEY(  3.86f, 0.50f );
	m_fSync[0][11] = FLOATKEY(  4.06f, 1.00f );
	m_fSync[0][12] = FLOATKEY(  4.57f, 1.00f );
	m_fSync[0][13] = FLOATKEY(  4.77f, 0.00f );
	m_fSync[0][14] = FLOATKEY(  4.89f, 0.70f );
	m_fSync[0][15] = FLOATKEY(  5.01f, 0.20f );
	m_fSync[0][16] = FLOATKEY(  5.21f, 1.00f );
	m_fSync[0][17] = FLOATKEY(  5.69f, 1.00f );
	m_fSync[0][18] = FLOATKEY(  5.89f, 0.20f );
	m_fSync[0][19] = FLOATKEY(  6.09f, 1.00f );
	m_fSync[0][20] = FLOATKEY(  6.81f, 1.00f );
	m_fSync[0][21] = FLOATKEY(  7.01f, 0.20f );
	m_fSync[0][22] = FLOATKEY(  7.21f, 1.00f );
	m_fSync[0][23] = FLOATKEY(  7.71f, 1.00f );
	m_fSync[0][24] = FLOATKEY(  7.91f, 0.30f );
	m_fSync[0][25] = FLOATKEY(  8.11f, 1.00f );
	m_fSync[0][26] = FLOATKEY(  9.97f, 1.00f );
	m_fSync[0][27] = FLOATKEY( 10.17f, 0.10f );
	m_fSync[0][28] = FLOATKEY( 10.29f, 0.60f );
	m_fSync[0][29] = FLOATKEY( 10.41f, 0.30f );
	m_fSync[0][30] = FLOATKEY( 10.61f, 1.00f );
	m_fSync[0][31] = FLOATKEY( 11.12f, 1.00f );
	m_fSync[0][32] = FLOATKEY( 11.32f, 0.30f );
	m_fSync[0][33] = FLOATKEY( 11.52f, 1.00f );
	m_fSync[0][34] = FLOATKEY( 12.23f, 1.00f );
	m_fSync[0][35] = FLOATKEY( 12.43f, 0.20f );
	m_fSync[0][36] = FLOATKEY( 12.63f, 1.00f );
	m_fSync[0][37] = FLOATKEY( 13.55f, 1.00f );
	m_fSync[0][38] = FLOATKEY( 13.75f, 0.10f );
	m_fSync[0][39] = FLOATKEY( 13.89f, 1.00f );
	m_fSync[0][40] = FLOATKEY( 14.02f, 0.20f );
	m_fSync[0][41] = FLOATKEY( 14.22f, 1.00f );
	m_fSync[0][42] = FLOATKEY( 14.91f, 1.00f );
	m_fSync[0][43] = FLOATKEY( 15.11f, 0.30f );
	m_fSync[0][44] = FLOATKEY( 15.31f, 1.00f );
	m_fSync[0][45] = FLOATKEY( 15.82f, 1.00f );
	m_fSync[0][46] = FLOATKEY( 16.02f, 0.40f );
	m_fSync[0][47] = FLOATKEY( 16.22f, 1.00f );
	m_fSync[0][48] = FLOATKEY( 17.15f, 1.00f );
	m_fSync[0][49] = FLOATKEY( 17.35f, 0.40f );
	m_fSync[0][50] = FLOATKEY( 17.55f, 1.00f );
	m_fSync[0][51] = FLOATKEY( 18.56f, 1.00f );
	m_fSync[0][52] = FLOATKEY( 18.76f, 0.10f );
	m_fSync[0][53] = FLOATKEY( 18.96f, 1.00f );
	m_fSync[0][54] = FLOATKEY( 21.44f, 1.00f );
	m_fSync[0][55] = FLOATKEY( 21.64f, 0.00f );
	m_fSync[0][56] = FLOATKEY( 21.75f, 1.00f );
	m_fSync[0][57] = FLOATKEY( 21.85f, 0.20f );
	m_fSync[0][58] = FLOATKEY( 22.05f, 1.00f );
	m_fSync[0][59] = FLOATKEY( 22.10f, 1.00f );
	m_fSync[0][60] = FLOATKEY( 22.30f, 0.10f );
	m_fSync[0][61] = FLOATKEY( 22.50f, 1.00f );

	InitFloatSpline( 62, m_fSync[0] );

	m_fSync[1][ 0] = FLOATKEY(  0.00f, 1.00f );
	m_fSync[1][ 1] = FLOATKEY(  0.18f, 1.00f );
	m_fSync[1][ 2] = FLOATKEY(  0.37f, 0.20f );
	m_fSync[1][ 3] = FLOATKEY(  0.57f, 1.00f );
	m_fSync[1][ 4] = FLOATKEY(  2.34f, 1.00f );
	m_fSync[1][ 5] = FLOATKEY(  2.54f, 0.10f );
	m_fSync[1][ 6] = FLOATKEY(  2.74f, 1.00f );
	m_fSync[1][ 7] = FLOATKEY(  4.14f, 1.00f );
	m_fSync[1][ 8] = FLOATKEY(  4.34f, 0.40f );
	m_fSync[1][ 9] = FLOATKEY(  4.54f, 1.00f );
	m_fSync[1][10] = FLOATKEY(  5.24f, 1.00f );
	m_fSync[1][11] = FLOATKEY(  5.44f, 0.10f );
	m_fSync[1][12] = FLOATKEY(  5.55f, 0.80f );
	m_fSync[1][13] = FLOATKEY(  5.67f, 0.30f );
	m_fSync[1][14] = FLOATKEY(  5.87f, 1.00f );
	m_fSync[1][15] = FLOATKEY(  6.38f, 1.00f );
	m_fSync[1][16] = FLOATKEY(  6.58f, 0.00f );
	m_fSync[1][17] = FLOATKEY(  6.67f, 0.70f );
	m_fSync[1][18] = FLOATKEY(  6.76f, 0.20f );
	m_fSync[1][19] = FLOATKEY(  6.96f, 1.00f );
	m_fSync[1][20] = FLOATKEY(  7.49f, 1.00f );
	m_fSync[1][21] = FLOATKEY(  7.69f, 0.20f );
	m_fSync[1][22] = FLOATKEY(  7.89f, 1.00f );
	m_fSync[1][23] = FLOATKEY(  9.49f, 1.00f );
	m_fSync[1][24] = FLOATKEY(  9.69f, 0.00f );
	m_fSync[1][25] = FLOATKEY(  9.89f, 1.00f );
	m_fSync[1][26] = FLOATKEY( 16.70f, 1.00f );
	m_fSync[1][27] = FLOATKEY( 16.90f, 0.10f );
	m_fSync[1][28] = FLOATKEY( 17.10f, 1.00f );
	m_fSync[1][29] = FLOATKEY( 17.58f, 1.00f );
	m_fSync[1][30] = FLOATKEY( 17.78f, 0.40f );
	m_fSync[1][31] = FLOATKEY( 17.98f, 1.00f );
	m_fSync[1][32] = FLOATKEY( 19.67f, 1.00f );
	m_fSync[1][33] = FLOATKEY( 19.87f, 0.20f );
	m_fSync[1][34] = FLOATKEY( 20.00f, 1.00f );

	InitFloatSpline( 35, m_fSync[1] );

	m_fSync[2][ 0] = FLOATKEY(  0.00f, 1.00f );
	m_fSync[2][ 1] = FLOATKEY(  0.20f, 1.00f );
	m_fSync[2][ 2] = FLOATKEY(  0.95f, 1.00f );
	m_fSync[2][ 3] = FLOATKEY(  1.15f, 0.20f );
	m_fSync[2][ 4] = FLOATKEY(  1.20f, 0.80f );
	m_fSync[2][ 5] = FLOATKEY(  1.26f, 0.10f );
	m_fSync[2][ 6] = FLOATKEY(  1.46f, 1.00f );
	m_fSync[2][ 7] = FLOATKEY(  2.77f, 1.00f );
	m_fSync[2][ 8] = FLOATKEY(  2.97f, 0.20f );
	m_fSync[2][ 9] = FLOATKEY(  3.08f, 1.00f );
	m_fSync[2][10] = FLOATKEY(  3.20f, 0.30f );
	m_fSync[2][11] = FLOATKEY(  3.40f, 1.00f );
	m_fSync[2][12] = FLOATKEY(  8.63f, 1.00f );
	m_fSync[2][13] = FLOATKEY(  8.83f, 0.10f );
	m_fSync[2][14] = FLOATKEY(  9.03f, 1.00f );
	m_fSync[2][15] = FLOATKEY(  9.28f, 1.00f );
	m_fSync[2][16] = FLOATKEY(  9.48f, 0.00f );
	m_fSync[2][17] = FLOATKEY(  9.68f, 1.00f );
	m_fSync[2][18] = FLOATKEY( 10.46f, 1.00f );
	m_fSync[2][19] = FLOATKEY( 10.66f, 0.40f );
	m_fSync[2][20] = FLOATKEY( 10.86f, 1.00f );
	m_fSync[2][21] = FLOATKEY( 11.81f, 1.00f );
	m_fSync[2][22] = FLOATKEY( 12.01f, 0.30f );
	m_fSync[2][23] = FLOATKEY( 12.10f, 0.70f );
	m_fSync[2][24] = FLOATKEY( 12.20f, 0.10f );
	m_fSync[2][25] = FLOATKEY( 12.40f, 1.00f );
	m_fSync[2][26] = FLOATKEY( 12.91f, 1.00f );
	m_fSync[2][27] = FLOATKEY( 13.11f, 0.40f );
	m_fSync[2][28] = FLOATKEY( 13.31f, 1.00f );
	m_fSync[2][29] = FLOATKEY( 14.25f, 1.00f );
	m_fSync[2][30] = FLOATKEY( 14.45f, 0.30f );
	m_fSync[2][31] = FLOATKEY( 14.57f, 1.00f );
	m_fSync[2][32] = FLOATKEY( 14.70f, 0.20f );
	m_fSync[2][33] = FLOATKEY( 14.90f, 1.00f );
	m_fSync[2][34] = FLOATKEY( 15.36f, 1.00f );
	m_fSync[2][35] = FLOATKEY( 15.56f, 0.50f );
	m_fSync[2][36] = FLOATKEY( 15.76f, 1.00f );
	m_fSync[2][37] = FLOATKEY( 16.15f, 1.00f );
	m_fSync[2][38] = FLOATKEY( 16.35f, 0.10f );
	m_fSync[2][39] = FLOATKEY( 16.45f, 1.00f );
	m_fSync[2][40] = FLOATKEY( 16.56f, 0.20f );
	m_fSync[2][41] = FLOATKEY( 16.76f, 1.00f );
	m_fSync[2][42] = FLOATKEY( 17.84f, 1.00f );
	m_fSync[2][43] = FLOATKEY( 18.04f, 0.10f );
	m_fSync[2][44] = FLOATKEY( 18.16f, 0.80f );
	m_fSync[2][45] = FLOATKEY( 18.28f, 0.40f );
	m_fSync[2][46] = FLOATKEY( 18.48f, 1.00f );
	m_fSync[2][47] = FLOATKEY( 19.24f, 1.00f );
	m_fSync[2][48] = FLOATKEY( 19.44f, 0.20f );
	m_fSync[2][49] = FLOATKEY( 19.64f, 1.00f );
	m_fSync[2][50] = FLOATKEY( 20.30f, 1.00f );
	m_fSync[2][51] = FLOATKEY( 20.50f, 0.40f );
	m_fSync[2][52] = FLOATKEY( 20.70f, 1.00f );
	m_fSync[2][53] = FLOATKEY( 21.04f, 1.00f );
	m_fSync[2][54] = FLOATKEY( 21.24f, 0.10f );
	m_fSync[2][55] = FLOATKEY( 21.50f, 1.00f );

	InitFloatSpline( 56, m_fSync[2] );

	m_fSync[3][ 0] = FLOATKEY(  0.00f, 1.00f );
	m_fSync[3][ 1] = FLOATKEY(  0.12f, 1.00f );
	m_fSync[3][ 2] = FLOATKEY(  0.24f, 0.20f );
	m_fSync[3][ 3] = FLOATKEY(  0.44f, 1.00f );
	m_fSync[3][ 4] = FLOATKEY(  3.93f, 1.00f );
	m_fSync[3][ 5] = FLOATKEY(  4.13f, 0.40f );
	m_fSync[3][ 6] = FLOATKEY(  4.33f, 1.00f );
	m_fSync[3][ 7] = FLOATKEY(  5.02f, 1.00f );
	m_fSync[3][ 8] = FLOATKEY(  5.22f, 0.25f );
	m_fSync[3][ 9] = FLOATKEY(  5.42f, 1.00f );
	m_fSync[3][10] = FLOATKEY(  5.92f, 1.00f );
	m_fSync[3][11] = FLOATKEY(  6.12f, 0.10f );
	m_fSync[3][12] = FLOATKEY(  6.32f, 1.00f );
	m_fSync[3][13] = FLOATKEY(  7.03f, 1.00f );
	m_fSync[3][14] = FLOATKEY(  7.23f, 0.40f );
	m_fSync[3][15] = FLOATKEY(  7.35f, 0.80f );
	m_fSync[3][16] = FLOATKEY(  7.47f, 0.20f );
	m_fSync[3][17] = FLOATKEY(  7.67f, 1.00f );
	m_fSync[3][18] = FLOATKEY(  8.20f, 1.00f );
	m_fSync[3][19] = FLOATKEY(  8.40f, 0.10f );
	m_fSync[3][20] = FLOATKEY(  8.50f, 1.00f );
	m_fSync[3][21] = FLOATKEY(  8.60f, 0.00f );
	m_fSync[3][22] = FLOATKEY(  8.80f, 1.00f );
	m_fSync[3][23] = FLOATKEY( 16.53f, 1.00f );
	m_fSync[3][24] = FLOATKEY( 16.74f, 0.50f );
	m_fSync[3][25] = FLOATKEY( 16.94f, 1.00f );
	m_fSync[3][26] = FLOATKEY( 19.44f, 1.00f );
	m_fSync[3][27] = FLOATKEY( 19.64f, 0.40f );
	m_fSync[3][28] = FLOATKEY( 19.84f, 1.00f );
	m_fSync[3][29] = FLOATKEY( 20.56f, 1.00f );
	m_fSync[3][30] = FLOATKEY( 20.76f, 0.10f );
	m_fSync[3][31] = FLOATKEY( 20.86f, 1.00f );
	m_fSync[3][32] = FLOATKEY( 20.96f, 0.50f );
	m_fSync[3][33] = FLOATKEY( 21.20f, 1.00f );

	InitFloatSpline( 34, m_fSync[3] );

	m_fSync[4][ 0] = FLOATKEY(  0.00f, 1.00f );
	m_fSync[4][ 1] = FLOATKEY(  0.20f, 1.00f );
	m_fSync[4][ 2] = FLOATKEY(  1.33f, 1.00f );
	m_fSync[4][ 3] = FLOATKEY(  1.53f, 0.50f );
	m_fSync[4][ 4] = FLOATKEY(  1.73f, 1.00f );
	m_fSync[4][ 5] = FLOATKEY(  3.20f, 1.00f );
	m_fSync[4][ 6] = FLOATKEY(  3.40f, 0.20f );
	m_fSync[4][ 7] = FLOATKEY(  3.60f, 1.00f );
	m_fSync[4][ 8] = FLOATKEY(  8.85f, 1.00f );
	m_fSync[4][ 9] = FLOATKEY(  9.05f, 0.10f );
	m_fSync[4][10] = FLOATKEY(  9.16f, 1.00f );
	m_fSync[4][11] = FLOATKEY(  9.26f, 0.50f );
	m_fSync[4][12] = FLOATKEY(  9.46f, 1.00f );
	m_fSync[4][13] = FLOATKEY( 11.28f, 1.00f );
	m_fSync[4][14] = FLOATKEY( 11.48f, 0.20f );
	m_fSync[4][15] = FLOATKEY( 11.68f, 1.00f );
	m_fSync[4][16] = FLOATKEY( 12.44f, 1.00f );
	m_fSync[4][17] = FLOATKEY( 12.64f, 0.00f );
	m_fSync[4][18] = FLOATKEY( 12.76f, 1.00f );
	m_fSync[4][19] = FLOATKEY( 12.88f, 0.50f );
	m_fSync[4][20] = FLOATKEY( 13.07f, 1.00f );
	m_fSync[4][21] = FLOATKEY( 14.03f, 1.00f );
	m_fSync[4][22] = FLOATKEY( 14.23f, 0.10f );
	m_fSync[4][23] = FLOATKEY( 14.43f, 1.00f );
	m_fSync[4][24] = FLOATKEY( 14.70f, 1.00f );
	m_fSync[4][25] = FLOATKEY( 14.90f, 0.10f );
	m_fSync[4][26] = FLOATKEY( 15.10f, 1.00f );
	m_fSync[4][27] = FLOATKEY( 15.66f, 1.00f );
	m_fSync[4][28] = FLOATKEY( 15.86f, 0.40f );
	m_fSync[4][29] = FLOATKEY( 16.06f, 1.00f );
	m_fSync[4][30] = FLOATKEY( 17.53f, 1.00f );
	m_fSync[4][31] = FLOATKEY( 17.73f, 0.20f );
	m_fSync[4][32] = FLOATKEY( 17.93f, 1.00f );
	m_fSync[4][33] = FLOATKEY( 18.97f, 1.00f );
	m_fSync[4][34] = FLOATKEY( 19.17f, 0.25f );
	m_fSync[4][35] = FLOATKEY( 19.37f, 1.00f );
	m_fSync[4][36] = FLOATKEY( 21.89f, 1.00f );
	m_fSync[4][37] = FLOATKEY( 22.09f, 0.10f );
	m_fSync[4][38] = FLOATKEY( 22.30f, 1.00f );

	InitFloatSpline( 39, m_fSync[4] );

	m_fSync[5][ 0] = FLOATKEY(  0.00f, 1.00f );
	m_fSync[5][ 1] = FLOATKEY(  0.20f, 1.00f );
	m_fSync[5][ 2] = FLOATKEY(  0.38f, 1.00f );
	m_fSync[5][ 3] = FLOATKEY(  0.58f, 0.50f );
	m_fSync[5][ 4] = FLOATKEY(  0.78f, 1.00f );
	m_fSync[5][ 5] = FLOATKEY(  2.12f, 1.00f );
	m_fSync[5][ 6] = FLOATKEY(  2.32f, 0.20f );
	m_fSync[5][ 7] = FLOATKEY(  2.52f, 1.00f );
	m_fSync[5][ 8] = FLOATKEY( 10.70f, 1.00f );
	m_fSync[5][ 9] = FLOATKEY( 10.90f, 0.20f );
	m_fSync[5][10] = FLOATKEY( 10.98f, 0.60f );
	m_fSync[5][11] = FLOATKEY( 11.06f, 0.20f );
	m_fSync[5][12] = FLOATKEY( 11.26f, 1.00f );
	m_fSync[5][13] = FLOATKEY( 13.15f, 1.00f );
	m_fSync[5][14] = FLOATKEY( 13.35f, 0.20f );
	m_fSync[5][15] = FLOATKEY( 13.55f, 1.00f );
	m_fSync[5][16] = FLOATKEY( 18.32f, 1.00f );
	m_fSync[5][17] = FLOATKEY( 18.52f, 0.20f );
	m_fSync[5][18] = FLOATKEY( 18.72f, 1.00f );
	m_fSync[5][19] = FLOATKEY( 19.86f, 1.00f );
	m_fSync[5][20] = FLOATKEY( 20.06f, 0.20f );
	m_fSync[5][21] = FLOATKEY( 20.17f, 1.00f );
	m_fSync[5][22] = FLOATKEY( 20.28f, 0.50f );
	m_fSync[5][23] = FLOATKEY( 20.50f, 1.00f );

	InitFloatSpline( 24, m_fSync[5] );

	m_dwSplineCount[0] = 62;
	m_dwSplineCount[1] = 35;
	m_dwSplineCount[2] = 56;
	m_dwSplineCount[3] = 34;
	m_dwSplineCount[4] = 39;
	m_dwSplineCount[5] = 24;

	for( i = 0 ; i < 64 ; i++ )
		m_dwCubesIndices[i] = rand()%6;
}

CCubes::~CCubes()
{
	SAFE_RELEASE( m_pvbCube );
	SAFE_RELEASE( m_pvbCube2 );
	SAFE_RELEASE( m_pvbDotCube );
	SAFE_RELEASE( m_pvbFxPod );

	SAFE_DELETE( m_piqEfx );		
	SAFE_DELETE( m_piqDisplay );
	SAFE_DELETE( m_piqNoise );
	SAFE_DELETE( m_piqFade );
}

BOOL CCubes::UpdateFrame( FLOAT fTime )
{
	// standard setup

	m_pDevice->SetTransform( D3DTS_PROJECTION, ProjectionMtx( 30.0f, 0.75f, 1.0f, 20.0f ) );
	m_pDevice->SetTransform( D3DTS_VIEW, CameraMtx( he3d_CVector( 0, 0, -10 ), he3d_CVector( 0, 0, 3 ), 0 ) );

	m_fRotation = fTime*0.5f;	

	if( fTime >= 3.0f )
	{
		m_bRenderSolid = TRUE;
		
		m_fShift = -0.25f*(7.0f - fTime);

		if( m_fShift > 0.0f )
			m_fShift = 0.0f;
	}

	FLOAT					fT = fTime - m_fSwapTime;
	DWORD					dwIndex;

	m_fAlpha[m_dwCurrentPod] = 1.0f - fT;
	m_fAlpha[(m_dwCurrentPod+1)%PODCOUNT] = fT;

	if( m_fAlpha[m_dwCurrentPod] <= 0.0f )
	{
		m_fSwapTime = fTime;
		fT = 0.0f;
		m_fAlpha[m_dwCurrentPod] = 0.0f;
		m_fAlpha[(m_dwCurrentPod+1)%PODCOUNT] = 1.0f;		

		m_fRotationSpeed[m_dwCurrentPod] = 0.5f + 0.5f*RAND();
		m_fOffset[m_dwCurrentPod] = H_PI*RAND();

		if( m_dwCurrentPod == 0 )
			m_dwCurrentPod = PODCOUNT - 1;
		else
			m_dwCurrentPod = m_dwCurrentPod - 1;
	}
	
	dwIndex = (m_dwCurrentPod + 1)%PODCOUNT;

	for( DWORD i = 0 ; i < PODCOUNT ; i++, dwIndex = (dwIndex + 1)%PODCOUNT )
	{
		m_fScale[dwIndex] = 1.0f - 0.33f*( fT + (FLOAT)i );
		m_fPodRotation[dwIndex] = 0.25f*m_fRotationSpeed[dwIndex]*fTime + m_fOffset[dwIndex];
	}	

	PTLVERTEX				pVertices;

	m_pvbFxPod->Lock( 0, 0, (LPBYTE*)&pVertices, 0 );

	for( i = 0 ; i < PODCOUNT ; i++, pVertices += 6 )	
		SetupPod( pVertices, m_fScale[i], m_fPodRotation[i], (DWORD)(m_fAlpha[i]*ALPHAFACTOR ) );	

	m_pvbFxPod->Unlock();

	m_piqEfx->Lock( (LPBYTE*)&pVertices );

	pVertices[0].m_fV = m_fRotation*0.25f;
	pVertices[1].m_fV = m_fRotation*0.25f;
	pVertices[2].m_fV = 1.0f + m_fRotation*0.25f;

	pVertices[3].m_fV = 1.0f + m_fRotation*0.25f;
	pVertices[4].m_fV = m_fRotation*0.25f;
	pVertices[5].m_fV = 1.0f + m_fRotation*0.25f;	

	pVertices[0].m_fU = m_fShift;
	pVertices[1].m_fU = 1.0f + m_fShift;
	pVertices[2].m_fU = m_fShift;

	pVertices[3].m_fU = m_fShift;
	pVertices[4].m_fU = 1.0f + m_fShift;
	pVertices[5].m_fU = 1.0f + m_fShift;

	m_piqEfx->Unlock();

	for( i = 0 ; i < 6 ; i++ )
	{	
		FLOAT					fTmp;

		fTmp = GetFloatSplineValue( fTime, m_dwSplineCount[i], m_fSync[i] );

		if( fTmp < 0.0f )
			fTmp = 0.0f;
		else if( fTmp > 1.0f )
			fTmp = 1.0f;

		fTmp = ( 1.0f - fTmp )*255.0f;

		m_dwColor[i] = D3DCOLOR_XRGB( (DWORD)fTmp, (DWORD)fTmp, (DWORD)fTmp );
	}


	PLVERTEX				pV;

	m_pvbCube->Lock( 0, 0, (LPBYTE*)&pV, 0 );

	for( i = 0 ; i < 64 ; i++, pV += 36 )
		for( DWORD j = 0 ; j < 36 ; j++ )
			pV[j].m_dwColor = m_dwColor[m_dwCubesIndices[i]];

	m_pvbCube->Unlock();

	FLOAT					fAlpha;

	if( fTime > 7.5f )
	{
		fAlpha = (fTime - 7.5f)*85.0f;

		if( fAlpha > 255.0f )
			fAlpha = 255.0f;

		m_bRenderDisplay = TRUE;

		m_piqDisplay->SetAlpha( (DWORD)fAlpha );

		fAlpha = ( 10.5f - fTime )*50.0f;

		if( fAlpha < 0.0f )
			fAlpha = 0.0f;

		m_piqDisplay->Resize( 150.0f - 128.0f - fAlpha, 200.0f - 128.0f - fAlpha, 150.0f + 128.0f + fAlpha, 200.0f + 128.0f + fAlpha );

		m_dwDispAlpha = 0x0;

		if( fTime > 10.5f )		
		{		
			fAlpha = 128.0f*( fTime - 10.5f );

			if( fAlpha > 255.0f )
				fAlpha = 255.0f;
			else if( fAlpha < 0.0f )
				fAlpha = 0.0f;

			m_dwDispAlpha = (DWORD)fAlpha;

			if( fTime > 10.75 )
			{			
				fAlpha = (fTime - 10.75f)*160.0f + 1.0f;

				if( fAlpha > 80.0f )
					fAlpha = 80.0f;

				m_piqNoise->Resize( 40.0f, 200.0f - fAlpha, 260.0f, 200.0f + fAlpha );
			}
			else
			{
				fAlpha = (fTime - 10.5f)*220.0f;

				if( fAlpha > 110.f )
					fAlpha = 110.0f;

				m_piqNoise->Resize( 150.0f - fAlpha, 199.0f, 150.0f + fAlpha, 201.0f );
			}
		}
	}

	if( fTime > 18.0f )
	{
		m_bRenderFade = TRUE;
		
		fAlpha = (fTime - 18.0f)*140.0f;

		if( fAlpha > 255.0f )
			fAlpha = 255.0f;

		m_piqFade->SetAlpha( (DWORD)fAlpha );
	}

	return TRUE;
}

BOOL CCubes::RenderEfx()
{		
	m_pDevice->SetRenderState( D3DRS_LIGHTING, FALSE );	

	if( m_bRenderDisplay )
	{
		m_pDevice->SetRenderState( D3DRS_ZENABLE, FALSE );
		m_pDevice->SetRenderState( D3DRS_CULLMODE, D3DCULL_NONE );
		m_pDevice->SetRenderState( D3DRS_STENCILENABLE, TRUE );
		m_pDevice->SetRenderState( D3DRS_STENCILFUNC, D3DCMP_ALWAYS );
		m_pDevice->SetRenderState( D3DRS_STENCILPASS, D3DSTENCILOP_REPLACE );
		m_pDevice->SetRenderState( D3DRS_STENCILREF, 0x1 );

		m_piqNoise->Render( m_pDevice );

		m_pDevice->SetRenderState( D3DRS_STENCILENABLE, FALSE );
		m_pDevice->SetRenderState( D3DRS_STENCILPASS, D3DSTENCILOP_KEEP );
		m_pDevice->SetRenderState( D3DRS_STENCILREF, 0x0 );
	}

	if( m_bRenderSolid )
	{	
		m_pDevice->SetRenderState( D3DRS_ZENABLE, FALSE );
		m_pDevice->SetRenderState( D3DRS_CULLMODE, D3DCULL_NONE );

		m_pDevice->SetRenderState( D3DRS_ALPHABLENDENABLE, TRUE );
		m_pDevice->SetRenderState( D3DRS_SRCBLEND, D3DBLEND_SRCALPHA );
		m_pDevice->SetRenderState( D3DRS_DESTBLEND, D3DBLEND_INVSRCALPHA );
		m_pDevice->SetTexture( 0, GetTexture( "fx.bmp" ) );
		m_pDevice->SetTextureStageState( 0, D3DTSS_COLOROP, D3DTOP_SELECTARG2 );
		m_pDevice->SetTextureStageState( 0, D3DTSS_ALPHAOP, D3DTOP_MODULATE );
	
		m_pDevice->SetVertexShader( FVF_TLVERTEX );		
		m_pDevice->SetStreamSource( 0, m_pvbFxPod, sizeof(TLVERTEX) );	
		m_pDevice->DrawPrimitive( D3DPT_TRIANGLELIST, 0, 2*PODCOUNT );	
				
		m_pDevice->SetTextureStageState( 0, D3DTSS_ALPHAOP, D3DTOP_SELECTARG1 );	
		m_pDevice->SetTextureStageState( 0, D3DTSS_ADDRESSU, D3DTADDRESS_CLAMP );

		if( m_bRenderDisplay )
		{		
			m_pDevice->SetRenderState( D3DRS_STENCILENABLE, TRUE );
			m_pDevice->SetRenderState( D3DRS_STENCILFUNC, D3DCMP_NOTEQUAL );
			m_pDevice->SetRenderState( D3DRS_STENCILREF, 0x1 );			
		}

		m_pDevice->SetTexture( 0, GetTexture( "model.bmp" ) );
		
		m_piqEfx->Render( m_pDevice );

		m_pDevice->SetTextureStageState( 0, D3DTSS_COLOROP, D3DTOP_MODULATE );
		m_pDevice->SetTextureStageState( 0, D3DTSS_ADDRESSU, D3DTADDRESS_WRAP );

		m_pDevice->SetRenderState( D3DRS_ALPHABLENDENABLE, FALSE );

		if( m_bRenderDisplay )
		{		
			m_pDevice->SetRenderState( D3DRS_STENCILENABLE, FALSE );
			m_pDevice->SetRenderState( D3DRS_STENCILFUNC, D3DCMP_ALWAYS );
			m_pDevice->SetRenderState( D3DRS_STENCILREF, 0x0 );
		}
	}
	
	// solid	
	m_pDevice->SetRenderState( D3DRS_ZENABLE, D3DZB_TRUE );
	m_pDevice->SetRenderState( D3DRS_CULLMODE, D3DCULL_CCW );	

	m_pDevice->SetRenderState( D3DRS_ALPHABLENDENABLE, TRUE );
	m_pDevice->SetRenderState( D3DRS_SRCBLEND, D3DBLEND_SRCALPHA );
	m_pDevice->SetRenderState( D3DRS_DESTBLEND, D3DBLEND_INVSRCALPHA );			
	
	m_pDevice->SetTexture( 0, GetTexture( "metal.jpg" ) );
	m_pDevice->SetTextureStageState( 0, D3DTSS_COLOROP, D3DTOP_SELECTARG1 );

	m_pDevice->SetTexture( 1, GetTexture( "model2.bmp" ) );
	m_pDevice->SetTextureStageState( 1, D3DTSS_COLOROP, D3DTOP_SELECTARG2 );
	m_pDevice->SetTextureStageState( 1, D3DTSS_ALPHAOP, D3DTOP_SELECTARG1 );
	m_pDevice->SetTextureStageState( 1, D3DTSS_TEXCOORDINDEX, D3DTSS_TCI_CAMERASPACEPOSITION );
	m_pDevice->SetTextureStageState( 1, D3DTSS_TEXTURETRANSFORMFLAGS, D3DTTFF_COUNT4 | D3DTTFF_PROJECTED );	
	m_pDevice->SetTextureStageState( 1, D3DTSS_ADDRESSU, D3DTADDRESS_CLAMP );
	
	if( m_bRenderSolid )
	{	
		m_pDevice->SetVertexShader( FVF_LVERTEX );
		m_pDevice->SetStreamSource( 0, m_pvbCube2, sizeof(LVERTEX) );
	
		m_pDevice->SetTransform( D3DTS_TEXTURE1, ProjectionMtx( 30.0f, 0.75f, 1.0f, 20.0f )*TranslationMtx( 1.0f, 1.0f, 0.0f )*ScaleMtx( 0.5f, -0.5f, 1.0f )*TranslationMtx( m_fShift, m_fRotation*0.25f, 0.0f ) );	
		m_pDevice->SetTransform( D3DTS_WORLD, RotationMtx( m_fRotation, -m_fRotation, 0.5f*m_fRotation ) );
	
		m_pDevice->DrawPrimitive( D3DPT_TRIANGLELIST, 0, 12*64 );
	}
	
	m_pDevice->SetRenderState( D3DRS_CLIPPLANEENABLE, FALSE );

	// transparent	
	
	m_pDevice->SetRenderState( D3DRS_ZENABLE, D3DZB_FALSE );
	m_pDevice->SetRenderState( D3DRS_CULLMODE, D3DCULL_NONE );
	m_pDevice->SetRenderState( D3DRS_ALPHABLENDENABLE, TRUE );
	m_pDevice->SetRenderState( D3DRS_SRCBLEND, D3DBLEND_SRCALPHA );
	m_pDevice->SetRenderState( D3DRS_DESTBLEND, D3DBLEND_INVSRCALPHA );		

	if( m_bRenderDisplay )
	{	
		m_pDevice->SetRenderState( D3DRS_STENCILENABLE, TRUE );
		m_pDevice->SetRenderState( D3DRS_STENCILFUNC, D3DCMP_NOTEQUAL );
		m_pDevice->SetRenderState( D3DRS_STENCILREF, 0x1 );
	}

	m_pDevice->SetTexture( 0, GetTexture( "border.bmp" ) );
	m_pDevice->SetTextureStageState( 0, D3DTSS_COLOROP, D3DTOP_SELECTARG2 );	
	m_pDevice->SetTexture( 1, GetTexture( "model.bmp" ) );
	m_pDevice->SetTextureStageState( 1, D3DTSS_ALPHAOP, D3DTOP_MODULATE );

	m_pDevice->SetVertexShader( FVF_LVERTEX );
	m_pDevice->SetStreamSource( 0, m_pvbCube, sizeof(LVERTEX) );
	
	m_pDevice->SetTransform( D3DTS_TEXTURE1, ProjectionMtx( 30.0f, 0.75f, 1.0f, 20.0f )*TranslationMtx( 1.0f, 1.0f, 0.0f )*ScaleMtx( 0.5f, -0.5f, 1.0f )*TranslationMtx( m_fShift, m_fRotation*0.25f, 0.0f ) );	
	m_pDevice->SetTransform( D3DTS_WORLD, RotationMtx( m_fRotation, -m_fRotation, 0.5f*m_fRotation ) );

	m_pDevice->DrawPrimitive( D3DPT_TRIANGLELIST, 0, 12*64 );	

	m_pDevice->SetTexture( 0, NULL );
	m_pDevice->SetTextureStageState( 0, D3DTSS_COLOROP, D3DTOP_MODULATE );

	m_pDevice->SetTexture( 1, NULL );
	m_pDevice->SetTextureStageState( 1, D3DTSS_COLOROP, D3DTOP_DISABLE );
	m_pDevice->SetTextureStageState( 1, D3DTSS_ALPHAOP, D3DTOP_DISABLE );
	m_pDevice->SetTextureStageState( 1, D3DTSS_TEXCOORDINDEX, D3DTSS_TCI_PASSTHRU );
	m_pDevice->SetTextureStageState( 1, D3DTSS_TEXTURETRANSFORMFLAGS, D3DTTFF_DISABLE );
	m_pDevice->SetTextureStageState( 1, D3DTSS_ADDRESSU, D3DTADDRESS_WRAP );	

	if( m_bRenderDisplay )
	{	
		m_pDevice->SetRenderState( D3DRS_STENCILENABLE, FALSE );
		m_pDevice->SetRenderState( D3DRS_STENCILFUNC, D3DCMP_ALWAYS );
		m_pDevice->SetRenderState( D3DRS_STENCILREF, 0x0 );		

		m_pDevice->SetRenderState( D3DRS_SRCBLEND, D3DBLEND_ONE );
		m_pDevice->SetRenderState( D3DRS_DESTBLEND, D3DBLEND_ONE );		
	
		m_pDevice->SetRenderState( D3DRS_STENCILENABLE, TRUE );
		m_pDevice->SetRenderState( D3DRS_STENCILFUNC, D3DCMP_EQUAL );
		m_pDevice->SetRenderState( D3DRS_STENCILREF, 0x1 );

		m_pDevice->SetRenderState( D3DRS_ZENABLE, FALSE );
		m_pDevice->SetTransform( D3DTS_WORLD, RotationMtx( m_fRotation, -m_fRotation, 0.5f*m_fRotation ) );
		m_pDevice->SetStreamSource( 0, m_pvbDotCube, sizeof(SIMPLEVERTEX) );
		m_pDevice->SetVertexShader( FVF_SIMPLEVERTEX );
		m_pDevice->DrawPrimitive( D3DPT_POINTLIST, 0, 296*64 );

		m_pDevice->SetRenderState( D3DRS_STENCILENABLE, FALSE );
		m_pDevice->SetRenderState( D3DRS_STENCILFUNC, D3DCMP_ALWAYS );
		m_pDevice->SetRenderState( D3DRS_STENCILREF, 0x0 );		

		m_pDevice->SetRenderState( D3DRS_SRCBLEND, D3DBLEND_SRCALPHA );
		m_pDevice->SetRenderState( D3DRS_DESTBLEND, D3DBLEND_INVSRCALPHA );		
	}

	if( m_bRenderDisplay )
	{	
		m_pDevice->SetRenderState( D3DRS_TEXTUREFACTOR, m_dwDispAlpha << 24 );

		m_pDevice->SetTexture( 0, GetTexture( "display.bmp" ) );
		m_pDevice->SetTextureStageState( 0, D3DTSS_COLOROP, D3DTOP_SELECTARG2 );
		m_pDevice->SetTextureStageState( 0, D3DTSS_ALPHAOP, D3DTOP_MODULATE );

		m_pDevice->SetTexture( 1, GetTexture( "display2.bmp" ) );
		m_pDevice->SetTextureStageState( 1, D3DTSS_COLOROP, D3DTOP_SELECTARG2 );
		m_pDevice->SetTextureStageState( 1, D3DTSS_ALPHAOP, D3DTOP_BLENDFACTORALPHA );

		m_piqDisplay->Render( m_pDevice );

		m_pDevice->SetTexture( 0, NULL );
		m_pDevice->SetTextureStageState( 0, D3DTSS_COLOROP, D3DTOP_MODULATE );
		m_pDevice->SetTextureStageState( 0, D3DTSS_ALPHAOP, D3DTOP_SELECTARG1 );

		m_pDevice->SetTexture( 1, NULL );
		m_pDevice->SetTextureStageState( 1, D3DTSS_COLOROP, D3DTOP_DISABLE );
		m_pDevice->SetTextureStageState( 1, D3DTSS_ALPHAOP, D3DTOP_DISABLE );
	}

	if( m_bRenderFade )
	{
		m_pDevice->SetTexture( 0, NULL );
		m_pDevice->SetRenderState( D3DRS_SRCBLEND, D3DBLEND_SRCALPHA );
		m_pDevice->SetRenderState( D3DRS_DESTBLEND, D3DBLEND_ONE );

		m_piqFade->Render( m_pDevice );
	}

	m_pDevice->SetRenderState( D3DRS_ALPHABLENDENABLE, FALSE );	

	

	return TRUE;
}
