/* 									2001
  Local Library (ll) is a library produced for making applications
  a bit platform independent. 

  This is the linux X-based version. 

  Programmer: Kirkov
*/


   /*####################################################################
   # Includes								#
   ####################################################################*/

#include <stdio.h>
#include <string.h>

#include <sys/time.h>
#include <unistd.h>

#include <GL/glx.h>
#include <GL/gl.h>

#include "ll.h"


   /*####################################################################
   # Globals								#
   ####################################################################*/

/* The current display and window, used by getEvents and swapBuffers */
Display *dpy;
Window win;

/* Window information. */
int winXPos,winYPos,winWidth,winHeight;

/* Store information about state for all keys */
unsigned char *llKeyTable;

/* Mouse information. */
short int mouseX,mouseY;		/* Where was mouse last llHandleEvents(). */
unsigned char mouseButtons;		/* Button state last llHandleEvents(). */
unsigned char oldMouseButtons;		/* Last button state so we can tell if a button have been pressed. */
short int mouseHomeX,mouseHomeY;	/* Where should mosue be set after reading in llHandleEvents(). */


   /*####################################################################
   # Function definitions						#
   ####################################################################*/

int setupLL(){
	llKeyTable=(char *)malloc(128);
	if(llKeyTable==NULL){
		printf("Keytable allocation failed.\n");
		return LL_ERROR;
	}
	memset(llKeyTable,0,128);

	/* Zero mouse valls. */
	mouseX=winWidth>>1;
	mouseY=winHeight>>1;
	mouseHomeX=mouseHomeY=-1;
	oldMouseButtons=mouseButtons=0;

	return LL_OK;
}

void freeLL(){
	free(llKeyTable);
}

/* Make some prepings and open a window. Set redering context for OpenGL. */
/* For further information check function prototype in 'll.h' */
/* 0=ERROR. 1=OK. */
int llCreateOpenGLWindow(char *name,unsigned char mask,int xPos,int yPos,int width,int height){

	XVisualInfo *xvInfo;
	int attribList[20];
	XSetWindowAttributes winAttrib;
	GLXContext context;
	int screen;
	int tmp;

	dpy=XOpenDisplay(NULL);
	if(dpy==NULL){
//		printf("LL: Could not open display!\n");
		return LL_ERROR;
	}

	/* Set proper values according to mask */
	tmp=0;
	attribList[tmp++]=GLX_RGBA;
	if(mask & LL_DEPTHBUFFER){ attribList[tmp++]=GLX_DEPTH_SIZE; attribList[tmp++]=16; }
	if(mask & LL_DOUBLEBUFFER)attribList[tmp++]=GLX_DOUBLEBUFFER;
	attribList[tmp++]=None;

	screen=DefaultScreen(dpy);
	xvInfo=glXChooseVisual(dpy,screen,attribList);
	if(xvInfo==NULL){
//		printf("LL: Could not open window!\n");
		return LL_ERROR;
	}

	context=glXCreateContext(dpy,xvInfo,None,GL_TRUE);
	if(context==NULL){
		printf("LL: Could not create context for new window!\n");
		return LL_ERROR;
	}

	winAttrib.colormap=XCreateColormap(dpy,RootWindow(dpy,xvInfo->screen),xvInfo->visual,AllocNone);
	winAttrib.border_pixel=0;
	winAttrib.event_mask=ExposureMask | StructureNotifyMask | KeyPressMask | KeyReleaseMask | ButtonPressMask | ButtonReleaseMask | PointerMotionMask;
	if(mask & LL_FULLSCREEN){
		winAttrib.override_redirect=True;
		width=DisplayWidth(dpy,screen);
		height=DisplayHeight(dpy,screen);
	}
	win=XCreateWindow(dpy ,
		RootWindow(dpy,xvInfo->screen) ,
		xPos ,yPos ,width ,height ,0 ,xvInfo->depth ,
		InputOutput, xvInfo->visual ,
		CWBorderPixel | CWColormap | CWEventMask | (mask & LL_FULLSCREEN?CWOverrideRedirect:0),
		&winAttrib
		);

	XSetStandardProperties(dpy,win,name,name,None,NULL,0,NULL);

	if(!glXMakeCurrent(dpy,win,context)){
//		printf("LL: Could not set context on new window as current!\n");
		return LL_ERROR;
	}

	XMapWindow(dpy,win);
	if(mask & LL_FULLSCREEN)XSetInputFocus(dpy,win,RevertToNone,CurrentTime);
	glViewport(0,0,width,height);

	if(setupLL()!=LL_OK)return LL_ERROR;

	/* Save info about window. */
	winXPos=xPos;
	winYPos=yPos;
	winWidth=width;
	winHeight=height;

	return LL_OK;
}

void llKillOpenGLWindow(){
	freeLL();
}

/* If double buffering use this to swap. */
void llSwapBuffers(){
	glXSwapBuffers(dpy,win);
}

/* Take care of events sent to our application. 'mode' indicates wheter we should wait or
   just check for events. */
int llHandleEvents(int mode){
	XEvent event;
	oldMouseButtons=mouseButtons;
	do{

		/* Quit if we should not wait for an event and there is no event. */
		if(mode==LL_EVENT_CHECK && !XPending(dpy))break;

		XNextEvent(dpy,&event);
		switch(event.type){
			/* If X tells us and we want to. Repaint window. */
			case Expose:
				/* Nothing yet. */
			break;
			/* Handle keyboard. Update key-table. */
			case KeyPress:
				if(llKeyTable[event.xkey.keycode]==0)
					llKeyTable[event.xkey.keycode]=3;
				else
					llKeyTable[event.xkey.keycode]=1;
			break;
			case KeyRelease:
				llKeyTable[event.xkey.keycode]=0;
			break;
			/* Mouse moves. */
			case MotionNotify:
				mouseX=event.xbutton.x;
				mouseY=event.xbutton.y;
				if(mouseHomeX!=-1 && !(mouseX==mouseHomeX && mouseY==mouseHomeY))llMouseSetPosition(mouseHomeX-(winWidth>>1),mouseHomeY-(winHeight>>1));
			break;
			/* Clicks. */
			case ButtonPress:
				if(event.xbutton.button==Button1){
					mouseButtons|=LL_MOUSE_BUTTON_LEFT;
				}else if(event.xbutton.button==Button3){
					mouseButtons|=LL_MOUSE_BUTTON_RIGHT;
				}
			break;
			case ButtonRelease:
				if(event.xbutton.button==Button1){
					mouseButtons&=~LL_MOUSE_BUTTON_LEFT;
				}else if(event.xbutton.button==Button3){
					mouseButtons&=~LL_MOUSE_BUTTON_RIGHT;
				}
			break;
		}
	} while(XPending(dpy));

	return LL_OK;
}

/* Mouse handling. Get state last time llHandleEvents() was called. */
short int llMouseGetXPos(){
	return mouseX-(winWidth>>1);
}
short int llMouseGetYPos(){
	return mouseY-(winHeight>>1);
}

/* Set mouse position (0,0) is middle of window. */
void llMouseSetPosition(short int x,short int y){
	XWarpPointer(dpy,None,win,0,0,0,0,x+(winWidth>>1),y+(winHeight>>1));
}

/* Set mouse home position. MPos will be set after reading in llHandleEvents(). */
/* If (x,y) is outside window no position will be set. */
void llMouseSetHome(short int x,short int y){
	mouseHomeX=x+(winWidth>>1);
	mouseHomeY=y+(winHeight>>1);

	/* Check if there is any point. */
	if(mouseHomeX<0 || mouseHomeX>=winWidth || mouseHomeY<0 || mouseHomeY>=winHeight)mouseHomeX=mouseHomeY=-1;
}

/* Some functions for retrieveing mouse button status. */
unsigned char llMouseButtonDown(unsigned char button){
	return (mouseButtons & button);
}
unsigned char llMouseButtonPressed(unsigned char button){
	return ((mouseButtons & button)==0?(oldMouseButtons & button):0);
}

/* Current time in milliseconds. */
unsigned long llGetTime(){
	struct timeval tv;
	gettimeofday(&tv,NULL);
	return (tv.tv_sec*1000+tv.tv_usec/1000);	/* Return milliseconds. */
}

/* Brake execution if needed. Brake if not 'time' has passed since last call. */
void llBrake(unsigned long time){
	static unsigned long lastTime=0;
	if(lastTime==0){
		lastTime=llGetTime();
		return;
	}
	while(llGetTime()-time<lastTime);
	lastTime=llGetTime();
}


