/*
    "console.cpp"   - <SDL/C++> Console class

    DDS - Dureks DemoSystem
    Copyright (C)2001 dureks

    This source code is licensed under the GNU GPL.
    See the GNU General Public License for more details.
*/

#include <fstream.h>
#include "console.h"
#include "error.h"
#include "sdlbase.h"
#include "surface.h"

namespace dds {

    /*  Console class
        ------------- */



    /*
        Here we initialize the needed libraries and subsystems according to the
        flags that is received.
    */
Console::Console(const int32 init)
{
    m_init = init;
    m_flags = SDL_DOUBLEBUF|SDL_HWSURFACE;   // always try double buffering!
    m_video_open = false;

    // currently, SDL has pretty bad support for both 2D and 3D
    if( (m_init & video) && (m_init & opengl) )
        throw Error("Console::Console()","Video and OpenGL combined is not a good idea");

    // SDL initialization
    if( m_init & video || m_init & opengl ) {
        if( SDL_Init(SDL_INIT_VIDEO) == -1 )
            throw Error("Console::Console()","Video initialization failed");

        // get video device information
        m_videoinfo = SDL_GetVideoInfo();
    }

    // FMOD soundsystem initialization
    if( m_init & sound ) {
        // try to avoid quality mixers; they are evil
        FSOUND_SetMixer(FSOUND_MIXER_QUALITY_AUTODETECT);

        // FIX: more dynamic sound initialization
        if( !FSOUND_Init(44100,128,0) ) {
            throw Error("Console::Console()","FSOUND_Init() failed");
        }

        //fprintf(stdout,"#smpz: %i\n",FSOUND_GetMaxSamples());
    }

    // SDL timer subsystem
    if( m_init & timer ) {
        if( SDL_InitSubSystem(SDL_INIT_TIMER) == -1)
            throw Error("Console::Console()","Timer initialization failed");
    }

    if( m_init & opengl )
        m_flags |= SDL_OPENGL;

    if( m_init & fullscreen )
        m_flags |= SDL_FULLSCREEN;
}

    /*
        Attempts to shut down all the libraries that were initialized when
        the console was created.
    */
Console::~Console()
{
    close();

    if( (m_init & video) || (m_init & timer) || (m_init & opengl) )
        SDL_Quit();

    if( m_init & sound)
        FSOUND_Close();

    // do not deallocate frambuffer in ~BaseSurface()
    m_surface = NULL;
}

    /*
        This function is used to set a desired video mode. The video surface's
        features are the ones that were selected when the console was created.
    */
void Console::open(const char title[], int32 width, int32 height, int32 depth)
{
    if( m_video_open )
        throw Error("Console::open()","Console is already open");

    if( !(m_init & video) && !(m_init & opengl) )
        throw Error("Console::open()","Video output not established");

    // videomode testing
    if( SDL_VideoModeOK(width,height,depth,m_flags) != 0 ) {
        // check for other depths
        if( depth != 0 )
            if( SDL_VideoModeOK(width,height,0,m_flags) != 0 )
                throw Error("Console::open()","Invalid video mode");
        depth = 0;
    }

	if( !m_videoinfo->blit_hw_CC && m_videoinfo->blit_sw )
		m_flags &= ~SDL_HWSURFACE;

	if( !m_videoinfo->hw_available )
		m_flags &= ~SDL_HWSURFACE;

    // no error should ever occur here
    m_surface = SDL_SetVideoMode(width,height,depth,m_flags);
    if( !m_surface ) {
        throw Error("Console::open()","SDL_SetVideoMode failed, flags:",(int)m_flags);
    }
    unlock();

    if( m_flags & SDL_OPENGL || m_flags & SDL_OPENGLBLIT ) {
        SDL_GL_GetAttribute(SDL_GL_DOUBLEBUFFER, &m_gl_doublebuffer);
        SDL_GL_GetAttribute(SDL_GL_DEPTH_SIZE,&m_gl_bpp);
        SDL_GL_GetAttribute(SDL_GL_BUFFER_SIZE,&m_gl_zbits);
        SDL_GL_GetAttribute(SDL_GL_STENCIL_SIZE,&m_gl_stencilbits);
    }

    caption(title);
    if( m_flags & SDL_FULLSCREEN )
        cursor(false);

    m_video_open = true;
    m_what_buf = 0; // we're at doublebuffer #0
}

    /*
        This function shuts down the video surface that was opened.
    */
void Console::close()
{
    if( m_video_open ) {
        // this is to ensure we end up at the right console buffer
        // when running the program from linux framebuffer device
        if( m_what_buf )    // always 0 when at buffer 0
            update();

        m_video_open = false;
    }
}

    /*
        The console update function flips the contents of the current buffer onto
        the screen. (This also includes updating the OpenGL rendering output.)
    */
void Console::update()
{
    // FIX: two float buffer swaps each update is dead slow, is there any way to avoid this?
    /*
    if( (m_init & opengl) && m_gl_doublebuffer )
        SDL_GL_SwapBuffers();

    if( m_init & video ) {*/
        SDL_Flip(m_surface);
        m_what_buf = !m_what_buf;
    //}
}


}   // namespace dds
