/*
    "demo.cpp"   - Demo class
     
    DDS - Dureks DemoSystem
    Copyright (C)2001 dureks

    This source code is licensed under the GNU GPL.
    See the GNU General Public License for more details.
*/

//#define STDXX_REDIRECTION 1

#include <stdio.h>
#include <math.h>
#include "demo.h"
#include "error.h"

namespace dds {

Demo::Demo()
{
#ifdef STDXX_REDIRECTION
    // redirect stdout and stderr
    stdout = freopen("stdout.dds","w",stdout);
    stderr = freopen("stderr.dds","w",stderr);
#endif
    console = NULL;
}

Demo::~Demo()
{
#ifdef STDXX_REDIRECTION
    if( ftell(stdout) == 0 ) {
    }
    if( ftell(stderr) == 0 ) {
    }
#endif
}

    /*
        This function handles all events such as keyboard press/release, mouse
        movement/button and window manager functions. It should be called when the
        demo is ready to run. The program will enter an infinite event loop, which
        only can be exited with a quit request.
    */
void Demo::run()
{
    m_event_quit = false;
	m_display_update = false;

    try {
        while( !m_event_quit ) {

            // win32 platform threading fix (=no threading:)
			if( m_display_update ) {
				display();
				m_display_update = false;
			}
				
            if( SDL_PollEvent(&poll_event) ) {
                switch( poll_event.type ) {
                    case SDL_ACTIVEEVENT:
                        if( poll_event.active.gain )
                            window_focus_gain(poll_event.active.state);
                        else
                            window_focus_lost(poll_event.active.state);
                        break;
                    case SDL_VIDEORESIZE:
                        window_resize(poll_event.resize.w, poll_event.resize.h);
                        break;
                    case SDL_KEYDOWN:
                        key_pressed(poll_event.key.keysym.sym, poll_event.key.keysym.mod);
                        break;
                    case SDL_KEYUP:
                        key_released(poll_event.key.keysym.sym, poll_event.key.keysym.mod);
                        break;
                    case SDL_MOUSEMOTION:
                        mouse_motion(poll_event.motion.x, poll_event.motion.y, poll_event.motion.xrel, poll_event.motion.yrel);
                        break;
                    case SDL_MOUSEBUTTONDOWN:
                        mouse = poll_event;
                        mouse_button_pressed(poll_event.button.button, poll_event.button.x, poll_event.button.y);
                        break;
                    case SDL_MOUSEBUTTONUP:
                        mouse.type = quit;
                        mouse_button_released(poll_event.button.button, poll_event.button.x, poll_event.button.y);
                        break;
                    case SDL_QUIT:
                        m_event_quit = true;
                        break;
                }
            }
        }
    }
    catch(Error &error) {
        throw Error("Demo::run()",error);
    }

    delete_threads();

    // Call EndOfDemo method
    eod();

    // FIX: fmod _must_ be uninitialized upon exit, or the thread will hang
    if( console )
        delete console;
}

    /*
        This is the user function to establish a threaded class. The first argument
        specifies which function to call, with the second parameter as the delay
        between each call.
    */
void Demo::thread(threadfunc func, int32 msec)
{
    DemoThread *thread = new DemoThread;
    thread->func = func;

    if( func == update_func )
        thread->id = SDL_AddTimer(msec,update_thread,this);

    m_threads.push(thread);
}

void Demo::delete_threads() {
    while( DemoThread *th = m_threads.pop(List::end) ) {
        SDL_RemoveTimer(th->id);
        delete th;
    }
}

void Demo::key_pressed(SDLKey key, SDLMod mod)
{
    // ESC
    if( key == SDLK_ESCAPE )
        request(quit);

    // Alt-F4
    if( key == SDLK_F4  && mod == KMOD_LALT )
        request(quit);
}

    /*
        This function keeps the variable updating on a steady rate by running it at
        a separate thread.
    */
int32 Demo::update_thread(int32 interval, void *param)
{
    Demo *temp = (Demo*)param;
    static int old_timer = SDL_GetTicks();
    static float rest;

    Uint32 ticks = SDL_GetTicks();
    float skip_frames = ((float)(ticks - old_timer))/(float)interval;
    //fprintf(stderr,"Skipframes: %f, Test: %u\n",skip_frames,ticks-old_timer);
    old_timer = ticks;
    rest += skip_frames - (int)skip_frames;
    if( rest > 1 ) {
        skip_frames++;
        rest--;
    }
    //int skip_frames = 1;
    skip_frames = floor(skip_frames);
    while( skip_frames-- ) {
        temp->update();

        if( temp->mouse.type != quit ) {
            temp->mouse_button_is_pressed(temp->mouse.button.button,temp->poll_event.motion.x,temp->poll_event.motion.y);
        }
    }

//#ifdef WIN32
    temp->request(redraw);
//#else
//    temp->display();
//#endif

    return(interval);
}

    /*
        Use to request an action, such as program exit or redraw the screen.
    */
void Demo::request(int32 flags)
{
    if( flags & redraw )
		m_display_update = true;

    if( flags & quit )
        m_event_quit = true;
}

}   // namespace dds
