#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include "types.h"
#include "error.h"
#include "emitter.h"

namespace dds {

Emitter::Emitter(int32 count, Vector2 pos)
{
    if( !count )
        throw Error("mEmitter::mEmitter","Cannot initialize emitter with zero particles");

    m_particles = count;
    m_particle = new Particle[m_particles];

    m_position = pos;
    m_speed = M_PARTICLE_SPEED;
    m_speed_rand = 0.0f;
    m_life = M_PARTICLE_LIFE;
    m_life_rand = M_PARTICLE_LIFE_RAND;
    m_fade = M_PARTICLE_FADE;
    m_color[0] = 1.0f;
    m_color[1] = 1.0f;
    m_color[2] = 1.0f;

    Particle *pt = m_particle;
    while( count-- ) {
        pt->active = false;
        memcpy(pt->color,m_color,sizeof(float[3]));
        pt->direction(0,0);
        pt->accel(0,0);
        pt->fade = 0.0f;
        pt->life = 0.0f;
        pt->position = m_position;
        pt->speed = 0.0f;
        pt++;
    }
}

Emitter::~Emitter()
{
    delete []m_particle;
}

Particle* Emitter::particle(int32 n)
{
    if( n > m_particles )
        return(NULL);

    return(&m_particle[n]);
}

void Emitter::update()
{
    // update all mEmitter
    int32 n = m_particles;
    Particle *pt = m_particle;

    while( n-- ) {

        if( pt->active ) {
            pt->life -= pt->fade;

            if( pt->life < 0.f ) // "dead" particle
                pt->active = false;

            pt->position += pt->direction*pt->speed;
        }

        pt++;
    }
}

void Emitter::revive(int32 count)
{
    int32 n = m_particles;
    Particle *pt = m_particle;

    while( n-- && count ) {
        if( !pt->active ) {
            pt->active = true;
            pt->position = m_position;

            pt->direction.x( ((float)(rand() % 100)/50.0f)*m_random.x()-m_random.x());
            pt->direction.y( ((float)(rand() % 100)/50.0f)*m_random.y()-m_random.y());
            pt->direction += m_direction;
            pt->direction.normalize();

            pt->life = m_life-(((float)(rand() % 100)/100.0f)*m_life_rand);
            pt->fade = m_fade;
            pt->speed = m_speed+((float)(rand() % 100)/50.0f)*m_speed_rand-m_speed_rand;

            memcpy(pt->color,m_color,sizeof(float[3]));
            count--;
        }
        pt++;
    }
}

void Emitter::draw(BaseSurface *sprite, BaseSurface *surface)
{
    int32 hw = sprite->width()/2;
    int32 hh = sprite->height()/2;

    Particle *particle = m_particle;
    for(int n=0;n<(int)m_particles;n++) {
        if( particle->active ) {
            char8 grade = (char8)((1.f-particle->life)*255.f);
            blit.addgrade((int)particle->position.x()-hw,(int)particle->position.y()-hh,grade,sprite,surface);
        }

        particle++;
    }
}


    // gravity physics class
    // ---------------------

Gravity::Gravity(float g)
{
    // not exactly physics
    m_gravity = (g/100.f);
}

void Gravity::apply(Emitter *emitter)
{
    Particle *particle = emitter->particle(0);
    int32 particles = emitter->particles();
    Vector2 d(0,m_gravity);

    while( particles-- ) {
        if( particle->active )
            particle->direction += d;
        particle++;
    }
}


    // attractor physics class
    // -----------------------

Attractor::Attractor(Vector2 position, float attraction)
{
    m_position = position;
    m_attraction = attraction;
}

void Attractor::apply(Emitter *emitter)
{
    Particle *particle = emitter->particle(0);
    int32 particles = emitter->particles();

    Vector2 d;

    while( particles-- ) {
        if( particle->active ) {
            d = particle->position - m_position;
            double l = d.length();
            if( l ) {
                d *= -m_attraction/(l*l);
                particle->direction += d;
            }
        }
        particle++;
    }
}

void Attractor::position(Vector2 pos)
{
    m_position = pos;
}

}   // namespace dds