/*
    "console.h"   - SDL/C++ GFX font class

    DDS - Dureks DemoSystem
    Copyright (C)2001 dureks

    This source code is licensed under the GNU GPL.
    See the GNU General Public License for more details.
*/

#include <fstream.h>
#include <stdlib.h>
#include <string.h>
#include "gfxfont.h"
#include "error.h"

namespace dds {

GFXFont::GFXFont(const char sfont[], int32 flags)
{
    for(int c=0;c<256;c++)
        letter[c] = NULL;

    try {
        image = new Image(sfont,flags);
        map_sfont();
    }
    catch(Error &error) {
        throw Error("GFXFont::GFXFont()",error);
    }
}

GFXFont::GFXFont(const char sfont[], const Format &format, int32 flags)
{
    for(int c=0;c<256;c++)
        letter[c] = NULL;

    try {
        image = new Image(sfont,format,flags);
        map_sfont();
    }
    catch(Error &error) {
        throw Error("GFXFont::GFXFont()",error);
    }
}

GFXFont::~GFXFont()
{
    for(int c=0;c<256;c++)
        delete letter[c];

    delete image;
}

void GFXFont::write(int px, int py, char text[], BaseSurface *dest)
{
    int length = strlen(text);
    int cx = px;
    for(int ch=0;ch<length;ch++) {
        if( text[ch] == SFONT_SEPARATOR )
            cx += m_spacing;
        else if( letter[text[ch]] ) {
            Area area(cx,py,cx+letter[text[ch]]->w(),py+letter[text[ch]]->h());
            dest->copy(image,*letter[text[ch]],area);
            cx += letter[text[ch]]->w();
        }
    }
}

void GFXFont::map_sfont()
{
    int32 height = image->height();
    int32 width = image->width();
    int32 start;
    bool get_pixel = false;
    int32 cc = SFONT_STARTCHAR;

    for(int32 x=0;x<width;x++) {
        Color *color = getpixel(x,0);

        if( color->r() == 0 && color->b() == 0 && !get_pixel ) {
            get_pixel = true;
            start = x;
        }

        if( color->r() != 0 && color->b() != 0 && get_pixel ) {
            get_pixel = false;
            letter[cc++] = new Area(start,0,x-1,height);

            if( !m_spacing )
                m_spacing = x-start;
            else
                m_spacing = (m_spacing+x-start)/2;
        }

        delete color;
    }
}

Color* GFXFont::getpixel(int32 x, int32 y)
{
    int32 pitch = image->pitch();
    int32 ppitch = image->pixelpitch();
    char8 *data = (char8*)image->lock();

    x *= ppitch;

    int32 pixel;
    switch( image->depth() ) {
        case 8:
            pixel = *(char8*)(data+x+y*pitch);
            break;
        case 16:
            pixel = *(int16*)(data+x*2+y*pitch);
            break;
        case 24:
            pixel = *(int32*)(data+x*3+y*pitch);  // FIX: rewrite!
            break;
        case 32:
            pixel = *(int32*)(data+x*4+y*pitch);
            break;
    }
    image->unlock();

    char8 r,g,b,a;
    SDL_GetRGBA(pixel,(SDL_PixelFormat*)&image->format().SDLPixelFormat(),&r,&g,&b,&a);
    Color *color = new Color(r,g,b,a);
    return(color);
}

}   // namespace dds