/*
    "image.cpp"   - <SDL/C++> Image class (Surface)

    DDS - Dureks DemoSystem
    Copyright (C)2001 dureks

    This source code is licensed under the GNU GPL.
    See the GNU General Public License for more details.
*/

#include <fstream.h>
#include <stdio.h>
#include <string.h>
#include "image.h"
#include "error.h"

namespace dds {

    /*  Image class
        ----------- */

Image::Image(const char file[],const int32 flags)
{
    try {
        ImageLoader_auto loader;
        m_surface = loader.load(file,flags);
    }
    catch(Error &error) {
        throw Error("Image::Image()",error);
    }

    m_name = new char[strlen(file)+1];
    strcpy(m_name,file);
    unlock();
}

Image::Image(const char file[],const Format &format,const int32 flags)
{
    if( format.depth() == 8 ) {
        throw Error("Image::Image()","Conversion to 8bpp indexed color is not supported:",file);
    }

    // temporary storage surface (before converting to requested format)
    SDL_Surface *surface;

    try {
        ImageLoader_auto loader;
        surface = loader.load(file,flags);
    }
    catch(Error &error) {
        throw Error("Image::Image()",error);
    }

    // FIX: alpha channel is not preserved when performing blit, avoid until fixed
    m_surface = SDL_CreateRGBSurface(flags,surface->w,surface->h,format.depth(),
                                     format.rmask(),format.gmask(),format.bmask(),format.amask());
    SDL_UnlockSurface(m_surface);

    copy(surface);
    SDL_FreeSurface(surface);
    m_name = new char[strlen(file)+1];
    strcpy(m_name,file);
    unlock();
}

Image::~Image()
{
    delete []m_name;
}

    /*  Image autoloader
        ---------------- */

bool ImageLoader_auto::check(const char fname[])
{
    ifstream *file = new ifstream(fname,ios::in|ios::nocreate);

        // check if file exists
    if(!file->is_open()) {
        delete file;
        return(false);
    }

    delete file;
    return(true);
}

    /* ImageLoader_auto::load() will invoke all the other loaders and attempt
        to load the image with the appropriate one. If none are found, an error
        is thrown.
    */
SDL_Surface* ImageLoader_auto::load(const char file[], const int32 flags)
{
    if( !check(file) )
        throw Error("ImageLoader_auto::load()","File not found",file);

    ImageLoader_PNG loadpng;
    ImageLoader_JPEG loadjpeg;

    try {
        if( loadpng.check(file) )       // PNG
            return loadpng.load(file,flags);

        if( loadjpeg.check(file) )      // JPEG
            return loadjpeg.load(file,flags);

    }   // if loader fails
    catch( Error &error ) {
        throw Error("ImageLoader_auto::load()",error);
    }

        // unknown file
    throw Error("ImageLoader_auto::load()","Unknown file format",file);
}


}   // namespace dds
