/*
    "jpeg.cpp"   - <SDL/C++> JPEG image loader
     
    DDS - Dureks DemoSystem
    Copyright (C)2001 dureks

    This source code is licensed under the GNU GPL.
    See the GNU General Public License for more details.
*/

#include <stdio.h>
#include <setjmp.h>
extern "C" {
#include <jpeglib.h>
}

#include "error.h"
#include "image.h"
#include "surface.h"

namespace dds {

    /*  JPEG Image loader
        ----------------- */

    // libjpeg error handling
struct my_error_mgr {
    struct jpeg_error_mgr pub;
    jmp_buf setjmp_buffer;
};

typedef struct my_error_mgr *my_error_ptr;

static void my_error_exit(j_common_ptr cinfo)
{
    // attempt to report JPEG library messages our standard way
    throw Error("ImageLoader_JPEG::load()","JPEG library error, code:",(int)cinfo->err->msg_code);
}

bool ImageLoader_JPEG::check(const char fname[])
{
    FILE *file;
    char buffer[2];

    if( !(file = fopen(fname,"rb")) ) {
        return(false);
    }

    // read two bytes for identification
    if( (fread(buffer,1,2,file)) != 2 ) {
        fclose(file);
        return(false);
    }
    fclose(file);

    // JPEG header check
    if( ((unsigned char)buffer[0] != 0xFF) || ((unsigned char)buffer[1] != 0xD8) ) {
        return(false);
    }

    return(true);
}

SDL_Surface* ImageLoader_JPEG::load(const char fname[], const int32 SDLflags)
{
    if( !check(fname) ) {
        throw Error("ImageLoader_JPEG::load()","Unable to load requested file:",fname);
    }

    FILE *file = fopen(fname,"rb");

    jpeg_decompress_struct cinfo;
    my_error_mgr jerr;

    // allocate JPEG decompression object
    cinfo.err = jpeg_std_error(&jerr.pub);
    jerr.pub.error_exit = my_error_exit;
    if (setjmp(jerr.setjmp_buffer)) {
        jpeg_destroy_decompress(&cinfo);
        fclose(file);
        throw Error("ImageLoader_JPEG::load()","JPEG library failed:",fname);
    }
    jpeg_create_decompress(&cinfo);

    // specify JPEG source
    jpeg_stdio_src(&cinfo,file);

    // read file parameters
    jpeg_read_header(&cinfo, TRUE);

    // start decompressor
    jpeg_start_decompress(&cinfo);

    // create surface to store image data in (JPEGs are always 24bpp)
    SDL_Surface *surface = SDL_CreateRGBSurface(SDLflags,cinfo.image_width,cinfo.image_height,24,0x00000FF,0x0000FF00,0x00FF0000,0x00000000);
    if( SDL_LockSurface(surface) != 0)
        throw Error("ImageLoader_JPEG::load()","SDL_LockSurface failed:",fname);

    // assign surface rows
    int32 count;
    int32 rowbytes = surface->pitch;
    unsigned char *buffer = (unsigned char*)surface->pixels;
    unsigned char **row_pointers = new unsigned char* [cinfo.image_height];
    for(count=0;count<(int)cinfo.image_height;count++)
        row_pointers[count] = buffer+count*rowbytes;

    // read JPEG scanlines to surface rows
    count = 0;
    while (cinfo.output_scanline < cinfo.output_height)
        jpeg_read_scanlines(&cinfo,&row_pointers[count++], 1);

    delete []row_pointers;
    SDL_UnlockSurface(surface);

    // finish decompression
    jpeg_finish_decompress(&cinfo);

    // release JPEG decompression object
    jpeg_destroy_decompress(&cinfo);
    fclose(file);

    return(surface);
}

}   // namespace dds