/*
    "matrix.h"   - Matrix class

    DDS - Dureks DemoSystem
    Copyright (C)2001 dureks

    This source code is licensed under the GNU GPL.
    See the GNU General Public License for more details.
*/

#ifndef _DDS_MATRIX_H
#define _DDS_MATRIX_H

#include "vector.h"
#include <math.h>

namespace dds {

class Matrix2 {
/*    todo:
		- implement Matrix3, Matrix 4
		- investigate 3DNOW(ext), MMX, T&L
*/

public:
	inline Matrix2(float, float, float, float);
    inline Matrix2(Vector2, Vector2);
	inline Matrix2();
	inline ~Matrix2() {};

	//data access
	inline void operator= (Matrix2);
	inline void operator() (Vector2, Vector2);
	inline void operator() (float, float, float, float);

	//read vectors
	inline const Vector2 row1() const { return row[0]; }
	inline const Vector2 row2() const { return row[1]; }

	//compare matrices
	inline const bool operator == (const Matrix2);
	inline const bool operator != (const Matrix2);

	//add, subtract
	inline void operator+= (Matrix2);
	inline Matrix2 operator+ (Matrix2);
	inline void operator-= (Matrix2);
	inline Matrix2 operator- (Matrix2);

	//multiply, divide with scalar
	inline void operator*= (float);
    inline Matrix2 operator* (float);
    inline void operator/= (float);
    inline Matrix2 operator/ (float);

	//multiply with matrix
	inline void operator*= (Matrix2);
	inline Matrix2 operator* (Matrix2);

	//matrix operations
	inline bool invert();
	inline Matrix2 inverse();
	inline Vector2 mulvec(Vector2 *);

    // transformation
    inline void rotation(float);
    inline void scalar(float,float);

protected:
	Vector2 row[2];
};

inline Matrix2::Matrix2(float x1, float y1, float x2, float y2)
{
    row[0](x1,y1);
    row[1](x2,y2);
}

inline Matrix2::Matrix2(Vector2 r1, Vector2 r2)
{
	row[0] = r1;
	row[1] = r2;
} 

inline Matrix2::Matrix2()
{
	row[0] = Vector2(0,0);
	row[1] = Vector2(0,0);
}

inline void Matrix2::operator() (Vector2 r1, Vector2 r2) 
{
	row[0] = r1;
	row[1] = r2;
}

inline void Matrix2::operator() (float x1, float y1, float x2, float y2)
{
    row[0](x1,y1);
    row[1](x2,y2);
}

inline void Matrix2::operator= (Matrix2 m) 
{
  row[0] = m.row1();
  row[1] = m.row2();
}

inline const bool Matrix2::operator== (Matrix2 m)
{
	if( m.row[0] == row[0] && m.row[1] == row[1] ) 
		return(true);
	return(false);
}

inline const bool Matrix2::operator!= (Matrix2 m)
{
	if ( m.row[0] != row[0] && m.row[1] != row[1] )
		return(true);
	return(false);
}

inline void Matrix2::operator+= (Matrix2 m)
{
	row[0] += m.row1();
	row[1] += m.row2();
}

inline Matrix2 Matrix2::operator+ (Matrix2 m)
{
	Matrix2 n = *this;
	n += m;
	return(n);
}

inline void Matrix2::operator-= (Matrix2 m)
{
	row[0] -= m.row1();
	row[1] -= m.row2();
}

inline Matrix2 Matrix2::operator- (Matrix2 m)
{
	Matrix2 n = *this;
	n -= m;
	return(n);
}

inline void Matrix2::operator*= (float scalar)
{
	row[0] *= scalar;
	row[1] *= scalar;
}

inline Matrix2 Matrix2::operator* (float scalar)
{
	Matrix2 n = *this;
	n *= scalar;
	return(n);
}

inline void Matrix2::operator/= (float scalar)
{
	row[0] /= scalar;
	row[1] /= scalar;
}

inline Matrix2 Matrix2::operator/ (float scalar)
{
	Matrix2 n = *this;
	n /= scalar;
	return(n);
}

inline void Matrix2::operator*= (Matrix2 m)
{
	Vector2 a1 = row[0];
	Vector2 b1 = m.row1();
	Vector2 a2 = row[1];
	Vector2 b2 = m.row2();
	row[0](a1.x()*b1.x()+a1.y()*b2.x(),a1.x()*b1.y()+a2.y()*b2.y());
	row[1](a2.x()*b1.x()+a2.y()*b2.x(),a2.x()*b1.y()+a2.y()*b2.y());
}

inline Matrix2 Matrix2::operator* (Matrix2 m)
{
	Matrix2 n = *this;
	n *= m;
	return(n);
}

inline bool Matrix2::invert()
{
	float determinant = row[0].x()*row[1].y()-row[1].x()*row[0].y();
	if( determinant == 0 )
		return(false);
	
	float a = row[0].x();
	float b = row[1].y();
	row[0](b,-row[0].y());
	row[1](-row[1].x(),a);
	*this *= (float)1/determinant;

	return(true);
}

inline Matrix2 Matrix2::inverse()
{
	Matrix2 n = *this;
	if( n.invert() )
		return(n);

	return(n); //how to handle non-invertibel matrices?
}

inline Vector2 Matrix2::mulvec(Vector2 *vec)
{
    float x = vec->x();
    float y = vec->y();
    vec->x(row[0].x()*x+row[0].y()*y);
    vec->y(row[1].x()*x+row[1].y()*y);
	return(*vec);
}

inline void Matrix2::rotation(float angle) 
{
	row[0](cos(angle),-sin(angle));
	row[1](sin(angle),cos(angle));
}

inline void Matrix2::scalar(float x, float y)
{
    row[0](x,0);
    row[1](0,y);
}

} //namespace dds


#endif //_DDS_MATRIX_H
