/*
    "png.cpp"   - <SDL/C++> PNG image loader
     
    DDS - Dureks DemoSystem
    Copyright (C)2001 dureks

    This source code is licensed under the GNU GPL.
    See the GNU General Public License for more details.
*/

#include <stdio.h>
extern "C" {
#include <png.h>
}

#include "error.h"
#include "image.h"
#include "sdlbase.h"
#include "surface.h"

namespace dds {

    // PCs
#define PNG_GAMMA_C 2.2f

    /*  PNG Image loader
        ---------------- */

bool ImageLoader_PNG::check(const char fname[])
{
  FILE *file;

    if( !(file = fopen(fname,"rb")) ) {
        return(false);
    }

    unsigned char pngheader[4];
    if( fread(pngheader,1,4,file) != 4) {
        fclose(file);
        return(false);
    }

    if( !png_check_sig(pngheader,4) ) {
        fclose(file);
        return(false);
    }

    fclose(file);
    return(true);
}

SDL_Surface* ImageLoader_PNG::load(const char fname[], const int32 SDLflags)
{
    if( !check(fname) ) {
        throw Error("ImageLoader_PNG::load()","Unable to load requested file:",fname);
    }

    FILE *file = fopen(fname,"rb");

    png_structp png;
    png_infop info;
    png_uint_32 width,height;
    int depth,color;

    // assign png and get information
    if( !(png = png_create_read_struct(PNG_LIBPNG_VER_STRING,NULL,NULL,NULL)) ) {
        fclose(file);
        throw Error("ImageLoader_PNG::load()","Not enough memory to load file:",fname);
    }

    if( !(info = png_create_info_struct(png)) ) {
        png_destroy_read_struct(&png,(png_infopp)NULL,(png_infopp)NULL);
        fclose(file);
        throw Error("ImageLoader_PNG::load()","Not enough memory to load file:",fname);
    }

    if( setjmp(png->jmpbuf) ) {
        png_destroy_read_struct(&png,&info,(png_infopp)NULL);
        fclose(file);
        throw Error("ImageLoader_PNG::load()","libpng failed to load file:",fname);
    }

    // initialize png read
    png_init_io(png,file);
    png_read_info(png,info);

    // setup misc. conversions (from libpng-howto)
    if( info->bit_depth == 16 ) { // avoid 16bits per channel images + make little-endian
        png_set_strip_16(png);
        png_set_swap(png);
    }

	if( (color == PNG_COLOR_TYPE_GRAY) || (color == PNG_COLOR_TYPE_GRAY_ALPHA) ) // gray -> RGB
		png_set_gray_to_rgb(png);

    if( info->bit_depth < 8 )   // avoid packed pixels
        png_set_packing(png);

    if( info->valid & PNG_INFO_tRNS )    // use transparency chunk (add alpha channel)
        png_set_expand(png);

    double img_gamma;
    if( png_get_gAMA(png,info,&img_gamma) )
        png_set_gamma(png,PNG_GAMMA_C,img_gamma);
    else
        png_set_gamma(png,PNG_GAMMA_C,0.45455f);

    // update information
    png_read_update_info(png,info);

    // get geometry
    png_get_IHDR(png,info,&width,&height,&depth,&color,NULL,NULL,NULL);

    // set up SDL_Surface
    int bpp = depth*png_get_channels(png,info);

    SDL_Surface *surface;
    switch( bpp ) {
        case 32:
            surface = SDL_CreateRGBSurface(SDLflags,width,height,32,0x000000FF,0x0000FF00,0x00FF0000,0xFF000000);
            break;
        case 24:
            surface = SDL_CreateRGBSurface(SDLflags,width,height,24,0x000000FF,0x0000FF00,0x00FF0000,0);
            break;
        case 16:
            surface = SDL_CreateRGBSurface(SDLflags,width,height,16,0x001F,0x07E0,0xF800,0);
            break;
        case 8:
        default:
            surface = SDL_CreateRGBSurface(SDLflags,width,height,8,0x0,0x0,0x0,0);
            break;
    }

    // allocate memory for unpacking data
    png_bytepp row_pointers;
    row_pointers = new png_bytep[height];
    if( !row_pointers ) {
        png_destroy_read_struct(&png,&info,(png_infopp)NULL);
        fclose(file);
        throw Error("ImageLoader_PNG::load()","Not enough memory to load file:",fname);
    }

    if( SDL_LockSurface(surface) != 0) {
        throw Error("ImageLoader_PNG::load()","SDL_LockSurface failed:",fname);
    }

    // read image data, using SDL_Surface scanline size for width (optimized surfaces)
    png_byte *img = (png_byte*)surface->pixels;
    int rowbytes = surface->pitch;
    for(unsigned int row=0;row<height;row++)
        row_pointers[row] = img+row*rowbytes;
    png_read_image(png,row_pointers);
    delete []row_pointers;
    SDL_UnlockSurface(surface);

    png_read_end(png,info);

    // load palette data into SDL_Surface
    if( bpp == 8 ) {
        int num_pal;
        png_colorp palette;
        png_get_PLTE(png,info,&palette,&num_pal);

        SDL_Color *colors = new SDL_Color[num_pal];
        SDL_Color *temp = colors;
        for(int i=0;i<num_pal;i++) {
            temp->r = palette[i].red;
            temp->g = palette[i].green;
            temp->b = palette[i].blue;
            temp++;
        }
        SDL_SetColors(surface,colors,0,num_pal);
        delete []colors;
    }

    // free memory used by libpng
    png_destroy_read_struct(&png,&info,(png_infopp)NULL);
    fclose(file);

    return surface;
}

}   // namespace dds