/*
    "sdlbase.cpp"   - SDL/C++ base classes
     
    DDS - Dureks DemoSystem
    Copyright (C)2001 dureks

    This source code is licensed under the GNU GPL.
    See the GNU General Public License for more details.
*/

#include <math.h>
#include <memory.h>
#include "sdlbase.h"
#include "error.h"

namespace dds {

    /*BaseSurface class
       --------------------*/

BaseSurface::BaseSurface()
{
    m_surface = NULL;
}

BaseSurface::~BaseSurface()
{
    SDL_FreeSurface(m_surface);
}

void BaseSurface::copy(BaseSurface *surface)
{
    // a NULL rectangle means everything (see SDL docs)
    SDL_BlitSurface(surface->SDLSurface(),NULL,m_surface,NULL);
    /*
    if( SDL_BlitSurface(surface->SDLSurface(),NULL,m_surface,NULL) == -1 ) {
        throw Error("BaseSurface::copy()","SDL_BlitSurface failed");
    }*/
}

void BaseSurface::copy(BaseSurface *surface,const Area &source,const Area &dest)
{
    SDL_Rect s_rct = source.SDLRect();
    SDL_Rect d_rct = dest.SDLRect();

    if( SDL_BlitSurface(surface->SDLSurface(),&s_rct,m_surface,&d_rct) == -1 ) {
        throw Error("BaseSurface::copy()","SDL_BlitSurface failed (custom area)");
    }
}

void BaseSurface::copy(SDL_Surface *sdlsurface)
{
    // a NULL rectangle means everything (see SDL docs)
    SDL_BlitSurface(sdlsurface,NULL,m_surface,NULL);
    /*
    if( SDL_BlitSurface(sdlsurface,NULL,m_surface,NULL) == -1 ) {
        throw Error("BaseSurface::copy()","SDL_BlitSurface failed");
    }*/
}

// crossfade with source on area
void BaseSurface::xfade(BaseSurface *source,const Area &area,float value)
{
    Uint8 ramp[256];
    Uint8 iramp[256];
    for(int i=0;i<256;i++) {
        ramp[i] = (Uint8)floor(i*value);
        iramp[i] = (Uint8)floor(i*(1.0f-value));
    }

    Uint8 *s = (Uint8*)source->lock();
    Uint8 *t = (Uint8*)lock();
    Uint32 f_x = area.x();
    Uint32 f_y = area.y();
    Uint32 h = area.h();
    Uint32 w = area.w();

    SDL_PixelFormat sformat = format().SDLPixelFormat();
    SDL_PixelFormat tformat = source->format().SDLPixelFormat();
    Uint8 sR,sG,sB,tR,tG,tB;
    // pixel pitch
    Uint8 sPPitch = sformat.BytesPerPixel;
    Uint8 tPPitch = tformat.BytesPerPixel;
    // scanline width
    Uint32 sPitch = source->pitch();
    Uint32 tPitch = pitch();

    s += f_x*sPPitch+f_y*sPitch;
    t += f_x*tPPitch+f_y*tPitch;
    sPitch -= w*sPPitch;
    tPitch -= w*tPPitch;

    for(Uint32 y=0;y<f_y+h;y++) {
        for(Uint32 x=0;x<f_x+w;++x) {
            if( sformat.BytesPerPixel == 2 ) {
                SDL_GetRGB(*(Uint16*)s,&sformat,&sR,&sG,&sB);
                SDL_GetRGB(*(Uint16*)t,&tformat,&tR,&tG,&tB);
            }
            else {
                SDL_GetRGB(*(Uint32*)s,&sformat,&sR,&sG,&sB);
                SDL_GetRGB(*(Uint32*)t,&tformat,&tR,&tG,&tB);
            }

            char8 dR = ramp[tR]+iramp[sR];
            char8 dG = ramp[tG]+iramp[sG];
            char8 dB = ramp[tB]+iramp[sB];

            if( sformat.BytesPerPixel == 2 )
                *(Uint16*)t = (Uint16)SDL_MapRGB(&tformat,dR,dG,dB);
            else
                *(Uint32*)t = (Uint32)SDL_MapRGB(&tformat,dR,dG,dB);

            t += tPPitch;
            s += sPPitch;
        }
        s += sPitch;
        t += tPitch;
   }
    source->unlock();
    unlock();
}

// crossfade with source on area with colorkey
void BaseSurface::xfade(BaseSurface *source,const Area &area, Uint32 x, Uint32 y, float value, Uint32 ckey)
{
    Uint8 ramp[256];
    Uint8 iramp[256];
    for(int i=0;i<256;i++) {
        ramp[i] = (Uint8)floor(i*value);
        iramp[i] = (Uint8)floor(i*(1.0f-value));
    }

    Uint8 *s = (Uint8*)source->lock();
    Uint8 *t = (Uint8*)lock();
    Uint32 f_x = area.x();
    Uint32 f_y = area.y();
    Uint32 h = area.h();
    Uint32 w = area.w();

    SDL_PixelFormat sformat = format().SDLPixelFormat();
    SDL_PixelFormat tformat = source->format().SDLPixelFormat();
    Uint8 sR,sG,sB,tR,tG,tB;
    // pixel pitch
    Uint8 sPPitch = sformat.BytesPerPixel;
    Uint8 tPPitch = tformat.BytesPerPixel;
    // scanline width
    Uint32 sPitch = source->pitch();
    Uint32 tPitch = pitch();
    t += x*tPPitch+y*tPitch;

    s += f_x*sPPitch+f_y*sPitch;
    t += f_x*tPPitch+f_y*tPitch;
    sPitch -= w*sPPitch;
    tPitch -= w*tPPitch;

    for(Uint32 y=0;y<f_y+h;y++) {
        for(Uint32 x=0;x<f_x+w;++x) {
            if( sformat.BytesPerPixel == 2 ) {
                SDL_GetRGB(*(Uint16*)s,&sformat,&sR,&sG,&sB);
                SDL_GetRGB(*(Uint16*)t,&tformat,&tR,&tG,&tB);
            }
            else {
                SDL_GetRGB(*(Uint32*)s,&sformat,&sR,&sG,&sB);
                SDL_GetRGB(*(Uint32*)t,&tformat,&tR,&tG,&tB);
            }

            //check for colorkey
//            fprintf(stderr,"x: %u,y: %u, sR: %u, sG: %u, sB: %u\n",x,y,sR,sG,sB);
            if( (sR != ckey) && (sG != ckey) && (sB != ckey) ) {

                char8 dR = ramp[tR]+iramp[sR];
                char8 dG = ramp[tG]+iramp[sG];
                char8 dB = ramp[tB]+iramp[sB];

                if( sformat.BytesPerPixel == 2 )
                    *(Uint16*)t = (Uint16)SDL_MapRGB(&tformat,dR,dG,dB);
                else
                    *(Uint32*)t = (Uint32)SDL_MapRGB(&tformat,dR,dG,dB);
            }

            t += tPPitch;
            s += sPPitch;
        }
        s += sPitch;
        t += tPitch;
    }
    source->unlock();
    unlock();
}

void BaseSurface::clear()
{
    // clear defaults to black (unless, of course, this is a paletted surface)
    SDL_FillRect(m_surface,NULL,0x00000000);
    /*
    if( SDL_FillRect(m_surface,NULL,0x00000000) == -1 ) {
        throw Error("BaseSurface::clear()","SDL_FillRect failed");
    }*/
}

void BaseSurface::clear(const Color &color)
{
    SDL_FillRect(m_surface,NULL,color.rgba());
    /*
    if( SDL_FillRect(m_surface,NULL,color.rgba()) == -1 ) {
        throw Error("BaseSurface::clear()","SDL_FillRect failed (custom color)");
    }*/
}

void BaseSurface::clear(const Color &color,const Area &dest)
{
    SDL_Rect d_rct = dest.SDLRect();

    SDL_FillRect(m_surface,&d_rct,color.rgba());
    /*
    if( SDL_FillRect(m_surface,&d_rct,color.rgba()) == -1 ) {
        throw Error("BaseSurface::clear()","SDL_FillRect failed (custom color and rectangle)");
    }*/
}

void* BaseSurface::lock()
{
    // ask SDL to lock the surface and return the pixel data
    SDL_LockSurface(m_surface);
    /*
    if( SDL_LockSurface(m_surface) == -1 ) {
        throw Error("BaseSurface::lock()","SDL_LockSurface failed");
    }*/

    return (void*)m_surface->pixels;
}

void BaseSurface::unlock()
{
    SDL_UnlockSurface(m_surface);
}

    /*  Color class
        ----------- */

Color::Color()
{
    m_color.r = 0;
    m_color.g = 0;
    m_color.b = 0;
    m_color.unused = 0;
}

Color::Color(const Color &color)
{
    memcpy(&m_color,&color.m_color,sizeof(SDL_Color));
}

Color::Color(const SDL_Color &color)
{
    memcpy(&m_color,&color,sizeof(SDL_Color));
}

Color::Color(char8 r, char8 g, char8 b, char8 a)
{
    m_color.r = r;
    m_color.g = g;
    m_color.b = b;
    m_color.unused = a;
}

const int32 Color::rgba() const
{
    return (int32)m_color.r
          +(int32)(m_color.g << 8)
          +(int32)(m_color.b << 16)
          +(int32)(m_color.unused << 24);
}

Color& Color::operator() (char8 r, char8 g, char8 b, char8 a)
{
    m_color.r = r;
    m_color.g = g;
    m_color.b = b;
    m_color.unused = a;

    return(*this);
}

Color& Color::operator= (const Color &color)
{
    memcpy(&m_color,&color.m_color,sizeof(SDL_Color));
    return(*this);
}

bool Color::operator== (const Color &color) const
{
    return( (bool)((m_color.r == color.m_color.r) &&
                  (m_color.g == color.m_color.g) &&
                  (m_color.b == color.m_color.b) &&
                  (m_color.unused == color.m_color.unused)) );
}

bool Color::operator!= (const Color &color) const
{
    return( (bool)((m_color.r != color.m_color.r) ||
                  (m_color.g != color.m_color.g) ||
                  (m_color.b != color.m_color.b) ||
                  (m_color.unused != color.m_color.unused)) );
}


    /*  Format class
        ------------ */

Format::Format()
{
    memset(&m_format,0,sizeof(SDL_PixelFormat));
}

Format::Format(const SDL_PixelFormat &format)
{
    memcpy(&m_format,&format,sizeof(SDL_PixelFormat));
    if( depth() == 8 ) {
        m_format.palette = new SDL_Palette;
        m_format.palette->ncolors = format.palette->ncolors;
        memcpy(m_format.palette->colors,format.palette->colors,format.
               palette->ncolors*sizeof(SDL_Color));
    }
}

Format::Format(char8 depth, int32 r, int32 g, int32 b, int32 a)
{
    // disallow overlapping pixel masks (it's _bad_)
    if( (r&g) || (r&b) || (r&a) || (g&b) || (g&a) || (b&a) )
        throw Error("Format::Format()","Overlapping pixel masks detected");

    m_format.BitsPerPixel = depth;
    m_format.BytesPerPixel = (depth+1)/8;
    m_format.Rmask = r;
    m_format.Gmask = g;
    m_format.Bmask = b;
    m_format.Amask = a;
}

void Format::operator() (char8 depth, int32 r, int32 g, int32 b, int32 a)
{
    // disallow overlapping pixel masks (it's _bad_)
    if( (r&g) || (r&b) || (r&a) || (g&b) || (g&a) || (b&a) )
        throw Error("Format::Format()","Overlapping pixel masks detected");

    m_format.BitsPerPixel = depth;
    m_format.BytesPerPixel = (depth+1)/8;
    m_format.Rmask = r;
    m_format.Gmask = g;
    m_format.Bmask = b;
    m_format.Amask = a;
}

Format& Format::operator= (const Format &format)
{
    memcpy(&m_format,&format.m_format,sizeof(SDL_PixelFormat));
    if( depth() == 8 ) {
        m_format.palette = new SDL_Palette;
        m_format.palette->ncolors = format.m_format.palette->ncolors;
        memcpy(m_format.palette->colors,format.m_format.palette->colors,
               format.m_format.palette->ncolors*sizeof(SDL_Color));
    }

    return(*this);
}

bool Format::operator== (const Format &format) const
{
    return( (bool)((m_format.Rmask == format.m_format.Rmask) &&
                   (m_format.Gmask == format.m_format.Gmask) &&
                   (m_format.Bmask == format.m_format.Bmask) &&
                   (m_format.Amask == format.m_format.Amask) &&
                   (m_format.BitsPerPixel == format.m_format.BitsPerPixel)) );
}

bool Format::operator!= (const Format &format) const
{
    return( (bool)((m_format.Rmask != format.m_format.Rmask) ||
                   (m_format.Gmask != format.m_format.Gmask) ||
                   (m_format.Bmask != format.m_format.Bmask) ||
                   (m_format.Amask != format.m_format.Amask) ||
                     (m_format.BitsPerPixel != format.m_format.BitsPerPixel)) );
}


    /*  Area class
        ---------- */

Area::Area()
{
    rect.x = 0;
    rect.y = 0;
    rect.w = 0;
    rect.h = 0;
}

Area::Area(const SDL_Rect &sdlrect)
{
    memcpy(&rect,&sdlrect,sizeof(SDL_Rect));
}

Area::Area(int x, int y, int32 w, int32 h)
{
    rect.x = x;
    rect.y = y;
    rect.w = w;
    rect.h = h;
}

void Area::operator() (int x, int y, int32 w, int32 h)
{
    rect.x = x;
    rect.y = y;
    rect.w = w;
    rect.h = h;
}


}   // namespace dds
