/*
    "sdlbase.h"   - SDL/C++ base classes

    DDS - Dureks DemoSystem
    Copyright (C)2001 dureks

    This source code is licensed under the GNU GPL.
    See the GNU General Public License for more details.
*/

#ifndef _DDS_SDLBASE_H
#define _DDS_SDLBASE_H

#include <SDL/SDL.h>
#include "types.h"

namespace dds {

    /*  Color class  */

class Color {

public:
    // setup
    Color();
    Color(const Color &color);
    Color(const SDL_Color &color);
    Color(char8 r, char8 g, char8 b, char8 a=0);
    ~Color() {}

    // access
    inline const char8 r() const { return(m_color.r); }
    inline const char8 g() const { return(m_color.g); }
    inline const char8 b() const { return(m_color.b); }
    inline const char8 a() const { return(m_color.unused); }
    const int32 rgba() const;
    inline const SDL_Color& SDLColor() const { return(m_color); }

    // operators
    Color& operator() (char8 r, char8 g, char8 b, char8 a=0);
    Color& operator= (const Color &color);
    bool operator== (const Color &color) const;
    bool operator!= (const Color &color) const;

private:
    SDL_Color m_color;
};


    /*  Format class  */

class Format {

public:
    // setup
    Format();
    Format(const SDL_PixelFormat &format);
    Format(char8 depth, int32 r=0, int32 g=0, int32 b=0, int32 a=0);
    ~Format() {};

    // access
    inline const char8 depth() const { return(m_format.BitsPerPixel); }
    inline const char8 pitch() const { return(m_format.BytesPerPixel); }
    inline const int32 rmask() const { return(m_format.Rmask); }
    inline const int32 gmask() const { return(m_format.Gmask); }
    inline const int32 bmask() const { return(m_format.Bmask); }
    inline const int32 amask() const { return(m_format.Amask); }
    inline const SDL_PixelFormat& SDLPixelFormat() const { return(m_format); }

    // operators
    void operator() (char8 depth, int32 r=0, int32 g=0, int32 b=0, int32 a=0);
    Format& operator= (const Format &format);
    bool operator== (const Format &format) const;
    bool operator!= (const Format &format) const;

private:
    SDL_PixelFormat m_format;
};


    /*  Area class  */

class Area {

public:
    // setup
    Area();
    Area(const SDL_Rect &sdlrect);
    Area(int x, int y, int32 w, int32 h);
    ~Area() {}

    void operator() (int x, int y, int32 w, int32 h);

    // FIX: SDL has a bug with the rectangle coordinates (v1.2.0)
    //       ("w" is really "x2")

    // data access
    inline const int x() const { return(rect.x); }
    inline const int y() const { return(rect.y); }
    inline const int32 w() const { return(rect.w-rect.x); }
    inline const int32 h() const { return(rect.h-rect.y); }
    inline const SDL_Rect& SDLRect() const { return(rect); }

private:
    SDL_Rect rect;
};

    /*  BaseSurface class  */

class BaseSurface {

public:
    BaseSurface();
    virtual ~BaseSurface();

    // copy routines
    virtual void copy(BaseSurface *surface);
    virtual void copy(BaseSurface *surface, const Area &source, const Area &dest);
    virtual void copy(SDL_Surface *sdlsurface);
    virtual void xfade(BaseSurface *source, const Area &area, float value = 0.5f);
    virtual void xfade(BaseSurface *source, const Area &area, Uint32 x, Uint32 y, float value, Uint32 ckey);

    // clear routines
    virtual void clear();
    virtual void clear(const Color &color);
    virtual void clear(const Color &color, const Area &dest);

    // direct memory access
    virtual void* lock();
    virtual void unlock();

    // color
    virtual int alpha(int32 alpha, int32 flags=SDL_SRCALPHA|SDL_RLEACCEL)
        { return( SDL_SetAlpha(m_surface,flags,alpha)); }
    virtual int colorkey(int32 key, int32 flags=SDL_SRCCOLORKEY|SDL_RLEACCEL)
        { return( SDL_SetColorKey(m_surface,flags,key)); }

    // data access
    virtual inline const int32 width() const { return((int32)m_surface->w); }
    virtual inline const int32 height() const { return((int32)m_surface->h); }
    virtual inline const int32 pitch() const { return((int32)m_surface->pitch); }
    virtual inline const int32 pixelpitch() const { return((int32)m_surface->format->BytesPerPixel); }
    virtual inline const int32 depth() const { return((int32)m_surface->format->BitsPerPixel); }
    virtual inline Area area() const { return(Area(m_surface->clip_rect)); }
    virtual inline Format format() const { return(Format(*m_surface->format)); }
    virtual inline SDL_Surface* SDLSurface() const { return(m_surface); }
    virtual inline const char* name() const { return(NULL); }

protected:
    SDL_Surface *m_surface;
};

    /*  BaseConsole class  */

class BaseConsole: public BaseSurface {

public:
    // setup
    virtual ~BaseConsole() {};

    // info
    virtual const int32 flags() const = 0;

    // screen management
    virtual void open(const char title[],const int32 width,const int32 height,const int32 depth) = 0;
    virtual void close() = 0;
    virtual void update() = 0;

    // window management
    virtual void caption(const char *title, const char *iconname=NULL) = 0;
    virtual void icon(BaseSurface *icon,char8 *mask=NULL) = 0;
    virtual void cursor(const bool show) = 0;
};


}   // namespace dds

#endif  // _DDS_SDLBASE_H
