/*
    "sprite.cpp"    - Blitter system

    DDS - Dureks DemoSystem
    Copyright (C)2001 dureks

    This source code is licensed under the GNU GPL.
    See the GNU General Public License for more details.
*/

#include <stdlib.h>
#include <stdio.h>
#include <iostream.h>
#include <fstream.h>
#include <string.h>
#include "cpuinfo.h"
#include "sprite.h"
#include "sdlbase.h"
#include "parser.h"
#include "fx_c.h"
#include "image.h"
#include "error.h"

namespace dds {

Blitter::Blitter()
{
    FX_CODE code = C_CODE;
    cpuinfo *cpu = new cpuinfo;

    switch( cpu->architechture() ) {
        case ARCH_INTEL:
            if( cpu->mmx() ) {
                if( !instance ) {
    		        fprintf(stdout,"[Blitter]: MMX code enabled\n");
    		        instance = true;
                }
                code = MMX_CODE;
                break;
            } else {
                code = C_CODE;
                break;
            }

        case ARCH_UNKNOWN:
            code = C_CODE;
            break;
    }

    // set appropriate effects
    switch( code ) {
        case C_CODE:
            throw Error("Blitter::Blitter()","MMX required");

        case MMX_CODE:
            add_pixels_32 = (muuTransparencyPtr) mmx_add_pixels_32;
            add_pixels_grade_32 = (muuTransparencyPtr) mmx_add_pixels_grade_32;
            add_pixels_16 = (muuTransparencyPtr) mmx_add_pixels_16;
            add_pixels_grade_16 = (muuTransparencyPtr) mmx_add_pixels_grade_16;

            sub_pixels_32 = (muuTransparencyPtr) mmx_sub_pixels_32;
            sub_pixels_16 = (muuTransparencyPtr) mmx_sub_pixels_16;

            mask_pixels_32 = (muuTransparencyPtr) c_mask_pixels_32;
            mask_pixels_16 = (muuTransparencyPtr) c_mask_pixels_16;
            break;
    }

    delete cpu;
}

muuTransparencyInterface *Blitter::trans_clip(signed int xpos, signed int ypos,
        int32 dest_width, int32 dest_height, int32 dest_nline,
        int32 sprite_width, int32 sprite_height, int32 sprite_nline,
        char8 pitch, char8 *spr, char8 *dst)
{
    if( xpos > (int)dest_width || ypos > (int)dest_height )
        return(NULL);

    if( xpos < (int)-sprite_width || ypos < (int)-sprite_height )
        return(NULL);

    // pos. on dest
    if (xpos > 0)
        dst += pitch*xpos;

    if (ypos > 0)
        dst += ypos*dest_nline;

    // clip put width
    int putw = sprite_width;
    if(xpos < 0) {
        putw += xpos;
        spr -= pitch*xpos;
    }
    else if( (xpos+putw) > dest_width )
        putw = dest_width-xpos;

    // clip put height
    int puth = sprite_height;
    if(ypos < 0) {
        puth += ypos;
        spr -= ypos*sprite_nline;
    }
    else if( (ypos+puth) > dest_height )
        puth = dest_height-ypos;

    if( putw > dest_width )
        putw = dest_width;
    if( puth > dest_height )
        puth = dest_height;

    muuTransparencyInterface *interface = new muuTransparencyInterface;

    interface->width = putw;
    interface->height = puth;
    interface->source.data = spr;
    interface->source.width = sprite_nline;
    interface->destination.data = dst;
    interface->destination.width = dest_nline;

    return(interface);
}

void Blitter::add(signed int xpos, signed int ypos, BaseSurface *sprite, BaseSurface *dest )
{
    if( sprite->depth() != dest->depth() )
        throw Error("Blitter::add","Sprite and destination depth doesn't match");

    char8 *spr = (char8*)sprite->lock();
    char8 *dst = (char8*)dest->lock();
    int32 depth = dest->depth();

    muuTransparencyInterface *interface = trans_clip(xpos,ypos,
                                          dest->width(),dest->height(),dest->pitch(),
                                          sprite->width(),sprite->height(),sprite->pitch(),
                                          sprite->pixelpitch(),spr,dst);

    if( interface ) {
        if( (interface->width > 0) && (interface->height > 0) )
            switch( depth ) {
                case 32:
                    add_pixels_32(interface);
                    break;
                case 16:
                    add_pixels_16(interface);
                    break;
            }
        delete interface;
    }

    dest->unlock();
    sprite->unlock();
}

void Blitter::addgrade(signed int xpos, signed int ypos, char8 grade, BaseSurface *sprite, BaseSurface *dest )
{
    if( sprite->depth() != dest->depth() )
        throw Error("Blitter::addgrade","Sprite and destination depth doesn't match");

    char8 *spr = (char8*)sprite->lock();
    char8 *dst = (char8*)dest->lock();

    muuTransparencyInterface *interface = trans_clip(xpos,ypos,
                                          dest->width(),dest->height(),dest->pitch(),
                                          sprite->width(),sprite->height(),sprite->pitch(),
                                          sprite->pixelpitch(),spr,dst);

    int32 depth = dest->depth();

    if( interface ) {
        interface->grade = grade;
        if( (interface->width > 0) && (interface->height > 0) )
            switch( depth ) {
                case 32:
                    add_pixels_grade_32(interface);
                    break;
                case 16:
                    add_pixels_grade_16(interface);
                    break;
            }
        delete interface;
    }

    dest->unlock();
    sprite->unlock();
}

void Blitter::sub(signed int xpos, signed int ypos, BaseSurface *sprite, BaseSurface *dest )
{
    if( sprite->depth() != dest->depth() )
        throw Error("Blitter::sub","Sprite and destination depth doesn't match");

    char8 *spr = (char8*)sprite->lock();
    char8 *dst = (char8*)dest->lock();
    int32 depth = dest->depth();

    muuTransparencyInterface *interface = trans_clip(xpos,ypos,
                                          dest->width(),dest->height(),dest->pitch(),
                                          sprite->width(),sprite->height(),sprite->pitch(),
                                          sprite->pixelpitch(),spr,dst);

    if( interface ) {
        if( (interface->width > 0) && (interface->height > 0) )
            switch( depth ) {
                case 32:
                    sub_pixels_32(interface);
                    break;
                case 16:
                    sub_pixels_16(interface);
                    break;
            }
        delete interface;
    }

    dest->unlock();
    sprite->unlock();
}

void Blitter::mask(signed int xpos, signed int ypos, int32 bg, BaseSurface *sprite, BaseSurface *dest )
{
    if( sprite->depth() != dest->depth() )
        throw Error("Blitter::addgrade","Sprite and destination depth doesn't match");

    char8 *spr = (char8*)sprite->lock();
    char8 *dst = (char8*)dest->lock();

    muuTransparencyInterface *interface = trans_clip(xpos,ypos,
                                          dest->width(),dest->height(),dest->pitch(),
                                          sprite->width(),sprite->height(),sprite->pitch(),
                                          sprite->pixelpitch(),spr,dst);

    int32 depth = dest->depth();

    if( interface ) {
        interface->mask = bg;
        if( (interface->width > 0) && (interface->height > 0) )
            switch( depth ) {
                case 32:
                    mask_pixels_32(interface);
                    break;
                case 16:
                    char8 r,g,b;
                    Format format32(32,0x00FF0000,0x0000FF00,0x000000FF);
                    SDL_GetRGB(bg,(SDL_PixelFormat*)&format32.SDLPixelFormat(),&r,&g,&b);
                    interface->mask = SDL_MapRGB((SDL_PixelFormat*)&sprite->format().SDLPixelFormat(),r,g,b);
                    mask_pixels_16(interface);
                    break;
            }
        delete interface;
    }

    dest->unlock();
    sprite->unlock();
}

// static instance variable
bool Blitter::instance;


    /* Sprite class
       ------------*/

Sprite::Sprite(char *script,const Format &format)
{
    ifstream *is = new ifstream(script,ios::in);
    Parser parser(is);

    int pos = 0;
    while( parser.getline() ) {
//        fprintf(stderr,"Text: %s",parser.text());
        if( strstr(parser.text(),"FRAMES") ) {
            sscanf(parser.text(),"FRAMES %u",&frame_count);
            frame = new BaseSurface*[frame_count];
            life_ratio = new float[frame_count];
        }
        else if( strstr(parser.text(),"FRAME") ) {
            char file[256];
            sscanf(parser.text(),"FRAME %s %f",file,&life_ratio[pos]);
            //fprintf(stderr,"Arg: %s",file);
            frame[pos++] = new Image(file,format);
        }
    }
}

Sprite::~Sprite()
{
    // does this work, or do we have to use a for loop
    delete[] frame;
}

void Sprite::draw(Uint32 x, Uint32 y, Uint32 frame_num, BaseSurface *surface)
{
    add(x,y,frame[frame_num],surface);
}


}   // namespace dds