/*
    "vector.h"   - Vector class
     
    DDS - Dureks DemoSystem
    Copyright (C)2001 dureks

    This source code is licensed under the GNU GPL.
    See the GNU General Public License for more details.
*/

#ifndef _DDS_VECTOR_H
#define _DDS_VECTOR_H

#include "types.h"
#include <math.h>

namespace dds {

    /*  Vector2 class  */

class Vector2 {

public:
    // setup
    inline Vector2(const float=0.0f,const float=0.0f);
    inline ~Vector2() {};

    // data access
    inline void operator() (float, float);
    inline void operator= (Vector2);

    // read x,y,z
    inline const float x() const { return v[0]; }
    inline const float y() const { return v[1]; }

    // write x,y,z
    inline void x(const float f) { v[0] = f; }
    inline void y(const float f) { v[1] = f; }

    // compare vectors
    inline const bool operator== (Vector2);
    inline const bool operator!= (Vector2);

    // add,subtract
    inline void operator+= (Vector2);
    inline Vector2 operator+ (Vector2);
    inline void operator-= (Vector2);
    inline Vector2 operator- (Vector2);

    // multiply,divide
    inline void operator*= (float);
    inline Vector2 operator* (float);
    inline void operator/= (float);
    inline Vector2 operator/ (float);

    // vector operations
    inline const float length() const;
    inline const float dot(Vector2) const;
    inline Vector2 normalize();

protected:
    float v[2];    // vector data
};


    /*  Vector3 class  */

class Vector3 {

public:
    // setup
    inline Vector3(const float=0.0f,const float=0.0f,const float=0.0f);
    inline ~Vector3() {};

    // data access
    inline void operator() (float, float, float);
    inline void operator= (Vector3);

    // read x,y,z
    inline const float x() const { return v[0]; }
    inline const float y() const { return v[1]; }
    inline const float z() const { return v[2]; }

    // write x,y,z
    inline void x(const float f) { v[0] = f; }
    inline void y(const float f) { v[1] = f; }
    inline void z(const float f) { v[2] = f; }

    // compare vectors
    inline const bool operator== (Vector3);
    inline const bool operator!= (Vector3);

    // add,subtract
    inline void operator+= (Vector3);
    inline Vector3 operator+ (Vector3);
    inline void operator-= (Vector3);
    inline Vector3 operator- (Vector3);

    // multiply,divide
    inline void operator*= (float);
    inline Vector3 operator* (float);
    inline void operator/= (float);
    inline Vector3 operator/ (float);

    // vector operations
    inline const float length() const;
    inline const float dot(Vector3) const;
    inline Vector3 cross(Vector3) const;
    inline Vector3 normalize();

protected:
    float v[3];    // vector data
};


    /*  Vector2 inline code
        ------------------- */

inline Vector2::Vector2(float x, float y)
{
    v[0] = x;
    v[1] = y;
}

inline void Vector2::operator ()(float ix,float iy)
{
    v[0] = ix;
    v[1] = iy;
}

inline void Vector2::operator =(Vector2 mv)
{
    v[0] = mv.v[0];
    v[1] = mv.v[1];
}

inline const bool Vector2::operator ==(Vector2 mv)
{
    if( (v[0] == mv.v[0]) && (v[1] == mv.v[1]) )
        return(true);
    return(false);
}

inline const bool Vector2::operator !=(Vector2 vec)
{
    if( (v[0] != vec.v[0]) || (v[1] != vec.v[1]) )
        return(true);
    return(false);
}

inline void Vector2::operator +=(Vector2 mv)
{
    v[0] += mv.v[0];
    v[1] += mv.v[1];
}

inline Vector2 Vector2::operator +(Vector2 mv)
{
    Vector2 n = *this;
    n += mv;
    return(n);
}

inline void Vector2::operator -=(Vector2 mv)
{
    v[0] -= mv.v[0];
    v[1] -= mv.v[1];
}

inline Vector2 Vector2::operator -(Vector2 mv)
{
    Vector2 n = *this;
    n -= mv;
    return(n);
}

inline void Vector2::operator *=(float scalar)
{
    v[0] *= scalar;
    v[1] *= scalar;
}

inline Vector2 Vector2::operator *(float scalar)
{
    Vector2 n = *this;
    n *= scalar;
    return(n);
}

inline void Vector2::operator /=(float scalar)
{
    v[0] /= scalar;
    v[1] /= scalar;
}

inline Vector2 Vector2::operator /(float scalar)
{
    Vector2 n = *this;
    n /= scalar;
    return(n);
}

inline const float Vector2::length() const
{
    return((float)sqrt(v[0]*v[0] + v[1]*v[1]));
}

inline const float Vector2::dot(Vector2 mv) const
{
    return(v[0]*mv.v[0] + v[1]*mv.v[1]);
}

inline Vector2 Vector2::normalize()
{
    if( length() )
        *this /= length();
    return(*this);
}


    /*  Vector3 inline code
        ------------------- */

inline Vector3::Vector3(float x, float y, float z)
{
    v[0] = x;
    v[1] = y;
    v[2] = z;
}

inline void Vector3::operator ()(float ix,float iy,float iz)
{
    v[0] = ix;
    v[1] = iy;
    v[2] = iz;
}

inline void Vector3::operator =(Vector3 mv)
{
    v[0] = mv.v[0];
    v[1] = mv.v[1];
    v[2] = mv.v[2];
}

inline const bool Vector3::operator ==(Vector3 mv)
{
    if( (v[0] == mv.v[0]) && (v[1] == mv.v[1]) && (v[2] == mv.v[2]) )
        return(true);
    return(false);
}

inline const bool Vector3::operator !=(Vector3 vec)
{
    if( (v[0] != vec.v[0]) || (v[1] != vec.v[1]) || (v[2] != vec.v[2]) )
        return(true);
    return(false);
}

inline void Vector3::operator +=(Vector3 mv)
{
    v[0] += mv.v[0];
    v[1] += mv.v[1];
    v[2] += mv.v[2];
}

inline Vector3 Vector3::operator +(Vector3 mv)
{
    Vector3 n = *this;
    n += mv;
    return(n);
}

inline void Vector3::operator -=(Vector3 mv)
{
    v[0] -= mv.v[0];
    v[1] -= mv.v[1];
    v[2] -= mv.v[2];
}

inline Vector3 Vector3::operator -(Vector3 mv)
{
    Vector3 n = *this;
    n -= mv;
    return(n);
}

inline void Vector3::operator *=(float scalar)
{
    v[0] *= scalar;
    v[1] *= scalar;
    v[2] *= scalar;
}

inline Vector3 Vector3::operator *(float scalar)
{
    Vector3 n = *this;
    n *= scalar;
    return(n);
}

inline void Vector3::operator /=(float scalar)
{
    v[0] /= scalar;
    v[1] /= scalar;
    v[2] /= scalar;
}

inline Vector3 Vector3::operator /(float scalar)
{
    Vector3 n = *this;
    n /= scalar;
    return(n);
}

inline const float Vector3::length() const
{
    return((float)sqrt(v[0]*v[0] + v[1]*v[1]+ v[2]*v[2]));
}

inline const float Vector3::dot(Vector3 mv) const
{
    return(v[0]*mv.v[0] + v[1]*mv.v[1] + v[2]*mv.v[2]);
}

inline Vector3 Vector3::cross(Vector3 mv) const
{
    Vector3 n;
    n.v[0] = v[1]*mv.v[2] - v[2]*mv.v[1];
    n.v[1] = v[2]*mv.v[0] - v[0]*mv.v[2];
    n.v[2] = v[0]*mv.v[1] - v[1]*mv.v[0];
    return(n);
}

inline Vector3 Vector3::normalize()
{
    if( length() )
        *this /= length();
    return(*this);
}


}   // namespace dds

#endif  // _DDS_VECTOR_H
