/*
    "render.cpp"   - <OpenGL> Renderer
     
    DDS - Dureks DemoSystem
    Copyright (C)2001 dureks

    This source code is licensed under the GNU GPL.
    See the GNU General Public License for more details.
*/

#include <fstream.h>
#include "render.h"
#include "image.h"
#include "error.h"
#include "console.h"

namespace dds {

    /* OpenGL extensions
       ----------------- */
#ifdef GL_ARB_multitexture
    PFNGLACTIVETEXTUREARBPROC glActiveTextureARB_ext = NULL;
    PFNGLMULTITEXCOORD2FARBPROC glMultiTexCoord2fARB_ext = NULL;
    PFNGLMULTITEXCOORD2FVARBPROC glMultiTexCoord2fvARB_ext = NULL;
#endif
    GLint glMultiTexStages = 1; // "no multitexturing"
    GLint glMaxTextureSize = 8; // this should be safe :)


    /*  Renderer class
        -------------- */

Renderer::Renderer(DeviceInfo &info)
{
    m_width = info.width;
    m_height = info.height;
    m_bpp = info.gl_bpp;
    m_zbits = info.gl_zbits;
    m_stencilbits = info.gl_stencilbits;

     // get OpenGL extensions and limits
    glGetIntegerv(GL_MAX_TEXTURE_SIZE,&glMaxTextureSize);
#ifdef GL_ARB_multitexture
    glGetIntegerv((GLenum)GL_MAX_ACTIVE_TEXTURES_ARB, &glMultiTexStages);
    glActiveTextureARB_ext = (PFNGLACTIVETEXTUREARBPROC)SDL_GL_GetProcAddress("glActiveTextureARB");
    glMultiTexCoord2fARB_ext = (PFNGLMULTITEXCOORD2FARBPROC)SDL_GL_GetProcAddress("glMultiTexCoord2fARB");
    glMultiTexCoord2fvARB_ext = (PFNGLMULTITEXCOORD2FVARBPROC)SDL_GL_GetProcAddress("glMultiTexCoord2fvARB");
#endif

    // -----------------------
    // -- renderer defaults --
    // -----------------------

    // matrices
    glMatrixMode(GL_TEXTURE);
    glLoadIdentity();
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();
    // clear
    glClearColor(0.0f,0.0f,0.0f,0.0f);
    glClearDepth(1.0f);
    // depth buffer
    glDepthFunc(GL_LEQUAL);
    glEnable(GL_DEPTH_TEST);
    // culling
    glEnable(GL_CULL_FACE);
    glCullFace(GL_BACK);
    glFrontFace(GL_CW);
    // lighting
    glDisable(GL_LIGHTING);
    glLightModeli(GL_LIGHT_MODEL_TWO_SIDE, GL_FALSE);
    glLightModeli(GL_LIGHT_MODEL_LOCAL_VIEWER, GL_TRUE);
    glColorMaterial(GL_FRONT_AND_BACK, GL_DIFFUSE);
    glEnable(GL_COLOR_MATERIAL);

    glDisable(GL_BLEND);
    glDisable(GL_ALPHA_TEST);
    glDisable(GL_NORMALIZE);
    glDisable(GL_FOG);

    glShadeModel(GL_SMOOTH);
    glHint(GL_PERSPECTIVE_CORRECTION_HINT,GL_NICEST);
    glEnable(GL_TEXTURE_2D);
    glViewport(0,0,m_width,m_height);

    // prepare linked lists
    m_imglist = new LinkedList<BaseSurface>;
    m_texlist = new LinkedList<Texture>;
}

Renderer::~Renderer()
{
    // dispose texture classes
    while( Texture *tex = m_texlist->pop(List::end) )
        delete tex;
    delete m_texlist;

    // ..and the belonging images
    while( BaseSurface *img = m_imglist->pop(List::end) )
        delete img;
    delete m_imglist;
}

Texture* Renderer::loadtexture(const char name[], Texture::textureMip mipmap)
{
    try {
//        Format format(32,0x000000FF,0x0000FF00,0x00FF0000,0xFF000000);
        Format format(24,0x000000FF,0x0000FF00,0x00FF0000,0x00000000);
        BaseSurface *image = new Image(name,format);
        m_imglist->push(image);
        return(createtexture(image,mipmap));
    }
    catch( Error &error ) {
        throw Error("Renderer::loadtexture()",error);
    }
}

Texture* Renderer::createtexture(BaseSurface *image, Texture::textureMip mipmap)
{
    try {
        Texture *texture = new Texture(image,mipmap);
        texture->upload();
        m_texlist->push(texture);
        return(texture);
    }
    catch( Error &error ) {
        throw Error("Renderer::createtexture()",error);
    }
}

void Renderer::texcoord(float u, float v, int32 stage)
{
#ifdef GL_ARB_multitexture
    if( glMultiTexCoord2fARB_ext ) {
        glMultiTexCoord2fARB_ext((GLenum)(GL_TEXTURE0_ARB+stage),u,v);
    }
    else
#endif
    glTexCoord2f(u,v);
}

}   // namespace dds
